#include "Fmm.h"
#include <bitset>
#include <algorithm>
#include <stdio.h>
#include <string.h>
#include <fstream>



Fmm::Fmm(int size) : distanceMatrix(NULL), firstMoveMatrix(NULL)
{
    //FMM contains continuous rows one after the other
    firstMoveMatrix = new unsigned short[size*size];
    //HammingLength is the amount of edges in a complete graph with size nodes
    distMatrixLength = ((size)*(size-1))/2;
    distanceMatrix = new int[distMatrixLength];
    //rowLength is equal to amount of nodes
    rowLength = size;
    nextEmptyRow = 0;
}

Fmm::~Fmm()
{
    delete[] firstMoveMatrix;
    delete[] distanceMatrix;
    //dtor
}

void Fmm::appendRow(const vector<unsigned short> &row) {
//    cout << "appending row:" << endl;
//    for (size_t i=0; i<row.size(); i++) {
//        bitset<16> binShort(row[i]);
//        cout << binShort << endl;
//    }
//    memcpy(&firstMoveMatrix[rowLength*nextEmptyRow++], &row[0], rowLength);
    for (int i=0; i<rowLength; i++) {
        firstMoveMatrix[rowLength*nextEmptyRow+i] = row[i];
    }
    nextEmptyRow++;
//    cout << "row contains: " << endl;
//    for (int i=0; i< rowLength; i++) {
//        bitset<16> binShort(firstMoveMatrix[rowLength*(nextEmptyRow-1)+i]);
//        cout << binShort << endl;
//    }
//    cout << "++++++++++++++++++++++++++++++++++" << endl;
}

void Fmm::printFmm() {
    cout << "First move matrix: " << endl;
    for (int i=0;i<rowLength;i++) {
        for (int j=0;j<rowLength;j++) {
            cout << firstMoveMatrix[j+i*rowLength] << "  ";
        }
        cout << endl;
    }
    cout << "------------------------------" << endl;
}

vector<unsigned short> Fmm::getRow(int index) {
    vector<unsigned short> returnRow(rowLength);
    for (int i=0; i<rowLength; i++) {
        returnRow[i] = firstMoveMatrix[i+index*rowLength];
    }
    return returnRow;
}

vector<unsigned short> Fmm::getRow(int index, const vector<int> &nodeOrder) {
    vector<unsigned short> returnRow(rowLength);
    for (int i=0; i<rowLength; i++) {
        returnRow[i] = firstMoveMatrix[nodeOrder[i]+index*rowLength];
    }
    return returnRow;
}

vector<unsigned short> Fmm::getCol(int index) {
    vector<unsigned short> returnCol(rowLength);
    for (int i=0; i<rowLength; i++) {
        returnCol[i] = firstMoveMatrix[index+i*rowLength];
    }
    return returnCol;
}

vector<unsigned short> Fmm::getCol(int index, const vector<int> &nodeOrder) {
    vector<unsigned short> returnCol(rowLength);
    for (int i=0; i<rowLength; i++) {
        returnRow[i] = firstMoveMatrix[index+nodeOrder[i]*rowLength];
    }
    return returnCol;
}

void Fmm::computeIdentityDistsBetweenColumns() {
    int newRowStartIndex=0;
    for (int mainColumn=0; mainColumn < rowLength-1; mainColumn++) {
        for (int targetColumn=mainColumn+1; targetColumn < rowLength; targetColumn++) {
//            cout << "HL " << hammingLength << " " << newRowStartIndex+targetColumn-1-mainColumn << endl;
            distanceMatrix[newRowStartIndex+targetColumn-1-mainColumn] = columnDiffIdentityBased(mainColumn, targetColumn);
        }
        newRowStartIndex+=(rowLength-mainColumn)-1;
    }
    cout << "Computed Distance Matrix" << endl;
}

void Fmm::computeDisjunctDistsBetweenColumns() {
    int newRowStartIndex=0;
    for (int mainColumn=0; mainColumn < rowLength-1; mainColumn++) {
        for (int targetColumn=mainColumn+1; targetColumn < rowLength; targetColumn++) {
//            cout << "HL " << hammingLength << " " << newRowStartIndex+targetColumn-1-mainColumn << endl;
            distanceMatrix[newRowStartIndex+targetColumn-1-mainColumn] = columnDiffDisjunct(mainColumn, targetColumn);
        }
        newRowStartIndex+=(rowLength-mainColumn)-1;
    }
    cout << "Computed Distance Matrix" << endl;
}

void Fmm::computeXORDistsBetweenColumns() {
    int newRowStartIndex=0;
    for (int mainColumn=0; mainColumn < rowLength-1; mainColumn++) {
        for (int targetColumn=mainColumn+1; targetColumn < rowLength; targetColumn++) {
//            cout << "HL " << hammingLength << " " << newRowStartIndex+targetColumn-1-mainColumn << endl;
            distanceMatrix[newRowStartIndex+targetColumn-1-mainColumn] = columnDiffXOR(mainColumn, targetColumn);
        }
        newRowStartIndex+=(rowLength-mainColumn)-1;
    }
    cout << "Computed Distance Matrix" << endl;
}

// Adds 1 if column entries aren't identical (Hamming distance between columns)
int Fmm::columnDiffIdentityBased(int mainCol, int targCol) {
    int count=0;
    for (int i=0; i < rowLength; i++) {
        if (firstMoveMatrix[mainCol+(i*rowLength)] != firstMoveMatrix[targCol+(i*rowLength)]) {
            count++;
        }
    }
    return count;
}

// Adds 1 only if column entries are disjunct bitwise
int Fmm::columnDiffDisjunct(int mainCol, int targCol) {
    int count=0;
    for (int i=0; i < rowLength; i++) {
        if (  (firstMoveMatrix[mainCol+(i*rowLength)] & firstMoveMatrix[targCol+(i*rowLength)]) == 0 ) {
            count++;
        }
    }
    return count;
}

// Adds 1 for every bit that is set in one column's element but not in the other's corresponding element
int Fmm::columnDiffXOR(int mainCol, int targCol) {
    int count=0;
    for (int i=0; i < rowLength; i++) {
        bitset<16> xoredEntries(firstMoveMatrix[mainCol+(i*rowLength)] ^ firstMoveMatrix[targCol+(i*rowLength)]);
        count += static_cast<int>(xoredEntries.count());
    }
    return count;
}

void Fmm::printHammingMatrix(const char* filename) {
    cout << "-------Hamming Matrix---------" << endl;
    if (filename==NULL) {
        int nextRowBreakAt = rowLength-2;
        int currentRowLength = rowLength-1;
        cout << " ";
        for (int i=0; i<distMatrixLength; i++) {
            cout << distanceMatrix[i]<<",";
            if (i==nextRowBreakAt) {
                nextRowBreakAt+=(--currentRowLength);
                cout << endl << string(rowLength-currentRowLength, ' ');
            }
        }
    }
    cout << "-------End of Matrix----------";
}

void Fmm::writeToConcordeInputFile(const char* filename) {
    ofstream outputFile;
    outputFile.open(filename, ios::trunc);
    outputFile << rowLength << " " << distMatrixLength << endl;
    int cur=0;
    for (int srcNode=0; srcNode<rowLength; srcNode++) {
        for (int trgNode = srcNode+1; trgNode<rowLength; trgNode++) {
            outputFile << srcNode << " " << trgNode << " " << distanceMatrix[cur++] << endl;
        }
    }
    outputFile.close();
}

void Fmm::writeToTSPLIBFile(const string &filename) {
    ofstream outputFile;
    cout << filename.c_str() << endl;
    outputFile.open(filename.c_str(), ios::trunc);
    outputFile << "NAME : hamDists" << endl;
    outputFile << "TYPE : TSP" << endl;
    outputFile << "DIMENSION : " << rowLength << endl;
    outputFile << "EDGE_WEIGHT_TYPE : EXPLICIT" << endl;
    outputFile << "EDGE_WEIGHT_FORMAT : UPPER_ROW" << endl;
    outputFile << "EDGE_WEIGHT_SECTION" << endl;
    for (int cur=0; cur<distMatrixLength; cur++) {
        outputFile << distanceMatrix[cur] << endl;
    }
    outputFile.close();
    cout << "Printed TSPLIB file" << endl;
}
