#include <deque>
#include <vector>
#include <algorithm>
#include <assert.h>
#include "Entry.h"

#include "Fmm.h"
#include "Timer.h"
#include "list_graph.h"
#include "mapper.h"
#include "cpd.h"
#include "order.h"
#include "adj_graph.h"
#include "dijkstra.h"
#include "balanced_min_cut.h"
#include "prefer_zero_cut.h"
#include <cstdio>
#include <bitset>
#include <iostream>
#include <cstring>
#include <fstream>
#include <sstream>
#include <iterator>
#include <time.h>

using namespace std;


#ifdef USE_PARALLELISM
#include <omp.h>
#endif

#define ARRAY_SIZE(array) (sizeof((array))/sizeof((array[0])))

const char *GetName()
{
	#ifndef USE_CUT_ORDER
	return "DFS-SRC-RLE";
	#else
	return "METIS-CUT-SRC-RLE";
	#endif
}

void PreprocessMap(std::vector<bool> &bits, int width, int height, const string &mapName, const string &mainPath, const string &concordePath)
{
	Mapper mapper(bits, width, height);
	printf("width = %d, height = %d, node_count = %d\n", width, height, mapper.node_count());
    NodeOrdering *order = new NodeOrdering(mapper.node_count());

//	printf("Computing node order\n");
//	#ifndef USE_CUT_ORDER
//	NodeOrdering order = compute_real_dfs_order(extract_graph(mapper));
//	#else
//	NodeOrdering order = compute_cut_order(extract_graph(mapper), prefer_zero_cut(balanced_min_cut));
//	#endif
//	mapper.reorder(order);


	printf("Computing first-move matrix\n");

	CPD cpd;
//	{
		AdjGraph g(extract_graph(mapper));

		{
			Dijkstra dij(g);
			Timer t;
			t.StartTimer();
			dij.run(0);
			t.EndTimer();
			printf("Estimated sequential running time : %dmin\n", static_cast<int>(t.GetElapsedTime()*g.node_count()/60.0));
		}

		Dijkstra dij(g);
		Fmm fmMatrix(g.node_count());
		for(int source_node=0; source_node < g.node_count(); ++source_node){
			if(source_node % (g.node_count()/10) == 0)
				printf("%d of %d done\n", source_node, g.node_count());


			const auto&allowed = dij.run(source_node);
			fmMatrix.appendRow(allowed);
		}
//        cout << "Map name: " << mapName << endl;

        const clock_t begin_time = clock();
		fmMatrix.computeXORDistsBetweenColumns();
        cout << "Time taken to calculate Distances: " << float( clock () - begin_time ) /  CLOCKS_PER_SEC << "s" << endl;

//		fmMatrix.printFmm();
        string tspFileName=mainPath;
        tspFileName.append(mapName);

		fmMatrix.writeToTSPLIBFile(tspFileName.replace(tspFileName.size()-3, 3, "tsp"));

		string params=concordePath;
		params.append(" -o ");

		string outputPath=mainPath;
		outputPath.append("Solutions/");
		string tmp=mapName;
		outputPath.append(tmp.replace(tmp.size()-3, 3, "out"));

		params.append(outputPath);

		params.append(" ");
		params.append(tspFileName);



//        char params[1000];
//        strcpy(params, "/home/patrick/BachelorProject/bsc_project_sourcecode/Concorde/concorde_src/TSP/concorde -o ");
//        char concordeOutputFile[] = "/home/patrick/BachelorProject/bsc_project_sourcecode/Maps/map/Solutions/output";
//        strcat(params, concordeOutputFile);
//        strcat(params, " ");
//        char tsplibFile[] = "/home/patrick/BachelorProject/bsc_project_sourcecode/Maps/map/hamDists.tsp";
//        strcat(params, tsplibFile);
        cout << endl << "------------Running Concorde--------------" << endl;
        system(params.c_str());
        cout << endl;

        ifstream tspSolution(outputPath.c_str());
        cout << "TSP output file name is: " << outputPath.c_str() << endl;
        string line;

        if (tspSolution.is_open()) {
            string tempstr;
            getline(tspSolution, tempstr);
            while(getline(tspSolution, tempstr)){
//                cout << tempstr << endl;
                line += tempstr;
            }
//            cout << line << endl;
        } else cout << "Could not open file" << endl;
        tspSolution.close();

        stringstream ss(line);
        istream_iterator<string> begin(ss);
        istream_iterator<string> end;
        vector<string> nodeOrderStringVector(begin, end);
        vector<int> nodeOrderIntVector(nodeOrderStringVector.size());
//        typedef int(*stoi_type)(const string&, size_t*, int);
//
//        transform(nodeOrderStringVector.begin(), nodeOrderStringVector.end(), back_inserter(nodeOrderIntVector),
//                  bind(static_cast<stoi_type>(&stoi),
//                       placeholders::_1, nullptr, mapper.node_count()));
//        cout << nodeOrderStringVector.size() << "   " << nodeOrderIntVector.size() << endl;
        for (size_t i=0;i<nodeOrderStringVector.size();i++) {
//            cout << "converting order element from string to int: " << nodeOrderStringVector[i] << endl;
            nodeOrderIntVector[i] = stoi(nodeOrderStringVector[i]);
        }
//        for (int i=0;i<nodeOrderIntVector.size();i++) {
//            cout << nodeOrderIntVector[i] << " ";
//        }cout << endl;

        cout << "Creating cpd and ordering..." << endl;
        for (int i=0; i<mapper.node_count(); i++) {
            vector<unsigned short> tempRow = fmMatrix.getRow(i, nodeOrderIntVector);
            cpd.append_row(i, tempRow);

            order->map(i, nodeOrderIntVector[i]);
//            cout << "Mapped " << i << " to " << nodeOrderIntVector[i] << endl;
        }
        mapper.reorder(*order);

        cout << "Created cpd and node ordering" << endl;
        string cpdPath=outputPath;
        cpdPath.replace(cpdPath.size()-3, 3, "cpd");
//        const char *outputFilename = "/home/patrick/BachelorProject/bsc_project_sourcecode/Maps/map/Solutions/cpd";
        FILE *f = fopen(cpdPath.c_str(), "wb");
        order->save(f);
        cpd.save(f);
        fclose(f);



//	}
//	filename = "/home/patrick/BachelorProject/bsc_project_sourcecode/Maps/map/output";
//	printf("Saving data to %s\n", filename);

//	FILE*f= fopen(filename, "wb");
//  order.save(f);
//	cpd.save(f);
//	fclose(f);
	printf("Done\n");

}

struct State{
	CPD cpd;
	Mapper mapper;
	AdjGraph graph;
	int current_node;
	int target_node;
};

void *PrepareForSearch(std::vector<bool> &bits, int w, int h, const char *filename)
{
	printf("Loading preprocessing data\n");
	State*state = new State;

	state->mapper = Mapper(bits, w, h);

	FILE*f = fopen(filename, "rb");
	NodeOrdering order;
	order.load(f);
	state->cpd.load(f);
	fclose(f);

	state->mapper.reorder(order);

	state->graph = AdjGraph(extract_graph(state->mapper));
	state->current_node = -1;

	printf("Loading done\n");


	return state;
}

#ifndef EXTRACT_ALL_AT_ONCE
bool GetPath(void *data, xyLoc s, xyLoc t, std::vector<xyLoc> &path)
{
	State*state = static_cast<State*>(data);
	if(path.empty()){
		state->current_node = state->mapper(s);
		state->target_node = state->mapper(t);
		auto first_move = state->cpd.get_first_move(state->current_node, state->target_node);
		if(first_move == 0xF){
			return true;
		} else if(state->current_node == state->target_node) {
			path.push_back(s);
			return true;
		} else {
			path.push_back(s);
			state->current_node = state->graph.out(state->current_node, first_move).target;
			return false;
		}
	} else {
		if(state->current_node != state->target_node){
			path.push_back(state->mapper(state->current_node));
			state->current_node = state->graph.out(state->current_node, state->cpd.get_first_move(state->current_node, state->target_node)).target;
			return false;
		} else {
			path.push_back(t);
			return true;
		}
	}
}
#else
bool GetPath(void *data, xyLoc s, xyLoc t, std::vector<xyLoc> &path)
{
	State*state = static_cast<State*>(data);

	int current_node = state->mapper(s);
	int target_node = state->mapper(t);

	unsigned char first_move = state->cpd.get_first_move(current_node, target_node);
	if(first_move == 0xF)
		return true;
	for(;;){
		path.push_back(state->mapper(current_node));
		current_node = state->graph.out(current_node, first_move).target;
		if(current_node == target_node)
			break;
		first_move = state->cpd.get_first_move(current_node, target_node);

	}
	path.push_back(t);
	return true;
}
#endif

