#! /usr/bin/env python

"""Utilities for performing search after having done translation only"""

import os
import os.path
import platform
import json
import shutil
from collections import namedtuple, defaultdict

Algorithm = namedtuple("Algorithm", "name arguments")
Problem = namedtuple("Problem", "domain_name problem_name, sas_file")


def _create_or_clear_dir(directory):
    if not os.path.exists(directory):
        print("Creating directory: " + directory)
        os.mkdir(directory)
    else:
        print("Cleaning directory: " + directory)
        # credit: https://stackoverflow.com/questions/185936/how-to-delete-the-contents-of-a-folder/185941#185941
        for filename in os.listdir(directory):
            file_path = os.path.join(directory, filename)
            try:
                if os.path.isfile(file_path) or os.path.islink(file_path):
                    os.unlink(file_path)
                elif os.path.isdir(file_path):
                    shutil.rmtree(file_path)
            except Exception as e:
                print('Failed to delete %s. Reason: %s' % (file_path, e))


class TranslationToSearch:
    def __init__(self, current_dir, translation_experiment_name, experiment_name, invar_method):
        data_dir = os.path.join(current_dir, "data")

        self.invar_method = invar_method

        self.translation_experiment_dir = os.path.join(data_dir, translation_experiment_name)
        self.translation_experiment_eval_dir = os.path.join(data_dir, translation_experiment_name + "-eval")

        self.target_translation_dir = os.path.join(current_dir, "data", experiment_name + "-translation")
        self.adapted_translation_dir = os.path.join(self.target_translation_dir, "adapted_properties")

    def get_problems(self):
        """
            returns a list of Problems (namedtuples)
        """
        # prepare search
        # 1. create translated folder in revision
        # ------------------------------------------------------------
        _create_or_clear_dir(self.target_translation_dir)
        translation_files_dir = os.path.join(self.target_translation_dir, "problem_files")
        _create_or_clear_dir(translation_files_dir)

        problems = []
        for dirname in os.listdir(self.translation_experiment_dir):
            if "runs" in dirname:
                run_batch_dir = os.path.join(self.translation_experiment_dir, dirname)
                for translation_run in os.listdir(run_batch_dir):
                    problem = self._get_problem(translation_run, translation_files_dir, run_batch_dir)
                    if problem:
                        problems.append(problem)
        return problems

    def _get_problem(self, translation_run, translation_files_dir, run_batch_dir):
        translation_run_dir = os.path.join(run_batch_dir, translation_run)
        with open(os.path.join(translation_run_dir, "static-properties")) as f:
            run_properties = json.load(f)
            if run_properties["algorithm"] == self.invar_method:
                domain_dir = os.path.join(translation_files_dir, run_properties["domain"])
                if not os.path.exists(domain_dir):
                    os.makedirs(domain_dir)
                problem = run_properties["problem"].replace(".pddl", "")
                translated_problem_file_new_name = os.path.join(domain_dir, problem + ".sas")
                translated_problem_file = os.path.join(translation_run_dir, 'output.sas')
                if os.path.exists(translated_problem_file):
                    shutil.copyfile(translated_problem_file, translated_problem_file_new_name)
                    return Problem(run_properties["domain"], problem, translated_problem_file_new_name)
        return None

    def multiplex_properties_files(self, algorithms):
        # create properties file with multiplexed properties to all planned algorithms and fetch it
        new_properties = {}
        for algo in algorithms:
            with open(os.path.join(self.translation_experiment_eval_dir, "properties")) as f:
                # reload instead of deepcopy (+ no debugging of deepcopy - slower, however acceptable)
                properties = json.load(f)
                for problem_run_name in properties:
                    problem_run = properties[problem_run_name]
                    if problem_run["algorithm"] == self.invar_method:
                        new_problem_run, new_name = self._modify_problem_run(problem_run, problem_run_name, algo.name)
                        new_properties[new_name] = new_problem_run

        _create_or_clear_dir(self.adapted_translation_dir)
        with open(os.path.join(self.adapted_translation_dir, "properties"), "w") as write_file:
            json.dump(new_properties, write_file)

    def _modify_problem_run(self, problem_run, problem_run_name, algo_name):

        problem_run["algorithm"] = algo_name
        problem_run["problem"] = problem_run["problem"].replace(".pddl", "")

        problem_run_id = problem_run["id"]  # list with 3 elements, third containing problem
        problem_run_id[0] = algo_name
        problem_run_id[2] = problem_run_id[2].replace(".pddl", "")

        problem_run["id"] = problem_run_id
        temp = problem_run_name.replace(".pddl", "")
        new_name = problem_run_name.replace(".pddl", "")  # + "_" + algo_name
        temp += " ,,  " + new_name + "(" + self.invar_method + " -> " + algo_name + ")"
        new_name = new_name.replace(self.invar_method, algo_name)
        temp += " ,,  " + new_name
        print(temp)

        return problem_run, new_name


def get_dp_to_runs(properties, key, value=None, dictionary=None):
    if not dictionary:
        dictionary = defaultdict(list)
    for run in properties:
        if key in properties[run]:
            if value:
                if properties[run][key] != value:
                    continue
            dictionary[properties[run]['domain'] + properties[run]['problem']].append(properties[run])
    return dictionary


def filter_translation(translation):
    pd_to_runs = get_dp_to_runs(translation, 'translator_time_finding_invariants')
    t_only_h = dict()
    t_both = dict()
    t_only_r = dict()
    for run in translation:
        dp = translation[run]['domain'] + translation[run]['problem']
        solve_detail = [run['algorithm'] for run in pd_to_runs[dp]]
        if 'helmert' in solve_detail and 'rin08' in solve_detail:
            t_both[run] = translation[run]
            continue
        if 'helmert' in solve_detail:
            t_only_h[run] = translation[run]
            continue
        if 'rin08' in solve_detail:
            t_only_r[run] = translation[run]
            continue

    return t_only_h, t_both, t_only_r


def merge_properties(search, translation):
    result = {}
    for key in search:
        result[key] = search[key]
    for key in translation:
        result["t_" + key] = translation[key]
    return result


def filter_search(t_both, search_h, search_r, heuristics, dir_search, t_base):
    # t_both has only domain+problem combinations which both algorithms managed to translate
    dp_to_search_runs = get_dp_to_runs(search_h, 'coverage', value=1)
    dp_to_search_runs = get_dp_to_runs(search_r, 'coverage', value=1, dictionary=dp_to_search_runs)

    # prepare translation
    dp_to_algos_to_t_run = defaultdict(dict)  # domain+problem to translation property
    for translation_run in t_both:
        dp = t_both[translation_run]['domain'] + t_both[translation_run]['problem'].replace(".pddl", "")
        heuristic = t_both[translation_run]['algorithm']
        assert heuristic not in dp_to_algos_to_t_run[dp] or len(
            dp_to_algos_to_t_run[dp][heuristic]) == 0  # every problem should only be translated once with h and r
        dp_to_algos_to_t_run[dp][heuristic] = t_both[translation_run]
    for dp in dp_to_algos_to_t_run:
        assert len(dp_to_algos_to_t_run[dp]) == 2

    # prepare search
    dph_to_algos = defaultdict(dict)
    for dp in dp_to_search_runs:
        for run in dp_to_search_runs[dp]:
            algo = run["algorithm"].split("_")[0]
            h = run["algorithm"].replace(algo + "_", "")  # necessary for merge_and_shrink
            if h in heuristics:
                dph = dp + h
                dph_to_algos[dph][algo] = run

    heuristic_to_solve_per_algo = defaultdict(dict)
    # produce merge
    for heuristic in heuristics:
        t_both_s_h = dict()
        t_both_s_both = dict()
        t_both_s_r = dict()
        full = dict()
        intersting_time = dict()
        intersting_eval = dict()
        solved_per_algo = heuristic_to_solve_per_algo[heuristic]
        solved_per_algo["both"]=0
        solved_per_algo["helmert"]=0
        solved_per_algo["rin08"]=0
        solved_per_algo["neither"]=0
        for dp in dp_to_search_runs:
            if len(dp_to_algos_to_t_run[dp]) != 2:
                continue  # only look at those that both managed to translate

            algos = dph_to_algos[dp + heuristic]
            # add where suitable (assume that at this point we only have those left that where searched for by both
            t_hel = dp_to_algos_to_t_run[dp]["helmert"]
            t_rin = dp_to_algos_to_t_run[dp]["rin08"]
            if 'helmert' in algos and 'rin08' in algos:
                hel_runs = [run for run in dp_to_search_runs[dp] if
                            "helmert" in run['algorithm'] and heuristic in run['algorithm']][0]
                rin_runs = \
                    [run for run in dp_to_search_runs[dp] if
                     "rin08" in run['algorithm'] and heuristic in run['algorithm']][
                        0]
                t_both_s_both['helmert-' + dp] = merge_properties(hel_runs, t_hel)
                t_both_s_both['rin08-' + dp] = merge_properties(rin_runs, t_rin)
                full['helmert-' + dp] = merge_properties(hel_runs, t_hel)
                full['rin08-' + dp] = merge_properties(rin_runs, t_rin)
                solved_per_algo["both"]+=1
                # filter out interesting:
                if "ipdb" in heuristic or "merge" in heuristic:
                    rin = algos["rin08"]
                    hel = algos["helmert"]
                    scalled_time_dif= (rin['search_time']-hel['search_time'])/rin['search_time']
                    scalled_evaluations_until_last_jump_dif= abs(rin['evaluations_until_last_jump']-hel['evaluations_until_last_jump'])/(rin['evaluations_until_last_jump']+hel['evaluations_until_last_jump'])
                    if abs(scalled_time_dif)>0.15:
                        hel["time_|r-h|)/(r+h)"] = scalled_time_dif
                        rin["time_|r-h|)/(r+h)"] = scalled_time_dif
                        intersting_time['helmert-' + dp] = merge_properties(hel, t_hel)
                        intersting_time['rin08-' + dp] = merge_properties(rin, t_rin)
                    if abs(scalled_evaluations_until_last_jump_dif) > 0.15:
                        hel["eval_|r-h|)/(r+h)"] = scalled_evaluations_until_last_jump_dif
                        rin["eval_|r-h|)/(r+h)"] = scalled_evaluations_until_last_jump_dif
                        intersting_eval['helmert-' + dp] = merge_properties(hel, t_hel)
                        intersting_eval['rin08-' + dp] = merge_properties(rin, t_rin)
                continue
            if 'helmert' in algos:
                hel_runs = [run for run in dp_to_search_runs[dp] if
                            "helmert" in run['algorithm'] and heuristic in run['algorithm']][0]
                t_both_s_h[dp] = merge_properties(hel_runs, t_hel)
                full[dp] = merge_properties(hel_runs, t_hel)
                solved_per_algo["helmert"]+=1
                continue
            if 'rin08' in algos:
                rin_runs = \
                    [run for run in dp_to_search_runs[dp] if
                     "rin08" in run['algorithm'] and heuristic in run['algorithm']][
                        0]
                t_both_s_r[dp] = merge_properties(rin_runs, t_rin)
                full[dp] = merge_properties(rin_runs, t_rin)
                solved_per_algo["rin08"]+=1
                continue
            solved_per_algo["neither"]+=1

        s_base = t_base + 'both-s_' + heuristic + "-"
        save_if_inexistent(dir_search, s_base + 'h', t_both_s_h)
        save_if_inexistent(dir_search, s_base + 'both', t_both_s_both)
        save_if_inexistent(dir_search, s_base + 'r', t_both_s_r)
        save_if_inexistent(dir_search, s_base + 'full', full)
        save_if_inexistent(dir_search, s_base + 'intersting_time', intersting_time)
        save_if_inexistent(dir_search, s_base + 'intersting_eval', intersting_eval)
    print(heuristic_to_solve_per_algo)
    return


def save_if_inexistent(dir, name, json_data):
    filepath = os.path.join(dir, name)
    if not os.path.isfile(filepath):
        json.dump(json_data, open(filepath, 'w'))


def new_split():
    # configuration zone start
    key_sat = 'sat'
    key_opt = 'opt'
    key_partial_opt = 'partial_opt'
    track_to_timeframes = {key_opt: ['3h'], key_sat: ['3h'],
                           key_partial_opt: ['8h']}  # , '5m', '3h', '24h' # TODO run partial_opt on 5min
    _opt_heuristics = ['blind', 'lmcut', 'hmax', 'ipdb', 'merge_and_shrink']
    _sat_heuristics = ['blind', 'ff']  # TODO would need to run with other than A*
    track_to_heuristics = {key_opt: _opt_heuristics, key_sat: _opt_heuristics,
                           key_partial_opt: _opt_heuristics}
    # paths
    ROOT = os.path.join(os.path.dirname(os.path.abspath(__file__)), "data", "do_evaluation")
    # configuration zone end

    for track in {key_sat, key_opt, key_partial_opt}:
        print(track)
        dir_track = os.path.join(ROOT, track)
        if not os.path.exists(dir_track):
            os.mkdir(dir_track)
        for timeframe in track_to_timeframes[track]:
            base = "properties-" + track + "-" + timeframe
            translation = json.load(open(os.path.join(ROOT, base + "-translation")))
            search_h = json.load(open(os.path.join(ROOT, base + "-search-helmert")))
            search_r = json.load(open(os.path.join(ROOT, base + "-search-rin08")))

            # --------------------filter translation and saving--------------------
            dir_timeframe = os.path.join(dir_track, timeframe)

            if not os.path.exists(dir_timeframe):
                os.mkdir(dir_timeframe)
            t_only_h, t_both, t_only_r = filter_translation(translation)

            t_base = base + '-t_'
            save_if_inexistent(dir_timeframe, t_base + 'h', t_only_h)
            save_if_inexistent(dir_timeframe, t_base + 'both', t_both)
            save_if_inexistent(dir_timeframe, t_base + 'r', t_only_r)
            del t_only_r
            del t_only_h
            del translation

            # --------------------filter search and saving--------------------
            dir_search = os.path.join(dir_timeframe, "search")
            if not os.path.exists(dir_search):
                os.mkdir(dir_search)
            filter_search(t_both, search_h, search_r, track_to_heuristics[track], dir_search, t_base)


if __name__ == "__main__":
    new_split()
