#! /usr/bin/env python

"""
Solve STRIPS tasks with A* using several heuristics.
The translation was done with Helmerts algorithm from 2009
"""

import os
import os.path
import platform
import json
import shutil

from downward.experiment import FastDownwardExperiment
from downward.reports.absolute import AbsoluteReport
from downward.reports.scatter import ScatterPlotReport
from lab import cached_revision
from lab.environments import BaselSlurmEnvironment, LocalEnvironment
from util_LAB_search import TranslationToSearch, Algorithm

# Modify these variables according to the experiment and invariant method targeted by this script
INVAR_METHOD = "helmert"

NODE = platform.node()
if NODE.endswith(".scicore.unibas.ch") or NODE.endswith(".cluster.bc2.ch"):
    TRANSLATION_EXPERIMENT_NAME = "strips-20_9_15-translation_3h-all_sat_domains"
    ENV = BaselSlurmEnvironment(email="severin.wyss@stud.unibas.ch")
    REPO = "MT/code/swmt_FD"
else:
    TRANSLATION_EXPERIMENT_NAME = "strips-20_9_15-translation_3h-all_sat_domains"
    ENV = LocalEnvironment(processes=4)
    REPO = "MT1/Code"

CURRENT_DIR = os.path.dirname(os.path.abspath(__file__))
EXP_NAME = os.path.splitext(os.path.basename(__file__))[0]
translation_to_search = TranslationToSearch(CURRENT_DIR, TRANSLATION_EXPERIMENT_NAME, EXP_NAME, INVAR_METHOD)

ATTRIBUTES = ["coverage", "cost", "evaluations", "search_time", "total_time", "initial_h_value",
              "error"]

merge_and_shrink = "astar(" \
                   "merge_and_shrink(" \
                   "shrink_strategy=shrink_bisimulation(greedy=false)," \
                   "merge_strategy=merge_sccs(order_of_sccs=topological," \
                        "merge_selector=score_based_filtering(scoring_functions=[goal_relevance,dfp,total_order]))," \
                        "label_reduction=exact(before_shrinking=true,before_merging=false)," \
                        "max_states=50k,threshold_before_merge=1))"

algorithms = [Algorithm(INVAR_METHOD + "_blind", ["--search", "astar(blind())"]),
              Algorithm(INVAR_METHOD + "_hmax", ["--search", "astar(hmax())"]),
              Algorithm(INVAR_METHOD + "_ipdb", ["--search", "astar(ipdb())"]),
              Algorithm(INVAR_METHOD + "_lmcut", ["--search", "astar(lmcut())"]),
              Algorithm(INVAR_METHOD + "_merge_and_shrink", ["--search", merge_and_shrink])
              ]

# If REVISION_CACHE is None, the default ./data/revision-cache is used.
REVISION_CACHE = os.environ.get("DOWNWARD_REVISION_CACHE")
REV = "sw_mt_invarsynt_main"

# Prepare properties files and problem.sas files
problems = translation_to_search.get_problems()

# experiment
exp = FastDownwardExperiment(environment=ENV, revision_cache=REVISION_CACHE)

# Add built-in parsers to the experiment.
exp.add_parser(exp.EXITCODE_PARSER)
exp.add_parser(exp.TRANSLATOR_PARSER)
exp.add_parser(exp.SINGLE_SEARCH_PARSER)
exp.add_parser(exp.PLANNER_PARSER)

for problem in problems:
    exp.add_task(problem.domain_name, problem.problem_name, problem.sas_file)

for algo in algorithms:
    exp.add_algorithm(algo.name, REPO, REV, algo.arguments)

# Add step that writes experiment files to disk.
exp.add_step("build", exp.build)

# Add step that executes all runs.
exp.add_step("start", exp.start_runs)

# Add step that collects properties from run directories and
# writes them to *-eval/properties.
exp.add_fetcher(name="fetch")
# translation_to_search.multiplex_properties_files(algorithms)
# exp.add_fetcher(translation_to_search.adapted_translation_dir, name="fetch prepared translation", merge=True)

# Add report step (AbsoluteReport is the standard report).
exp.add_report(AbsoluteReport(attributes=ATTRIBUTES), outfile="report.html")

# Parse the commandline and show or run experiment steps.
exp.run_steps()
