#! /usr/bin/env python

"""Translate a gripper (STRIPS) tasks using both Helmerts Algorithm from 2009 and Rintans Algorithm from 2008. Timelimit set to 8h"""

import os
import os.path
import platform

from downward.experiment import FastDownwardExperiment
from downward.reports.absolute import AbsoluteReport
from downward.reports.scatter import ScatterPlotReport
from lab import cached_revision
from lab.environments import BaselSlurmEnvironment, LocalEnvironment
from translator_invariants_parser import TranslatorInvariantParser


ATTRIBUTES = ["coverage", "error", "translator_invariant_number_of_actions", "translator_invariant_number_of_candidates_considered", "translator_invariant_number_of_facts", "translator_invariant_number_of_found_invariants", "translator_invariant_number_of_found_mutexes", "translator_invariant_number_of_grounded_actions", "translator_invariant_number_of_objects", "translator_invariant_number_of_predicates", "translator_invariant_number_of_test_of_candidates", "translator_time_finding_invariants", "translator_operators", "translator_variables", "translator_binary_variables", "translator_facts", "translator_invariant_number_of_iterations"]

NODE = platform.node()
if NODE.endswith(".scicore.unibas.ch") or NODE.endswith(".cluster.bc2.ch"):
    # Create bigger suites with suites.py from the downward-benchmarks repo.
    SUITE = ["depot", "gripper",  "blocks", "driverlog", "freecell", "grid", "logistics00", "logistics98", "miconic", "mprime", "mystery", "pipesworld-notankage",  "satellite", "zenotravel"]
    # excluded: movie (already fully done 8_28)
    ENV = BaselSlurmEnvironment(email="severin.wyss@stud.unibas.ch")
    REPO = os.environ["DOWNWARD_REPO"]
    BENCHMARKS_DIR = os.environ["DOWNWARD_BENCHMARKS"]
else:
    SUITE = ["gripper:prob01.pddl","gripper:prob02.pddl"]
    ENV = LocalEnvironment(processes=2)
    REPO = "MT1/Code"
    BENCHMARKS_DIR = "MT1/benchmark"

# If REVISION_CACHE is None, the default ./data/revision-cache is used.
REVISION_CACHE = os.environ.get("DOWNWARD_REVISION_CACHE")
REV = "sw_mt_invarsynt_main_evalrin08_20200827"

exp = FastDownwardExperiment(environment=ENV, revision_cache=REVISION_CACHE)

# Add built-in parsers to the experiment.
exp.add_parser(exp.EXITCODE_PARSER)
exp.add_parser(exp.TRANSLATOR_PARSER)
exp.add_parser(exp.SINGLE_SEARCH_PARSER)
exp.add_parser(exp.PLANNER_PARSER)
exp.add_parser(os.path.join(os.path.dirname(os.path.abspath(__file__)), "translator_invariants_parser.py"))

exp.add_suite(BENCHMARKS_DIR, SUITE)

time_string="24h"
time_number = "86400"

exp.add_algorithm("rin08", REPO, REV, ["--translate-options", "--invariant-generation-max-time", time_number, "--invariant_method","rintanen08", "--rintanen_max_invarsize", "2","--invariant_rin08_opt_sameoperatortesting"],[],["--translate","--overall-time-limit", time_string, "--translate-time-limit", time_string])

exp.add_algorithm("helmert", REPO, REV, ["--translate-options", "--invariant-generation-max-time", time_number, "--invariant_method","helmert"],[],["--translate","--overall-time-limit", time_string, "--translate-time-limit", time_string])

# Add step that writes experiment files to disk.
exp.add_step("build", exp.build)

# Add step that executes all runs.
exp.add_step("start", exp.start_runs)

exp.add_parse_again_step()

# Add step that collects properties from run directories and
# writes them to *-eval/properties.
exp.add_fetcher(name="fetch")

# Add report step (AbsoluteReport is the standard report).
exp.add_report(AbsoluteReport(attributes=ATTRIBUTES), outfile="report.html")

exp.add_report(
    AbsoluteReport(attributes=ATTRIBUTES, format="tex"),
    outfile="report-abs-combined.tex",
)

# Parse the commandline and show or run experiment steps.
exp.run_steps()
