#include "default_value_axioms_task.h"

#include "../task_proxy.h"

#include "../algorithms/sccs.h"
#include "../task_utils/task_properties.h"

#include "../utils/logging.h"

#include <deque>
#include <iostream>
#include <memory>
#include <set>

using namespace std;
using utils::ExitCode;

namespace tasks {
DefaultValueAxiomsTask::DefaultValueAxiomsTask(
    const shared_ptr<AbstractTask> &parent, AxiomHandlingType axioms, vector<UnrollingOptionType> unrolling_options)
    : DelegatingTask(parent),
      axioms(axioms),
      default_value_axioms_start_index(parent->get_num_axioms()),
      unrolling_vars_start_index(parent->get_num_variables()) {
    TaskProxy task_proxy(*parent);
    utils::g_log << "Total axioms before task transformation " << task_proxy.get_axioms().size() << endl;
    

    if(axioms == AxiomHandlingType::EXACT_NEGATIVE_CYCLES) {
        unordered_map<int, int> var_mapping;
        unordered_map<int, int> prev_mapping;
        unordered_map<int, int> curr_mapping;
        axioms_used_for_unrolling.assign(get_num_axioms(), false);
        variables_used_for_unrolling.assign(get_num_variables(), false);
        cycle_independent_axioms.assign(get_num_axioms(), false);
        // We only need these variables for the preprocessing step
        auto [pre_nondefault_dependencies, pre_default_dependencies] = create_nondefault_and_default_dependencies_for_all_axioms();
        std::vector<std::vector<int>> pre_axiom_ids_for_var = create_axiom_ids_for_all_vars();
        vector<vector<int>> pre_sccs;
        std::vector<std::vector<int> *> pre_var_to_scc = compute_var_to_scc_from_nondefault_dependencies(pre_nondefault_dependencies, pre_sccs);
        int vars_before_unrolling = get_num_variables();
        for (int i = 0; i < vars_before_unrolling; ++i) {
            /*  
                If the SCC a variable is in has more than one variable, 
                there is a cyclic dependency between those derived variables.
                If there is a cyclic dependency between several derived
                variables, the "obvious" way of negating the formula
                defining the derived variable is semantically wrong
                (see issue453).
                We need to unroll the variables first before we can
                process them further.
            */
            if(pre_var_to_scc[i]->size() > 1 && !variables_used_for_unrolling[i]) {
                if (pre_var_to_scc[i]->size() > 10 && std::find(unrolling_options.begin(), unrolling_options.end(), UnrollingOptionType::ONLY_SMALL_CYCLES) != unrolling_options.end()) {
                    continue;
                }
                unroll_negative_cycles(
                    i, pre_var_to_scc, pre_axiom_ids_for_var, var_mapping, prev_mapping, curr_mapping, unrolling_options);
            }
        }
    }
    // After the pre-processing for "EXACT_NEGATIVE_CYCLES" we can create the default value axioms normally
    auto [nondefault_dependencies, default_dependencies] = create_nondefault_and_default_dependencies_for_all_axioms();
    std::vector<std::vector<int>> axiom_ids_for_var = create_axiom_ids_for_all_vars();
    vector<vector<int>> sccs;
    std::vector<std::vector<int> *> var_to_scc = compute_var_to_scc_from_nondefault_dependencies(nondefault_dependencies, sccs);

    unordered_set<int> default_needed = get_vars_with_relevant_default_value(
    nondefault_dependencies, default_dependencies, var_to_scc);
    for (int var : default_needed) {
        int default_value = get_variable_default_axiom_value(var);
        if (axioms == AxiomHandlingType::APPROXIMATE_NEGATIVE || 
            (axioms == AxiomHandlingType::APPROXIMATE_NEGATIVE_CYCLES &&
                var_to_scc[var]->size() > 1)) {
             /*
                We perform a naive overapproximation
                which assumes that derived variables occurring
                in the cycle can be false unconditionally. This is good
                enough for correctness of the code that uses these
                default value axioms, but loses accuracy (see issue453).
            */
            default_value_axioms.emplace_back(
                FactPair(var, default_value), vector<FactPair>());
        }
        else {
            add_default_value_axioms_for_var(
                FactPair(var, get_variable_default_axiom_value(var)),
                axiom_ids_for_var);
        }
    }
    

    if (axioms == AxiomHandlingType::EXACT_NEGATIVE_CYCLES) {
        utils::g_log << "Axioms created with unrolling: " << unrolling_axioms_counter << endl;
        utils::g_log << "Percentage of unrolling axioms: " << (float)unrolling_axioms_counter / (task_proxy.get_axioms().size() + default_value_axioms.size()) << endl;
        utils::g_log << "Variables created with unrolling: " << unrolling_variables.size() << endl;
    }
    utils::g_log << "Default Value Axioms created: " << default_value_axioms.size() - unrolling_axioms_counter<< endl;
    utils::g_log << "Percentage of default value axioms: " << (float)(default_value_axioms.size() - unrolling_axioms_counter) / (task_proxy.get_axioms().size() + default_value_axioms.size()) << endl;
}

/*
  Collect for which derived variables it is relevant to know how they
  can obtain their default value. This is done by tracking for all
  derived variables which of their values are needed.

  Initially, we know that var=val is needed if it appears in a goal or
  operator condition. Then we iteratively do the following:
  a) If var=val is needed and var'=nondefault is in the body of an
     axiom setting var=nondefault, then var'=val is needed.
  b) If var=val is needed and var'=default is in the body of an axiom
     setting var=nondefault, then var'=!val is needed, where
       - !val=nondefault if val=default
       - !val=default if val=nondefault
  (var and var' are always derived variables.)

  If var=default is needed but we already know that the axioms we will
  introduce for var=default are going to have an empty body, then we don't
  apply a)/b) (because the axiom for var=default will not depend on anything).
*/
unordered_set<int> DefaultValueAxiomsTask::get_vars_with_relevant_default_value(
    const vector<vector<int>> &nondefault_dependencies,
    const vector<vector<int>> &default_dependencies,
    const vector<vector<int> *> &var_to_scc) {
    // Store which derived vars are needed default (true) / nondefault(false).
    utils::HashSet<pair<int, bool>> needed;

    TaskProxy task_proxy(*parent);

    // Collect derived variables that occur as their default value.
    for (const FactProxy &goal : task_proxy.get_goals()) {
        VariableProxy var_proxy = goal.get_variable();
        if (var_proxy.is_derived()) {
            bool default_value =
                goal.get_value() == var_proxy.get_default_axiom_value();
            needed.emplace(goal.get_pair().var, default_value);
        }
    }
    for (OperatorProxy op : task_proxy.get_operators()) {
        for (FactProxy condition : op.get_preconditions()) {
            VariableProxy var_proxy = condition.get_variable();
            if (var_proxy.is_derived()) {
                bool default_value = condition.get_value() ==
                                     var_proxy.get_default_axiom_value();
                needed.emplace(condition.get_pair().var, default_value);
            }
        }
        for (EffectProxy effect : op.get_effects()) {
            for (FactProxy condition : effect.get_conditions()) {
                VariableProxy var_proxy = condition.get_variable();
                if (var_proxy.is_derived()) {
                    bool default_value = condition.get_value() ==
                                         var_proxy.get_default_axiom_value();
                    needed.emplace(condition.get_pair().var, default_value);
                }
            }
        }
    }

    deque<pair<int, bool>> to_process(needed.begin(), needed.end());
    while (!to_process.empty()) {
        int var = to_process.front().first;
        bool default_value = to_process.front().second;
        to_process.pop_front();

        /*
          If we process a default value and already know that the axiom we
          will introduce has an empty body (either because we trivially
          overapproximate everything or because the variable has cyclic
          dependencies), then the axiom (and thus the current variable/value
          pair) doesn't depend on anything.
        */
        if ((default_value) &&
            (axioms == AxiomHandlingType::APPROXIMATE_NEGATIVE ||
             var_to_scc[var]->size() > 1)) {
            continue;
        }

        for (int nondefault_dep : nondefault_dependencies[var]) {
            auto insert_retval = needed.emplace(nondefault_dep, default_value);
            if (insert_retval.second) {
                to_process.emplace_back(nondefault_dep, default_value);
            }
        }
        for (int default_dep : default_dependencies[var]) {
            auto insert_retval = needed.emplace(default_dep, !default_value);
            if (insert_retval.second) {
                to_process.emplace_back(default_dep, !default_value);
            }
        }
    }

    unordered_set<int> default_needed;
    for (pair<int, bool> entry : needed) {
        if (entry.second) {
            default_needed.insert(entry.first);
        }
    }
    return default_needed;
}

void DefaultValueAxiomsTask::add_default_value_axioms_for_var(
    FactPair head, const vector<vector<int>> &axiom_ids_for_var) {
    const vector<int> &axiom_ids = axiom_ids_for_var[head.var];

    /*
      If no axioms change the variable to its non-default value,
      then the default is always true.
    */
    if (axiom_ids.empty()) {
        default_value_axioms.emplace_back(head, vector<FactPair>());
        return;
    }

    vector<set<FactPair>> conditions_as_cnf; 
    conditions_as_cnf.reserve(axiom_ids.size());
    for (int axiom_id : axiom_ids) {
        conditions_as_cnf.emplace_back();
        for (int j = 0; j < get_num_operator_effect_conditions(axiom_id, 0, true); ++j) {
            FactPair cond = get_operator_effect_condition(axiom_id, 0, j, true);
            int cond_var = cond.var;
            int num_vals = get_variable_domain_size(cond_var);
            for (int value = 0; value < num_vals; ++value) {
                if (value != cond.value) {
                    conditions_as_cnf.back().insert({cond_var, value}); 
                }
            }
        }
    }

    // We can see multiplying out the cnf as collecting all hitting sets.
    set<FactPair> current;
    unordered_set<int> current_vars;
    set<set<FactPair>> hitting_sets;
    collect_non_dominated_hitting_sets_recursively(
        conditions_as_cnf, 0, current, current_vars, hitting_sets);

    for (const set<FactPair> &c : hitting_sets) {
        default_value_axioms.emplace_back(
            head, vector<FactPair>(c.begin(), c.end()));
    }
}

void DefaultValueAxiomsTask::collect_non_dominated_hitting_sets_recursively(
    const vector<set<FactPair>> &set_of_sets, size_t index,
    set<FactPair> &hitting_set, unordered_set<int> &hitting_set_vars,
    set<set<FactPair>> &results) {
    if (index == set_of_sets.size()) {
        /*
           Check whether the hitting set is dominated.
           If we find a fact f in hitting_set such that no set in the
           set of sets is covered by *only* f, then hitting_set \ {f}
           is still a hitting set that dominates hitting_set.
        */
        set<FactPair> not_uniquely_used(hitting_set);
        for (const set<FactPair> &set : set_of_sets) {
            vector<FactPair> intersection;
            set_intersection(
                set.begin(), set.end(), hitting_set.begin(), hitting_set.end(),
                back_inserter(intersection));
            if (intersection.size() == 1) {
                not_uniquely_used.erase(intersection[0]);
            }
        }
        if (not_uniquely_used.empty()) {
            results.insert(hitting_set);
        }
        return;
    }

    const set<FactPair> &set = set_of_sets[index];
    for (const FactPair &elem : set) {
        /*
          If the current set is covered with the current hitting set
          elements, we continue with the next set.
        */
        if (hitting_set.find(elem) != hitting_set.end()) {
            collect_non_dominated_hitting_sets_recursively(
                set_of_sets, index + 1, hitting_set, hitting_set_vars, results);
            return;
        }
    }

    for (const FactPair &elem : set) {
        // We don't allow choosing different facts from the same variable.
        if (hitting_set_vars.find(elem.var) != hitting_set_vars.end()) {
            continue;
        }

        hitting_set.insert(elem);
        hitting_set_vars.insert(elem.var);
        collect_non_dominated_hitting_sets_recursively(
            set_of_sets, index + 1, hitting_set, hitting_set_vars, results);
        hitting_set.erase(elem);
        hitting_set_vars.erase(elem.var);
    }
}

void DefaultValueAxiomsTask::unroll_negative_cycles(
    int var,
    const vector<vector<int> *> &var_to_scc,
    const vector<vector<int>> &axiom_ids_for_var,
    unordered_map<int, int> &var_mapping,
    unordered_map<int, int> &prev_mapping,
    unordered_map<int, int> &curr_mapping,
    vector<UnrollingOptionType> &unrolling_options) {
    // The maximum number of timestamps needed to keep the same semantics is equal the number of variables in the SCC
    int timestamps = var_to_scc[var]->size();
    bool prune_unreachable = 
        (std::find(unrolling_options.begin(), unrolling_options.end(), UnrollingOptionType::PRUNE_UNREACHABLE) != unrolling_options.end());
    bool replace_propagation_axioms = 
        (std::find(unrolling_options.begin(), unrolling_options.end(), UnrollingOptionType::REPLACE_PROPAGATION_AXIOMS) != unrolling_options.end());

    if (!prune_unreachable) {
        create_unrolling_variable_mapping_and_initialize_unrolling_variables(*var_to_scc[var], var_mapping);
    }
    vector<FactPair> new_conditions;
    FactPair cond(0,0);
    FactPair new_head(0,0);

    // Create cycle-independent axioms first (axioms that do not depend on any variable in the current SCC)
    for (int v : *var_to_scc[var]) {
        for (int a : axiom_ids_for_var[v]) {
            int new_conditions_size = get_num_operator_effect_conditions(a, 0, true);
            new_conditions.clear();
            new_conditions.reserve(new_conditions_size);
            for (int c = 0; c < new_conditions_size; ++c) {
                cond = get_operator_effect_condition(a, 0, c, true);
                if (var_to_scc[cond.var] != var_to_scc[var]) {
                    // Not in the current SCC, keep condition as is
                    new_conditions.emplace_back(cond);
                }
                else {
                    // In same SCC, continue with the next axiom
                    break;
                }
            }
            // Variable not cycle independent, can be skipped
            if ((int)new_conditions.size() < new_conditions_size) {
                continue;
            }
            int new_head_var;
            if (prune_unreachable) {
                new_head_var = initialize_new_unrolling_var(v, 0, timestamps);
                curr_mapping[v] = new_head_var;
            }
            else {
                new_head_var = var_mapping.at(v);
            }
            new_head = FactPair(
                new_head_var,
                get_operator_effect(a, 0, true).value);
            default_value_axioms.emplace_back(
                new_head, vector<FactPair>(new_conditions.begin(), new_conditions.end()));
            ++unrolling_axioms_counter;
            cycle_independent_axioms[a] = true;
        }
    }
    // Next, create cycle-dependent axioms (axioms that depend on at least one variable in the current SCC)
    for (int t = 0; t < timestamps - 1; ++t) {
        prev_mapping = std::move(curr_mapping);
        curr_mapping.clear();
        for (int v : *var_to_scc[var]) {
            variables_used_for_unrolling[v] = true;
            for (int a : axiom_ids_for_var[v]) {
                axioms_used_for_unrolling[a] = true;
                if (cycle_independent_axioms[a] && !replace_propagation_axioms) {
                    continue;
                }
                int new_conditions_size = get_num_operator_effect_conditions(a, 0, true);
                new_conditions.clear();
                new_conditions.reserve(new_conditions_size);
                for (int c = 0; c < new_conditions_size; ++c) {
                    cond = get_operator_effect_condition(a, 0, c, true);
                    if (var_to_scc[cond.var] != var_to_scc[var]) {
                        // Not in the current SCC, keep condition as is
                        new_conditions.emplace_back(cond);
                    }
                    else {
                        // Considered variable is part of the current SCC, need to unroll
                        int new_var;
                        if (prune_unreachable) {
                            if (prev_mapping.count(cond.var)) {
                                new_var = prev_mapping.at(cond.var);
                            }
                            else {
                                // Variable unreachable, since we have a stratified axiom program
                                // we can ignore the check whether this variable appears as it's default value
                                // since it can only be set to its non-default value if it is reachable
                                break;
                            }
                        }
                        else {
                            new_var = get_unrolling_variable_id(var_mapping, cond.var, t);
                        }
                        new_conditions.emplace_back(
                            new_var, 
                            cond.value);
                    }
                } 
                // Variable unreachable, can be skipped (only applies to and will ever be reached
                // from the prune_unreachable improvement)
                if ((int)new_conditions.size() < new_conditions_size) {
                    continue;
                }
                int new_head_var;
                if (prune_unreachable) {
                    if (t == timestamps - 2) {
                        // Last timestamp uses original variable
                        new_head_var = v;
                    }
                    else if (curr_mapping.count(v)) {
                        new_head_var = curr_mapping.at(v);
                    }
                    else{
                        new_head_var = initialize_new_unrolling_var(v, t + 1, timestamps);
                        curr_mapping[v] = new_head_var;
                    }
                }
                else {
                    new_head_var = get_unrolling_variable_id(var_mapping, v, t + 1);
                }
                new_head = FactPair(
                    new_head_var,
                    get_operator_effect(a, 0, true).value);
                default_value_axioms.emplace_back(
                    new_head, vector<FactPair>(new_conditions.begin(), new_conditions.end()));
                ++unrolling_axioms_counter;
            }

        // Finally, create a propagation axiom to propagate the non-default value to the next timestamp
        if (!replace_propagation_axioms) {
            int non_default_value = 1 - get_variable_default_axiom_value(v); // Either 0 -> 1 or 1 -> 0
            int new_propagation_head_var;
            int new_propagation_var;
            if (prune_unreachable) {
                if (prev_mapping.count(v)) {
                    new_propagation_var = prev_mapping.at(v);
                    if (t == timestamps - 2) {
                        // Last timestamp uses original variable
                        new_propagation_head_var = v;
                    }
                    else if (curr_mapping.count(v)) {
                        new_propagation_head_var = curr_mapping.at(v);
                    }
                    else{
                        new_propagation_head_var = initialize_new_unrolling_var(v, t + 1, timestamps);
                        curr_mapping[v] = new_propagation_head_var;
                    }
                }
                else {
                    // Variable unreachable, can be ignored
                    continue;
                }
            }
            else {
                new_propagation_var = get_unrolling_variable_id(var_mapping, v, t);
                new_propagation_head_var = get_unrolling_variable_id(var_mapping, v, t + 1);
            }
            new_head = FactPair(
                    new_propagation_head_var,
                    non_default_value);
            new_conditions.clear();
            new_conditions.emplace_back( 
                new_propagation_var,
                non_default_value);
            default_value_axioms.emplace_back(
                new_head, vector<FactPair>(new_conditions.begin(), new_conditions.end()));
            ++unrolling_axioms_counter;
            }
        }
    }
}


int DefaultValueAxiomsTask::get_unrolling_variable_id(
    const unordered_map<int, int> &var_mapping, 
    int var, 
    int timestamp) {
    int var_at_mapping = var_mapping.at(var);
    int max_timestamps = get_variable_max_timestamps(var_at_mapping);
    // If the variable is at the last timestamp, we use the original value, so we don't have to change the variables in the rest of the axioms
    if (timestamp == max_timestamps - 1) {
        return var;
    }
    // Else we use the variable at the given timestamp
    else {
        return var_at_mapping + timestamp;
    }
}

void DefaultValueAxiomsTask::create_unrolling_variable_mapping_and_initialize_unrolling_variables(
    const vector<int> &vars,
    unordered_map<int, int> &var_mapping) {
    /* 
      The variable mapping maps each variable of the current SCC to their respective variable at timestamp 0
      The actual variable mapping works as follows: current_num_of_vars + t + index_in_scc * (timestamps - 1)
      Example for SCC with current = 10 and using var1, var2 and var3 -> 3 timestamps):
      var1 at t=0 -> 10 + 0 + 0 * (3 - 1) = 10
      var1 at t=1 -> 10 + 1 + 0 * (3 - 1) = 11
      var1 at t=2 -> var1
      var2 at t=0 -> 10 + 0 + 1 * (3 - 1) = 12
      var2 at t=1 -> 10 + 1 + 1 * (3 - 1) = 13
      var2 at t=2 -> var2
      ...
    */
    // Clear old mapping first
    var_mapping.clear();
    int num_variables = get_num_variables();
    int timestamps = vars.size();
    for (int i = 0; i < timestamps; ++i) {
        var_mapping[vars[i]] = num_variables + i * (timestamps - 1);
        initialize_new_unrolling_vars(vars[i], timestamps); // Initialize new variables here to ensure that the mapping is correct
    }
}

void DefaultValueAxiomsTask::initialize_new_unrolling_vars(
    int var,
    int timestamps) {
    for (int t = 0; t < timestamps - 1; ++t) { // No need for last timestamp, as it uses the original variable
        initialize_new_unrolling_var(var, t, timestamps);
    }
}

int DefaultValueAxiomsTask::initialize_new_unrolling_var(
    int var, int timestamp, int max_timestamps) {
    unrolling_variables.emplace_back(
        2, // Domain size, derived variables are binary
        get_variable_name(var) + "_" + to_string(timestamp), // Name
        get_variable_axiom_layer(var), // Axiom layer stays the same
        get_variable_default_axiom_value(var), // Default axiom value stays the same
        timestamp, // Current timestamp
        max_timestamps // Max timestamps
    );
    return get_num_variables() - 1;
}

tuple<vector<vector<int>>, vector<vector<int>>> DefaultValueAxiomsTask::create_nondefault_and_default_dependencies_for_all_axioms() {
    /*
      (non)default_dependencies store for each variable v all derived
      variables that appear with their (non)default value in the body of
      an axiom that sets v.
      Note that the vectors go over *all* variables (also non-derived ones),
      but only the indices that correspond to a variable ID of a derived
      variable actually have content.
     */
    TaskProxy task_proxy(*parent);
    vector<vector<int>> nondefault_dependencies(get_num_variables());
    vector<vector<int>> default_dependencies(get_num_variables());

    for (int i = 0; i < get_num_axioms(); ++i) {
        if (axioms == AxiomHandlingType::EXACT_NEGATIVE_CYCLES && i < default_value_axioms_start_index && axioms_used_for_unrolling[i]) {
            continue; // Skip axioms that were used for unrolling, this may only be axioms from before the default value axioms
        }
        int head_var = get_operator_effect(i, 0, true).var;
        for (int j = 0; j < get_num_operator_effect_conditions(i, 0, true); ++j) {
            FactPair cond = get_operator_effect_condition(i, 0, j, true);
            int cond_var = cond.var;
            bool is_derived = false;
            if (cond_var >= unrolling_vars_start_index) {
                is_derived = true;
            } else {
                is_derived = task_proxy.get_variables()[cond_var].is_derived();
            }
                if (is_derived) {
                    if (cond.value == get_variable_default_axiom_value(cond_var)) {
                        default_dependencies[head_var].push_back(cond_var);
                    } else {
                        nondefault_dependencies[head_var].push_back(cond_var);
                }
            } 
        }
    }

    return {nondefault_dependencies, default_dependencies};
}

vector<vector<int>> DefaultValueAxiomsTask::create_axiom_ids_for_all_vars() {
    /*
      axiom_ids_for_var stores for each derived variable v which
      axioms set v to their nondefault value.
      Note that the vectors go over *all* variables (also non-derived ones),
      but only the indices that correspond to a variable ID of a derived
      variable actually have content.
    */
    vector<vector<int>> axiom_ids_for_var(get_num_variables());
    for (int i = 0; i < get_num_axioms(); ++i) {
        if (axioms == AxiomHandlingType::EXACT_NEGATIVE_CYCLES && i < default_value_axioms_start_index && axioms_used_for_unrolling[i]) {
            continue; // Skip axioms that were used for unrolling, this may only be axioms from before the default value axioms
        }
        int head_var = get_operator_effect(i, 0, true).var;
        axiom_ids_for_var[head_var].push_back(i);
    }
    return axiom_ids_for_var;
}

vector<vector<int> *> DefaultValueAxiomsTask::compute_var_to_scc_from_nondefault_dependencies(
    vector<vector<int>> &nondefault_dependencies,
    vector<vector<int>> &sccs) {
    /*
       Get the sccs induced by nondefault dependencies.
       We do not include default dependencies because they cannot
       introduce additional cycles (this would imply that the axioms
       are not stratifiable, which is already checked in the translator).
    */
    vector<vector<int> *> var_to_scc;
    // We don't need the sccs if we set axioms "v=default <- {}" everywhere.
    if (axioms == AxiomHandlingType::APPROXIMATE_NEGATIVE_CYCLES ||
        axioms == AxiomHandlingType::EXACT_NEGATIVE_CYCLES) {
        sccs = sccs::compute_maximal_sccs(nondefault_dependencies);
        var_to_scc =
            vector<vector<int> *>(get_num_variables(), nullptr);
        for (int i = 0; i < (int)sccs.size(); ++i) {
            for (int var : sccs[i]) {
                var_to_scc[var] = &sccs[i];
            }
        }
    }
    return var_to_scc;
}

int DefaultValueAxiomsTask::get_num_variables() const {
    return unrolling_vars_start_index + unrolling_variables.size();
}

string DefaultValueAxiomsTask::get_variable_name(int var) const {
    if (var < unrolling_vars_start_index) {
        return parent->get_variable_name(var);
    }

    return unrolling_variables[var - unrolling_vars_start_index].name;
}

int DefaultValueAxiomsTask::get_variable_domain_size(int var) const {
    if (var < unrolling_vars_start_index) {
        return parent->get_variable_domain_size(var);
    }

    return 2;
}

int DefaultValueAxiomsTask::get_variable_axiom_layer(int var) const {
    if (var < unrolling_vars_start_index) {
        return parent->get_variable_axiom_layer(var);
    }

    return unrolling_variables[var - unrolling_vars_start_index].axiom_layer;
}

int DefaultValueAxiomsTask::get_variable_default_axiom_value(int var) const {
    if (var < unrolling_vars_start_index) {
        return parent->get_variable_default_axiom_value(var);
    }

    return unrolling_variables[var - unrolling_vars_start_index].axiom_default_value;
}

string DefaultValueAxiomsTask::get_fact_name(const FactPair &fact) const {
    if (fact.var < unrolling_vars_start_index) {
        return parent->get_fact_name(fact);
    }

    return "<none of those>";
}

vector<int> DefaultValueAxiomsTask::get_initial_state_values() const {
    return parent->get_initial_state_values();
}

int DefaultValueAxiomsTask::get_variable_timestamp(int var) const {
    assert(var >= unrolling_vars_start_index);

    return unrolling_variables[var - unrolling_vars_start_index].timestamp;
}

int DefaultValueAxiomsTask::get_variable_max_timestamps(int var) const {
    assert(var >= unrolling_vars_start_index);

    return unrolling_variables[var - unrolling_vars_start_index].max_timestamps;
}

int DefaultValueAxiomsTask::get_operator_cost(int index, bool is_axiom) const {
    if (!is_axiom || index < default_value_axioms_start_index) {
        return parent->get_operator_cost(index, is_axiom);
    }

    return 0;
}

string DefaultValueAxiomsTask::get_operator_name(
    int index, bool is_axiom) const {
    if (!is_axiom || index < default_value_axioms_start_index) {
        return parent->get_operator_name(index, is_axiom);
    }

    return "";
}

int DefaultValueAxiomsTask::get_num_operator_preconditions(
    int index, bool is_axiom) const {
    if (!is_axiom || index < default_value_axioms_start_index) {
        return parent->get_num_operator_preconditions(index, is_axiom);
    }

    return 1;
}

FactPair DefaultValueAxiomsTask::get_operator_precondition(
    int op_index, int fact_index, bool is_axiom) const {
    if (!is_axiom || (op_index < default_value_axioms_start_index)) {
        return parent->get_operator_precondition(
            op_index, fact_index, is_axiom);
    }

    assert(fact_index == 0);
    FactPair head =
        default_value_axioms[op_index - default_value_axioms_start_index].head;
    return FactPair(head.var, 1 - head.value);
}

int DefaultValueAxiomsTask::get_num_operator_effects(
    int op_index, bool is_axiom) const {
    if (!is_axiom || op_index < default_value_axioms_start_index) {
        return parent->get_num_operator_effects(op_index, is_axiom);
    }

    return 1;
}

int DefaultValueAxiomsTask::get_num_operator_effect_conditions(
    int op_index, int eff_index, bool is_axiom) const {
    if (!is_axiom || op_index < default_value_axioms_start_index) {
        return parent->get_num_operator_effect_conditions(
            op_index, eff_index, is_axiom);
    }

    assert(eff_index == 0);
    return default_value_axioms[op_index - default_value_axioms_start_index]
        .condition.size();
}

FactPair DefaultValueAxiomsTask::get_operator_effect_condition(
    int op_index, int eff_index, int cond_index, bool is_axiom) const {
    if (!is_axiom || op_index < default_value_axioms_start_index) {
        return parent->get_operator_effect_condition(
            op_index, eff_index, cond_index, is_axiom);
    }

    assert(eff_index == 0);
    return default_value_axioms[op_index - default_value_axioms_start_index]
        .condition[cond_index];
}

FactPair DefaultValueAxiomsTask::get_operator_effect(
    int op_index, int eff_index, bool is_axiom) const {
    if (!is_axiom || op_index < default_value_axioms_start_index) {
        return parent->get_operator_effect(op_index, eff_index, is_axiom);
    }

    assert(eff_index == 0);
    return default_value_axioms[op_index - default_value_axioms_start_index]
        .head;
}

int DefaultValueAxiomsTask::get_num_axioms() const {
    return parent->get_num_axioms() + default_value_axioms.size();
}

shared_ptr<AbstractTask> get_default_value_axioms_task_if_needed(
    const shared_ptr<AbstractTask> &task, AxiomHandlingType axioms, std::vector<UnrollingOptionType> unrolling_options) {
    TaskProxy proxy(*task);
    if (task_properties::has_axioms(proxy)) {
        return make_shared<tasks::DefaultValueAxiomsTask>(
            DefaultValueAxiomsTask(task, axioms, unrolling_options));
    }
    return task;
}

void add_axioms_option_to_feature(plugins::Feature &feature) {
    feature.add_option<AxiomHandlingType>(
        "axioms",
        "How to compute axioms that describe how the negated "
        "(=default) value of a derived variable can be achieved.",
        "approximate_negative_cycles");
}

tuple<AxiomHandlingType> get_axioms_arguments_from_options(
    const plugins::Options &opts) {
    return make_tuple<AxiomHandlingType>(opts.get<AxiomHandlingType>("axioms"));
}

void add_unrolling_options_to_feature(plugins::Feature &feature) {
    feature.add_list_option<UnrollingOptionType>(
        "unrolling_options",
        "Can only be used with exact_negative_cycles."
        "How to handle the unrolling of negative cycles, "
        "can show improved performance."
        "Zero or more options can be selected.",
        plugins::ArgumentInfo::NO_DEFAULT 
    );
}

tuple<vector<UnrollingOptionType>> get_unrolling_options_arguments_from_options(
    const plugins::Options &opts) {
    vector<UnrollingOptionType> unrolling_options;
    if (opts.contains("unrolling_options")) {
        unrolling_options = opts.get_list<UnrollingOptionType>("unrolling_options");
    }
    return make_tuple(unrolling_options);
}

static plugins::TypedEnumPlugin<AxiomHandlingType> _enum_plugin(
    {{"approximate_negative",
      "Overapproximate negated axioms for all derived variables by "
      "setting an empty condition, indicating the default value can "
      "always be achieved for free."},
     {"approximate_negative_cycles",
      "Overapproximate negated axioms for all derived variables which "
      "have cyclic dependencies by setting an empty condition, "
      "indicating the default value can always be achieved for free. "
      "For all other derived variables, the negated axioms are computed "
      "exactly. Note that this can potentially lead to a combinatorial "
      "explosion."},
    {"exact_negative_cycles",
      "Transform cyclic dependencies into an exact acyclic representation "
      "and calculate negated axioms exactly for all derived variables."
      "Note that this can lead to a significant increase in preprocessing time " 
      "and memory."}});

static plugins::TypedEnumPlugin<UnrollingOptionType> _improvement_enum_plugin(
    {{"prune_unreachable", 
      "Does not create axioms and variables that are not reachable."},
    {"replace_propagation_axioms", 
      "Creates more cycle-independent axioms to replace propagation axioms."},
    {"only_small_cycles", 
      "Only unrolls cycles that are smaller than 10 variables."}});
}
