/**
 * This is the main file.
 * For now, all it does is some test stuff using other classes.
 */

#include <algorithm>
#include <cmath>
#include <ctime>
#include <fstream>
#include <iostream>
#include <iterator>
#include <stdexcept>
#include <string>
#include <vector>

#include "ExperimentGenerator.h"
#include "ICTS.h"
#include "ICTSNode.h"
#include "JMDD.h"
#include "MapGraphNode.h"
#include "MDDFactory.h"
#include "ScenarioLoader.h"
#include "AMDDNode.h"

using namespace std;


//MAIN METHOD BELOW, AFTER A BUNCH OF TEST METHODS


void showNeighbours(int index) {
	GraphNode* testNode =  MapGraphNode::_getGraphNodeWithIndex(index);
	vector<GraphNode*> neighbours = testNode->getNeighbours();
	cout << "Tile " << testNode->toString() << " has " << neighbours.size() << " neighbours:" << endl;
	for (auto it = neighbours.begin(); it != neighbours.end(); ++it) {
		cout << (*it)->toString() << endl;
	}
	cout << endl;
}

void showPath(vector<GraphNode*>* path, int origin, int destination) {
	GraphNode* originNode =  MapGraphNode::_getGraphNodeWithIndex(origin);
	GraphNode* destinationNode =  MapGraphNode::_getGraphNodeWithIndex(destination);

	const int width = 32;
	const int height = 32;

	if (path == 0) {
		cout << "No path from " << originNode->toString() << " to " << destinationNode->toString() << " exists." << endl;
	} else {

		cout << "The optimal conflict-free path from " << originNode->toString() << " to " << destinationNode->toString() << " has " << path->size()-1 << " steps:" << endl;
		//for (auto it = path->begin(); it != path->end(); ++it) {
		//	cout << '\t' << (*it)->toString() << endl;
		//}
		cout << "	see " << "path_" + originNode->toString() + "_" + destinationNode->toString() + ".ppm" << endl;

		int visMap[width*height];
		for (int i = 0; i < width*height; ++i) {
			visMap[i] = MapGraphNode::_isMapPassableAt(i);
		}
		for (auto it = path->begin(); it != path->end(); ++it) {
			visMap[(*it)->getIndex()] += 2;
		}

		std::ofstream visualizationFile;
		visualizationFile.open ("TEST2_path_" + originNode->toString() + "_" + destinationNode->toString() + ".ppm");
		visualizationFile << "P3\n"; //ascii encoded pixmap
		visualizationFile << width << " " << height << " 7\n"; //image dimensions
		for (int i = 0; i < width*height; ++i) {
			if (visMap[i] == 0) {
				visualizationFile << "0 0 0"; //black,	obstacle
			} else if (visMap[i] == 1) {
				visualizationFile << "7 7 7"; //white,	terrain
			} else if (visMap[i] == 2) {
				visualizationFile << "7 7 0"; //yellow,	error, passing unpassable terrain
			} else if (visMap[i] == 3) {
				visualizationFile << "7 0 0"; //red,	passed terrain
			} else {
				visualizationFile << "0 0 7"; //blue,	waiting, probably
			}
			visualizationFile << '\t';
			if ((i+1) % width == 0) visualizationFile << '\n';
		}
		visualizationFile.close();

		//delete path;
	}
	cout << endl;
}
void showPath(int origin, int destination) {
	GraphNode* originNode =  MapGraphNode::_getGraphNodeWithIndex(origin);
	GraphNode* destinationNode =  MapGraphNode::_getGraphNodeWithIndex(destination);
	showPath(originNode->getShortestPathTo(destinationNode), origin, destination);
}

//spits out a whole load of images, showing the agents' progression step by step. Having lots of agents is no problem but long path lengths lead to hundreds of images at 1.5 MB each
void showMultiAgentPath(string experimentName, vector<vector<GraphNode*>>* paths, int frameBound) {

	cout << "start writing frames for experiment '" << experimentName << "'" << endl;

	const int width = 32;
	const int height = 32;

	int agentCount = paths->size();
	int maxPathLength = 0;
	for (auto agentIterator = paths->begin(); agentIterator != paths->end(); ++agentIterator) {
		if ((int) agentIterator->size() > maxPathLength) maxPathLength = (int) agentIterator->size();
	}

	cout << maxPathLength << " frames expected" << endl;
	if (frameBound > 0) {
		if (maxPathLength > frameBound) {
			maxPathLength = frameBound;
			cout << " truncating to " << frameBound << " frames" << endl;
		}
	}

	bool* baseMap = new bool[width*height];
	for (int i = 0; i < width*height; ++i) {
		baseMap[i] = MapGraphNode::_isMapPassableAt(i);
	}

	int* trailMap = new int[width*height];
	for (int i = 0; i < width*height; ++i) {
		trailMap[i] = -1;
	}

	vector<string> agentColours; //darker
	agentColours.push_back("4 0 0"); //red
	agentColours.push_back("0 0 4"); //blue
	agentColours.push_back("0 3 0"); //green
	agentColours.push_back("4 3 0"); //yellow
	agentColours.push_back("0 4 4"); //cyan
	agentColours.push_back("4 0 4"); //magenta
	agentColours.push_back("2 0 4"); //purple
	agentColours.push_back("0 2 4"); //sky blue
	agentColours.push_back("4 2 0"); //orange
	agentColours.push_back("4 0 2"); //pink
	agentColours.push_back("2 4 0"); //lime
	agentColours.push_back("0 4 2"); //teal
	

	vector<string> trailColours; //lighter, faded
	trailColours.push_back("8 6 6"); //red
	trailColours.push_back("6 6 8"); //blue
	trailColours.push_back("6 8 6"); //green
	trailColours.push_back("8 8 6"); //yellow
	trailColours.push_back("6 8 8"); //cyan
	trailColours.push_back("8 6 8"); //magenta
	trailColours.push_back("7 6 8"); //purple
	trailColours.push_back("6 7 8"); //sky blue
	trailColours.push_back("8 7 6"); //orange
	trailColours.push_back("8 6 7"); //pink
	trailColours.push_back("7 8 6"); //lime
	trailColours.push_back("6 8 7"); //teal

	for (int frameIndex = 0; frameIndex < maxPathLength; ++frameIndex) {
		std::ofstream visualizationFile;
		visualizationFile.open (experimentName + "_frame_" + std::to_string(frameIndex) + ".ppm");
		visualizationFile << "P3\n"; //ascii encoded pixmap
		visualizationFile << width << " " << height << " 8\n"; //image dimension

		for (int mapTile = 0; mapTile < width*height; ++mapTile) {

			string nextEntry = "";

			//background: collision map
			if (baseMap[mapTile]) {
				nextEntry = "8 8 8";
			} else {
				nextEntry = "0 0 0";
			}

			//middle layer: trails
			if (trailMap[mapTile] != -1) {
				nextEntry = trailColours[trailMap[mapTile] % trailColours.size()];
			}

			//highest layer: agents
			for (int agentIndex = 0; agentIndex < agentCount; ++agentIndex) {
				int boundedFrameIndex = frameIndex;
				if (frameIndex >= (int) (*paths)[agentIndex].size()) {
					boundedFrameIndex = (int) (*paths)[agentIndex].size() - 1;
				}
				if (mapTile == (*paths)[agentIndex][boundedFrameIndex]->getIndex()) {
					nextEntry = agentColours[agentIndex % agentColours.size()];
					trailMap[mapTile] = agentIndex;
				}
			}


			visualizationFile << nextEntry << '\t';
			if ((mapTile+1) % width == 0) visualizationFile << '\n';
		}

		visualizationFile.close();

	}

	cout << "finished writing '" << experimentName << "'" << endl;

	delete baseMap;
	delete trailMap;

}

void showMultiAgentPath(string experimentName, vector<vector<GraphNode*>>* paths) {
	showMultiAgentPath(experimentName, paths, -1);
}

void visualizeHeatMap(string experimentName) {

	const double * heatMap = MDDFactory::_getHeatMap();

	std::ofstream visualizationFile;
	visualizationFile.open (experimentName + "_heatMap.ppm");
	visualizationFile << "P3\n"; //ascii encoded pixmap
	visualizationFile << MapGraphNode::_getMapWidth() << " " << MapGraphNode::_getMapHeight() << " 255\n"; //image dimension

	for (int mapTile = 0; mapTile < MapGraphNode::_getMapSize(); ++mapTile) {

		string nextEntry = "";

		if (heatMap[mapTile] <= 0.0) {
			if (MapGraphNode::_isMapPassableAt(mapTile)) {
				nextEntry = "255 255 255";
			} else {
				nextEntry = "0 0 255";
			}
		} else {
			cout << heatMap[mapTile] << " ";
			if ((mapTile+1) % 512 == 0) cout << endl;
			/*
			int intensity = round(heatMap[mapTile] * 48.0);
			intensity = max(0, min(765, intensity));
			if (intensity >= 510) {
				nextEntry = to_string(765 - intensity) + " 0 0";
			} else if (intensity >= 255) {
				nextEntry = "255 " + to_string(510 - intensity) + " 0";
			} else {
				nextEntry = "255 255 " + to_string(255 - intensity);
			}
			*/
			int intensity = round(heatMap[mapTile] * 32.0);
			intensity = max(0, min(510, intensity));
			if (intensity >= 255) {
				nextEntry = to_string(510 - intensity) + " 0 0";
			} else {
				nextEntry = "255 " + to_string(255 - intensity) + " " + to_string(255 - intensity);
			}
		}


		visualizationFile << nextEntry << '\t';
		if ((mapTile+1) % 512 == 0) visualizationFile << '\n';
	}

	visualizationFile.close();
}

/**
 * Previously I had false positives in conflict detection which proved a huge problem.
 * I'm not letting the next method I deploy get away without at least checking everything on an 8x8 grid; every false positive I can think of would appear even on a 4x4 and I dont think there are any that only appear past a certain scale.
 */
void verifyConflictDetection() {

	//MapGraphNode::_loadMap("benchmarks/8x8grid/8x8grid.map");
	MapGraphNode::_loadMap("benchmarks/32x32grid/0obstacles/32x32grid_0obstacles.map");
	ICTS::_setGraph(MapGraphNode::_getGraphNodeWithIndex(0));

	GraphNode* sourceNode = MapGraphNode::_getGraphNodeWithIndex(0);

	int width = MapGraphNode::_getMapWidth();
	int height = MapGraphNode::_getMapHeight();

	int errorCount = 0;

	cout << "begin rigorous checking of conflict detection" << endl;

	for (int o_a_y = 0; o_a_y < height; ++o_a_y) {
		for (int o_a_x = 0; o_a_x < width; ++o_a_x) {
			for (int o_b_y = 0; o_b_y < height; ++o_b_y) {
				for (int o_b_x = 0; o_b_x < width; ++o_b_x) {
					for (int diff_a_x = -1; diff_a_x <= 1; ++diff_a_x) {
						int d_a_x = o_a_x + diff_a_x;
						if (d_a_x < 0 || d_a_x >= width) continue;
						for (int diff_a_y = -1; diff_a_y <= 1; ++diff_a_y) {
							int d_a_y = o_a_y + diff_a_y;
							if (d_a_y < 0 || d_a_y >= height) continue;
							for (int diff_b_x = -1; diff_b_x <= 1; ++diff_b_x) {
								int d_b_x = o_b_x + diff_b_x;
								if (d_b_x < 0 || d_b_x >= width) continue;
								for (int diff_b_y = -1; diff_b_y <= 1; ++diff_b_y) {
									int d_b_y = o_b_y + diff_b_y;
									if (d_b_y < 0 || d_b_y >= height) continue;

									bool conflict = sourceNode->isTransitionPairConflicting(o_a_x + width * o_a_y,
																							d_a_x + width * d_a_y,
																							o_b_x + width * o_b_y,
																							d_b_x + width * d_b_y);
									//but is it a REAL conflict?
									bool realConflict = false;
									string type = "unknown";
									//if (o_a_x == o_b_x && o_a_y == o_b_y) { //decided that is not a real conflict; at least not in THIS step. Rather the previous step would have a conflict to result in those starting positions
									//	//same starting point
									//	realConflict = true;
									//	type = "same start";
									//} else
									if (d_a_x == d_b_x && d_a_y == d_b_y) {
										//same destination point
										realConflict = true;
										type = "same destination";
									} else if (o_a_x == d_b_x && o_a_y == d_b_y && d_a_x == o_b_x && d_a_y == o_b_y) {
										//frontal collision
										realConflict = true;
										type = "frontal collision";
									} else if (abs(o_a_x - d_a_x) == 1 && abs(o_a_y - d_a_y) == 1 && abs(o_b_x - d_b_x) == 1 && abs(o_b_y - d_b_y) == 1) {
										//both are diagonal but not necessarily conflicting
										type = "diagonal";
										if (d_a_x > o_a_x) {
											//pointing right
											if (d_a_y > o_a_y) {
												//pointing up
												if (o_b_x == o_a_x + 1 && o_b_y == o_a_y  &&  d_b_x == d_a_x - 1 && d_b_y == d_a_y) {
													realConflict = true;
												} else if (o_b_y == o_a_y + 1 && o_b_x == o_a_x  &&  d_b_y == d_a_y - 1 && d_b_x == d_a_x) {
													realConflict = true;
												}
											} else if (d_a_y < o_a_y) {
												//pointing down
												if (o_b_x == o_a_x + 1 && o_b_y == o_a_y  &&  d_b_x == d_a_x - 1 && d_b_y == d_a_y) {
													realConflict = true;
												} else if (o_b_y == o_a_y - 1 && o_b_x == o_a_x  &&  d_b_y == d_a_y + 1 && d_b_x == d_a_x) {
													realConflict = true;
												}
											} else {
												throw std::runtime_error("Made a mistake creating conflict detection test - diagonal is not diagonal.");
											}
										} else if (d_a_x < o_a_x) {
											//pointing left
											if (d_a_y > o_a_y) {
												//pointing up
												if (o_b_x == o_a_x - 1 && o_b_y == o_a_y  &&  d_b_x == d_a_x + 1 && d_b_y == d_a_y) {
													realConflict = true;
												} else if (o_b_y == o_a_y + 1 && o_b_x == o_a_x  &&  d_b_y == d_a_y - 1 && d_b_x == d_a_x) {
													realConflict = true;
												}
											} else if (d_a_y < o_a_y) {
												//pointing down
												if (o_b_x == o_a_x - 1 && o_b_y == o_a_y  &&  d_b_x == d_a_x + 1 && d_b_y == d_a_y) {
													realConflict = true;
												} else if (o_b_y == o_a_y - 1 && o_b_x == o_a_x  &&  d_b_y == d_a_y + 1 && d_b_x == d_a_x) {
													realConflict = true;
												}
											} else {
												throw std::runtime_error("Made a mistake creating conflict detection test - diagonal is not diagonal.");
											}
										} else {
											throw std::runtime_error("Made a mistake creating conflict detection test - diagonal is not diagonal.");
										}
									}

									if (conflict && !realConflict) {
										cout << "<!!!> FALSE POSITIVE <!!!>" << endl;
										cout << "\t(" << o_a_x << "," << o_a_y << ")->(" << d_a_x << "," << d_a_y << ") and (" << o_b_x << "," << o_b_y << ")->(" << d_b_x << "," << d_b_y << ")" << endl;
										cout << "\twith " << (o_a_x + width * o_a_y) - (o_b_x + width * o_b_y) << endl;
										++errorCount;
									} else if (!conflict && realConflict) {
										cout << "<!!!> FALSE NEGATIVE <!!!>" << endl;
										cout << "\ttype = " << type << endl;
										cout << "\t(" << o_a_x << "," << o_a_y << ")->(" << d_a_x << "," << d_a_y << ") and (" << o_b_x << "," << o_b_y << ")->(" << d_b_x << "," << d_b_y << ")" << endl;
										cout << "\twith " << (o_a_x + width * o_a_y) - (o_b_x + width * o_b_y) << endl;
										++errorCount;
									}
								}
							}
						}
					}
				}
			}
		}
	}

	cout << "end of conflict detection check" << endl;
	if (errorCount == 0) {
		cout << "no errors detected" << endl;
	} else {
		cout << errorCount << " errors detected" << endl;
	}

}

string tileToString(int tile) {
	return
			"(" +
			std::to_string(tile % MapGraphNode::_getMapWidth()) +
			"," +
			std::to_string(tile / MapGraphNode::_getMapWidth()) +
			")" ;
}

void visualizeMDD(MDD* toVisualize, string name) {
	std::ofstream visualizationFile;
	visualizationFile.open(name + "_MDD.gv");
	visualizationFile << "digraph " << name << " {\n";

	std::set<MDDNode*> currentLayer;
	currentLayer.insert(toVisualize->getHead());
	std::set<MDDNode*> nextLayer;
	for (int i = 0; i <= toVisualize->getCost(); ++i) {
		for (auto layer_it = currentLayer.begin(); layer_it != currentLayer.end(); ++layer_it) {
			for (auto successor_it = (*layer_it)->successors.begin(); successor_it != (*layer_it)->successors.end(); ++successor_it) {
				nextLayer.insert(*successor_it);
			}
			string displayName = "";
			vector<int> allTiles = (*layer_it)->getAllTiles();
			auto tileIterator = allTiles.begin();
			displayName += tileToString(*tileIterator);
			for (++tileIterator; tileIterator != allTiles.end(); ++tileIterator) {
				displayName += "\\n";
				displayName += tileToString(*tileIterator);
			}
			visualizationFile << ((size_t) *layer_it) << " [label=\"" << displayName << "\"];\n";
			for (auto predecessor_it = (*layer_it)->predecessors.begin(); predecessor_it != (*layer_it)->predecessors.end(); ++predecessor_it) {
				visualizationFile << ((size_t) *predecessor_it) << " -> " << ((size_t) *layer_it) << ";\n";
			}
		}
		currentLayer.clear();
		currentLayer = std::move(nextLayer);
		nextLayer.clear();
	}

	visualizationFile << "}\n";
	visualizationFile.close();
}

/**
 * MAIN METHOD
 */
int main(int argc, char* argv[]) {

	//Parsing arguments
	if (argc <= 2) {
		cout << "Usage: '" << argv[0] << " mapfile.map scenariofile.scen'" << endl;
		cout << "Use: '" << argv[0] << " --help' for more information" << endl;
		return 0;
	}

	char* mapFileName;
	char* scenarioFileName;
	string abstractionType = "none";
	int forcedAdjacency = 0;
	bool abstract = false;
	double heatThresholdMultiplier = 1.0;
	string pathOutputType = "none";

	int i;
	for (i = 1; i < argc; ++i) {
		string argument = argv[i];
		string prefix = "";

		//--help
		prefix = "--help";
		if(argument.substr(0, prefix.size()) == prefix) {
			cout << "Usage: '" << argv[0] << " mapfile.map scenariofile.scen'" << endl;
			cout << "Possible arguments to give before that:" << endl;
			cout << "\t--help \tdisplays this message." << endl;
			cout << "\t--abstraction=type \tuses the abstraction method 'type'. Abstraction methods are" << endl;
			cout << "\t\tnone \tno abstraction (default)" << endl;
			cout << "\t\theatMapThreshold \tthe currently only abstraction included" << endl;
			cout << "\t--forceAdjacency=number \tmakes the map use 'numer'-adjacency. Can only be 4, 8, or 0 which takes the map's default" << endl;
			cout << "\t--pathOutput=type \tmakes the program print the resulting paths. Can be none (no path output), index (graph node indices), or string (graph node coordinates)" << endl;
			//cout << "\t--verbose \tmakes the program print a bunch of stuff to the standard output" << endl; //maybe later
			return 0;
		}

		//--abstraction
		prefix = "--abstraction=";
		if(argument.substr(0, prefix.size()) == prefix) {
			abstractionType = argument.substr(prefix.size());
			if (abstractionType == "none") {
				abstract = false;
			} else if (abstractionType == "heatMapThreshold") {
				abstract = true;
			} else {
				cout << "Unknown abstraction type '" << abstractionType << "'!" << endl;
				cout << "Please use 'none' or 'heatMapThreshold'." << endl;
				return -1;
			}
			continue;
		}

		//--forceAdjacency
		prefix = "--forceAdjacency=";
		if(argument.substr(0, prefix.size()) == prefix) {
			string adjacencyType = argument.substr(prefix.size());
			if (adjacencyType == "4" || adjacencyType == "quartile") {
				forcedAdjacency = 4;
			} else if (adjacencyType == "8" || adjacencyType == "octile") {
				forcedAdjacency = 8;
			} else if (adjacencyType == "0" || adjacencyType == "default") {
				forcedAdjacency = 0;
			} else {
				cout << "Unknown adjacency '" << adjacencyType << "'!" << endl;
				cout << "Please use '4' / 'quartile, '8' / 'octile' or '0' / 'default'." << endl;
				return -1;
			}
			continue;
		}

		//--threshold
		prefix = "--threshold=";
		if(argument.substr(0, prefix.size()) == prefix) {
			string thresholdType = argument.substr(prefix.size());
			if (thresholdType == "low") {
				heatThresholdMultiplier = 0.5;
			} else if (thresholdType == "medium" || thresholdType == "normal" || thresholdType == "default") {
				heatThresholdMultiplier = 1.0;
			} else if (thresholdType == "high") {
				heatThresholdMultiplier = 5.0 / 3.0; //~1.6667
			} else if (thresholdType == "infinite" || thresholdType == "infinity") {
				heatThresholdMultiplier = 10000.0; //not truly infinite, but good luck getting above that
			} else {
				try {
					heatThresholdMultiplier = std::stod(thresholdType);
				} catch (const std::invalid_argument& e) {
					cout << "Unknown threshold quantifier '" << thresholdType << "'!" << endl;
					cout << "Please use low (0.5), normal (1.0), high (1.6667), infinite, or any decimal number." << endl;
					return -1;
				}
			}
			continue;
		}

		//--pathOutput
		prefix = "--pathOutput=";
		if(argument.substr(0, prefix.size()) == prefix) {
			string outputType = argument.substr(prefix.size());
			if (outputType == "none" || outputType == "default") {
				pathOutputType = "none";
			} else if (outputType == "index" || outputType == "indices") {
				pathOutputType = "index";
			} else if (outputType == "string") {
				pathOutputType = "string";
			} else {
				cout << "Unknown path output type '" << outputType << "'!" << endl;
				cout << "Please use 'none', 'index' or 'string'." << endl;
				return -1;
			}
			continue;
		}

		//since we got here without matching a known parameter, we either have an unknown parameter or we now get the map and scenario file
		prefix = "--";
		if(argument.substr(0, prefix.size()) == prefix) {
			cout << "Unknown parameter '" << argument << "'!" << endl;
			cout << "Use: '" << argv[0] << " --help' for more information" << endl;
			cout << "Keep in mind some parameters such as --abstraction' must be followed by '=value' to be recognized." << endl;
			return -1;
		}

		//when we get here we can only assume what's left is the map and scenario files
		break;
	}
	if (argc - i != 2) { //if we don't have exactly 2 arguments left after breaking from parameter parsing above
		cout << "Wrong number of parameters. 2 arguments expected (map and scenario), " << argc - i << " received." << endl;
		cout << "Usage: '" << argv[0] << " mapfile.map scenariofile.scen'" << endl;
		return -1;
	}
	mapFileName = argv[i];
	scenarioFileName = argv[i+1];

	//print parameters
	cout << "Using map '" << mapFileName << "'";
	if (forcedAdjacency == 4) {
		cout << " with forced 4-adjacency." << endl;
	} else if (forcedAdjacency == 8) {
		cout << " with forced 8-adjacency." << endl;
	} else {
		cout << " with default adjacency." << endl;
	}
	cout << "Using scenario '" << scenarioFileName << "'." << endl;
	cout << "Using abstraction type '" << abstractionType << "'";
	if (abstract) {
		cout << " with heat threshold multiplier " << heatThresholdMultiplier;
	}
	cout << "." << endl;
	cout << "Using path display type '" << pathOutputType << "'." << endl;

	//Loads arguments as files for a 8-adjacency grid and a scenario on such a grid.
	//4-adjacency grids are supported, but none of my test maps use them per default.
	//Other types of graphs are in theory supported (the ICTS functions don't presuppose a grid) but no file format for them is supported.
	MapGraphNode::_loadMap(mapFileName);
	if (forcedAdjacency == 4) {
		MapGraphNode::_setAdjacency(false);
	} else if (forcedAdjacency == 8) {
		MapGraphNode::_setAdjacency(true);
	}
	ICTS::_setGraph(MapGraphNode::_getGraphNodeWithIndex(0));
	ScenarioLoader loader(scenarioFileName);

	int mapWidth = MapGraphNode::_getMapWidth();

	ICTS::_setHeatThresholdMultiplier(heatThresholdMultiplier);

	//START OF USING ALL AGENTS
	cout << "Using all " << loader.GetNumExperiments() << " available agents." << endl;
	if (loader.GetNumExperiments() <= 0) throw std::runtime_error("Could not open scenario file.");
	vector<Agent>* agents = new vector<Agent>();
	agents->reserve(loader.GetNumExperiments());
	vector<int>* initialCosts = new vector<int>();
	initialCosts->reserve(loader.GetNumExperiments());

	Experiment experiment = loader.GetNthExperiment(0);

	for (int agentIndex = 0; agentIndex < loader.GetNumExperiments(); ++agentIndex) {
		experiment = loader.GetNthExperiment(agentIndex);

		//check if start is unobstructed
		if (!MapGraphNode::_isMapPassableAt(experiment.GetStartX() + mapWidth * experiment.GetStartY())) {
			cout << "Encountered impossible agent " << agentIndex << " while loading scenario." << endl;
			cout << "Start Position (" << experiment.GetStartX() << ", " << experiment.GetStartY() << ") obstructed." << endl;
			cout << "CANCELLING EXECUTION" << endl;
			return -1;
		}

		//encode as agent
		(*agents).push_back(
				Agent(
						agentIndex,
						experiment.GetStartX() + mapWidth * experiment.GetStartY(),
						experiment.GetGoalX() + mapWidth * experiment.GetGoalY()));
		(*initialCosts).push_back(
				MapGraphNode::_getGraphNodeWithIndex((*agents)[agentIndex].startTile)
				->getDistanceTo(
				MapGraphNode::_getGraphNodeWithIndex((*agents)[agentIndex].goalTile)));

		//check if singe agent path is possible
		if ((*initialCosts)[agentIndex] < 0) {
			cout << "Encountered impossible agent " << agentIndex << " while loading scenario." << endl;
			cout << "No path from (" << experiment.GetStartX() << ", " << experiment.GetStartY() << ") to (" << experiment.GetGoalX() << ", " << experiment.GetGoalY() << ")." << endl;
			cout << "CANCELLING EXECUTION" << endl;
			return -1;
		}

	}
	//END OF USING ALL AGENTS

	cout << "<<< Beginning ICTS Algorithm. >>>" << endl;

	//START TIMED BLOCK
	clock_t begin = clock();
	vector<vector<GraphNode*>>* paths = ICTS::_performIncreasingCostTreeSearchWithIndependenceDetection(agents, initialCosts, abstract);
	clock_t end = clock();
	//END TIMED BLOCK

	double elapsed_secs = double(end - begin) / CLOCKS_PER_SEC;
	cout << "Search time = " << std::to_string(elapsed_secs) << " seconds" << endl;
	if (elapsed_secs < 1) {
		cout << "under 1 second" << endl;
	}
	if (elapsed_secs < 60) {
		cout << "under 1 minute" << endl;
	}
	if (elapsed_secs < 300) {
		cout << "under 5 minutes" << endl;
	}
	if (elapsed_secs < 600) {
		cout << "under 10 minutes" << endl;
	}

	if (paths != 0) {
		int makespan = 0;
		int sumofcosts = 0;
		cout << "ICTS returned paths." << endl;
		cout << "It has " << paths->size() << " agents" << endl;
		if (paths->size() > 0) { //just safety measure in case of 0-agent trick problem
			cout << "Its paths have lengths ";
			cout << paths->begin()->size();
			makespan = (int) paths->begin()->size();
			sumofcosts = (int) paths->begin()->size();
			for (auto pathIterator = paths->begin() + 1; pathIterator != paths->end(); ++pathIterator) {
				cout << ", " << pathIterator->size();
				if ((int) pathIterator->size() > makespan) makespan = (int) pathIterator->size();
				sumofcosts += (int) pathIterator->size();
			}
			cout << endl;
		}
		cout << "Makespan = " << makespan << endl;
		cout << "Sum of Costs = " << sumofcosts << endl;
		//showMultiAgentPath("experimentName", paths, 32);

		//display paths
		int agentIndex = 0;
		if (pathOutputType != "none") {
			for (auto agentIterator = paths->begin(); agentIterator != paths->end(); ++agentIterator) {
				cout << "Agent " << agentIndex++ << " has path ";
				if (pathOutputType == "index") {
					cout << (*agentIterator->begin())->getIndex();
					for (auto tileIterator = agentIterator->begin() + 1; tileIterator != agentIterator->end(); ++tileIterator) {
						cout << ", " << (*tileIterator)->getIndex();
					}
					cout << endl;
				} else {
					cout << (*agentIterator->begin())->toString();
					for (auto tileIterator = agentIterator->begin() + 1; tileIterator != agentIterator->end(); ++tileIterator) {
						cout << ", " << (*tileIterator)->toString();
					}
					cout << endl;
				}
			}
		}
	} else {
		cout << "The path is null; ICTS returned no paths." << endl;
	}

	//end of testing; free memory
	delete agents;
	delete initialCosts;
	delete paths;
	ICTS::_freeStoredMDDs();
	MDDFactory::_freeMemory();
	MapGraphNode::_freeMap();

cout << endl << ">>>End Program" << endl;
return 0;
}
