/*
 * MapGraphNode.cpp
 *
 *  Created on: 15.10.2017
 *      Author: Simon
 */

#include "MapGraphNode.h"

#include <cstdlib>
#include <stdexcept>

#include "MapLoader.h"

MapGraphNode::MapGraphNode(int index) {
	this->index = index;
}

MapGraphNode::~MapGraphNode() {} //destructor, don't think I actually need one

// default values for static variables. Actual (usable) values are set by calling MapGraphNode::_loadMap()
bool* MapGraphNode::_map = 0;
int MapGraphNode::_mapWidth = 0;
int MapGraphNode::_mapHeight = 0;
bool MapGraphNode::_eightAdjacency = false;
MapGraphNode** MapGraphNode::_nodes = 0;

//Use Sturtevant's map format
//TODO Doesn't catch any exceptions yet (just crashes if file not found)
//TODO do I really want to generate a node for every unpassable tile?
//  I mean, I probably won't ever need them
//  If I don't I'll have to change _nodes from an array to a std:map
//    But do I want to go from O(1) random access to O(log n)
//    Seeing how the memory required will stay O(n)
//    I can see memory being the bottleneck overall, but not this piece in particular
// I suppose if I'm that worried I can keep nullpointers for unpassable tiles
// But I'll only do that if I find myself thinking:
//   "Wow, these 2MB of Objects really are too much, I'll have to cut down on those"
void MapGraphNode::_loadMap(const char* filename) {
	MapData mapdata = MapLoader::_loadSturtevantMapAsBoolArray(filename);
	_map = mapdata.map;
	_mapWidth = mapdata.width;
	_mapHeight = mapdata.height;
	_eightAdjacency = mapdata.eightAdjacency;
	_nodes = new MapGraphNode*[_mapWidth * _mapHeight];
	for (int i = 0; i < _mapWidth * _mapHeight; ++i) {
		_nodes[i] = new MapGraphNode(i); //This is the ONLY time new MapGraphNode objects should be generated. Ever.
	}
}

//Test version, creates an empty 5x5 map
void MapGraphNode::_loadMap() {
	_map = new bool[25];
	for (int i = 0; i < 25; ++i) {
		_map[i] = true;
	}
	_mapWidth = 5;
	_mapHeight = 5;
	_eightAdjacency = false;
	_nodes = new MapGraphNode*[25];
	for (int i = 0; i < 25; ++i) {
		_nodes[i] = new MapGraphNode(i);
	}
}

//releases held memory, kind of the static deconstructor if you will
void MapGraphNode::_freeMap() {
	if (_nodes != 0) {
		for (int i = 0; i < _mapWidth * _mapHeight; ++i) {
			delete _nodes[i];
		}
		delete[] _nodes;
	}
	if (_map != 0) {
		delete[] _map;
	}
}

void MapGraphNode::_setAdjacency(bool adjacency) {
	_eightAdjacency = adjacency;
}

bool MapGraphNode::_isMapPassableAt(int index) {
	if (_nodes == 0) throw std::runtime_error("No map loaded for MapGraphNode");
	return _map[index];
}

std::vector<GraphNode*> MapGraphNode::getNeighbours() {
	if (_nodes == 0) throw std::runtime_error("No map loaded for MapGraphNode");

	std::vector<GraphNode*> neighbours;

	bool leftCleared = (index % _mapWidth != 0);  //true iff we're not on the left edge of the map
	bool rightCleared = ((index + 1) % _mapWidth != 0);  //true iff we're not on the right edge of the map
	bool topCleared = (index >= _mapWidth);  //true iff we're not on the top edge of the map
	bool bottomCleared = ((index / _mapWidth) < (_mapHeight - 1));  //true iff we're not on the bottom edge of the map

	//left
	if (leftCleared) { //if we're not on the left edge of the map
		if (_map[index - 1]) { //if the left tile is unobstructed
			neighbours.push_back(_nodes[index - 1]);
		}
	}
	//right
	if (rightCleared) { //if we're not on the right edge of the map
		if (_map[index + 1]) { //if the right tile is unobstructed
			neighbours.push_back(_nodes[index + 1]);
		}
	}
	//top
	if (topCleared) { //if we're not on the top edge of the map
		if (_map[index - _mapWidth]) { //if the top tile is unobstructed
			neighbours.push_back(_nodes[index - _mapWidth]);
		}
	}
	//bottom
	if (bottomCleared) { //if we're not on the bottom edge of the map
		if (_map[index + _mapWidth]) { //if the bottom tile is unobstructed
			neighbours.push_back(_nodes[index + _mapWidth]);
		}
	}

	if (_eightAdjacency) { //if 8-adjacency is activated we have 4 more neighbours to check

		//top left
		if (topCleared && leftCleared) { //if we're not on the top or left edge of the map
			if (_map[index - _mapWidth - 1]) { //if the top left tile is unobstructed
				neighbours.push_back(_nodes[index - _mapWidth - 1]);
			}
		}

		//top right
		if (topCleared && rightCleared) { //if we're not on the top or right edge of the map
			if (_map[index - _mapWidth + 1]) { //if the top right tile is unobstructed
				neighbours.push_back(_nodes[index - _mapWidth + 1]);
			}
		}

		//bottom left
		if (bottomCleared && leftCleared) { //if we're not on the bottom or left edge of the map
			if (_map[index + _mapWidth - 1]) { //if the bottom left tile is unobstructed
				neighbours.push_back(_nodes[index + _mapWidth - 1]);
			}
		}

		//bottom right
		if (bottomCleared && rightCleared) { //if we're not on the bottom or right edge of the map
			if (_map[index + _mapWidth + 1]) { //if the bottom right tile is unobstructed
				neighbours.push_back(_nodes[index + _mapWidth + 1]);
			}
		}

	}

	return neighbours;
}

string MapGraphNode::toString() {
	return "(" + std::to_string(index % _mapWidth) + "," + std::to_string(index / _mapWidth) + ")";
}

int MapGraphNode::getGraphSize() {
	return _mapWidth * _mapHeight;
}

GraphNode* MapGraphNode::_getGraphNodeWithIndex(int index) {
	if (_nodes == 0) throw std::runtime_error("No map loaded for MapGraphNode");
	return _nodes[index];
}

GraphNode* MapGraphNode::getNodeFromGraph(int index) {
	return MapGraphNode::_getGraphNodeWithIndex(index);
}

double MapGraphNode::estimateDistanceTo(int destinationIndex){
	int xDistance = abs((destinationIndex % _mapWidth) - (index % _mapWidth));
	int yDistance = abs((destinationIndex / _mapWidth) - (index / _mapWidth));
	if (_eightAdjacency) {
		return (double) (xDistance > yDistance ? xDistance : yDistance);
	} else {
		return (double) (xDistance + yDistance);
	}
}

double MapGraphNode::estimateDistanceTo(GraphNode* destination){
	return estimateDistanceTo(destination->getIndex());
}

int MapGraphNode::_getMapWidth() {
	if (_nodes == 0) throw std::runtime_error("No map loaded for MapGraphNode");
	return _mapWidth;
}

int MapGraphNode::_getMapHeight() {
	if (_nodes == 0) throw std::runtime_error("No map loaded for MapGraphNode");
	return _mapHeight;
}

int MapGraphNode::_getMapSize() {
	if (_nodes == 0) throw std::runtime_error("No map loaded for MapGraphNode");
	return _mapWidth * _mapHeight;
}

//rigorously tested on all combinations of transitions in a 32x32 grid.
bool inline MapGraphNode::isTransitionPairConflicting(int originA, int destinationA, int originB, int destinationB) {
	if (destinationA == destinationB) return true;
	else if (originA - originB == destinationB - destinationA) { //should handle all "going over cross" be it on the same edge or diagonally with 8-adjacency but also contains false positives
		int dist = originA - originB;
		if (  dist == 1 || dist == -1 || dist == _mapWidth || dist == -_mapWidth || dist == _mapWidth+1 || dist == -_mapWidth+1 || dist == _mapWidth-1 || dist == -_mapWidth-1  ) { //should filter out all false positives
			return true;
		} else {
			return false;
		}
	} else {
		return false;
	}
}
