/*
 * MDDFactory.cpp
 *
 *  Created on: 20.10.2017
 *      Author: Simon
 */

#include "MDDFactory.h"

#include <algorithm>
#include <climits>
#include <cmath>
#include <iterator>
#include <queue>
#include <set>
#include <stdexcept>
#include <unordered_set>
#include <utility>

#include "AMDDNode.h"
#include "EMDDNode.h"
#include "GraphNode.h"
#include "ICTS.h"
#include "JMDD.h"
#include "MDD.h"

using std::vector;
using std::map;
using std::make_pair;
using std::pair;
using std::queue;
using std::priority_queue;
using std::set;
using std::unordered_set;
using std::pow;

//default static field values
map<int,int*> MDDFactory::_distanceMap;
double* MDDFactory::_heatMap;

void MDDFactory::_initializeHeatMap(int size) {
	_heatMap = new double[size];
	for (int i = 0; i < size; ++i) {
		_heatMap[i] = 0.0;
	}
}

//Traverses graph in breadth-first order to fill distancemap.
void MDDFactory::_addDistanceMapEntry(GraphNode* goalNode) {
	if (_distanceMap.find(goalNode->getIndex()) != _distanceMap.end()) return; //entry already exists

	int* distanceMap = new int[goalNode->getGraphSize()];
	for (int i = 0; i < goalNode->getGraphSize(); ++i) {
		distanceMap[i] = INT_MAX; // initialize distance map to INT_MAX
		                          // effectively means "unreachable node"
	}
	distanceMap[goalNode->getIndex()] = 0;

	queue<GraphNode*> open; //stores pointers to border nodes in FIFO manner
	bool seen[goalNode->getGraphSize()]; //stores shared pointers to all nodes seen already
	                                     //by the end of the algorithm will have all reachable nodes of the graph
	for (int i = 0; i < goalNode->getGraphSize(); ++i) {
		seen[i] = 0; //initialize seen; important, omitting leads to bugs!
	}

	open.push(goalNode); //start with goal node (and work backwards through the map)
	seen[goalNode->getIndex()] = true;

	GraphNode* next;
	vector<GraphNode*> neighbours;
	int neighbourIndex;

	while (!open.empty()) {

		next = open.front();
		open.pop();

		//expand
		neighbours = next->getNeighbours();
		for (auto it = neighbours.begin(); it != neighbours.end(); ++it) { //for every neighbour...
			neighbourIndex = (*it)->getIndex();
			if (!seen[neighbourIndex]) { //... that we haven't handled already in the past...
				seen[neighbourIndex] = true; //... mark it as seen, ...
				distanceMap[neighbourIndex] = 1 + distanceMap[next->getIndex()]; //... enter its distance...
				open.push(*it); //... and add it to the open list.
			}
		}
	}

	_distanceMap[goalNode->getIndex()] = distanceMap;
}

//Contains duplicate code from _addDistanceMapEntry() (at least for now)
void MDDFactory::_addToHeatMap(GraphNode* startNode, GraphNode* endNode, int addedBounds) {
	if (_distanceMap.find(endNode->getIndex()) == _distanceMap.end()) {
		_addDistanceMapEntry(endNode);
	}
	int* oldDistanceMap = _distanceMap[endNode->getIndex()];

	int* newDistanceMap = new int[startNode->getGraphSize()];
	for (int i = 0; i < startNode->getGraphSize(); ++i) {
		newDistanceMap[i] = INT_MAX; // initialize distance map to INT_MAX
		                          // effectively means "unreachable node"
	}
	newDistanceMap[startNode->getIndex()] = 0;

	int goalDistance = oldDistanceMap[startNode->getIndex()];
	int maxCombinedDistance = goalDistance + addedBounds;

	queue<GraphNode*> open; //stores pointers to border nodes in FIFO manner
	bool seen[startNode->getGraphSize()]; //stores shared pointers to all nodes seen already
	                                     //by the end of the algorithm will have all reachable nodes of the graph
	for (int i = 0; i < startNode->getGraphSize(); ++i) {
		seen[i] = 0; //initialize seen; important, omitting leads to bugs!
	}

	open.push(startNode); //start with goal node (and work backwards through the map)
	seen[startNode->getIndex()] = true;
	_heatMap[startNode->getIndex()] += 1.0;

	GraphNode* next;
	vector<GraphNode*> neighbours;
	int neighbourIndex;
	int combinedDistance;

	while (!open.empty()) {

		next = open.front();
		open.pop();

		//expand
		neighbours = next->getNeighbours();
		for (auto it = neighbours.begin(); it != neighbours.end(); ++it) { //for every neighbour...
			neighbourIndex = (*it)->getIndex();
			combinedDistance = newDistanceMap[next->getIndex()] + 1 + oldDistanceMap[neighbourIndex]; //the +1 is because we want the value for next's neighbours whose newDistanceMap value is 1 greater that next's
			if (!seen[neighbourIndex] && (combinedDistance <= maxCombinedDistance)) { //... that we haven't handled already in the past AND that is not too far off the optimal paths...
				seen[neighbourIndex] = true; //... mark it as seen, ...
				newDistanceMap[neighbourIndex] = 1 + newDistanceMap[next->getIndex()]; //... enter its distance...
				open.push(*it); //... and add it to the open list...

				//... before adding its value to the heat map.
				int deviationFromOptimalPath = combinedDistance - goalDistance;
				if (deviationFromOptimalPath <= 0) { //shouldn't ever be '< 0' but at best == 0 for tiles on optimal paths
					_heatMap[neighbourIndex] += 1.0;
				} else {
					_heatMap[neighbourIndex] += 1.0 / pow(2,deviationFromOptimalPath);
				}

			}
		}
	}
}

const double * MDDFactory::_getHeatMap() {
	return _heatMap;
}


// Return 0 if no path exists
JMDD* MDDFactory::_makeJMDDFromDistanceMap(int agentID, GraphNode* startNode, GraphNode* endNode, int cost) {

	//check for impossibility special case: from goal to goal in 1 time step
	if (cost == 1 && startNode == endNode) return 0;

	//check for entry in distance map and if need be make one
	if (_distanceMap.find(endNode->getIndex()) == _distanceMap.end()) {
		//std::cout << "new distanceMap entry" << std::endl;
		_addDistanceMapEntry(endNode);
	}
	int* distanceMap = _distanceMap[endNode->getIndex()];

	//check for impossibility normal case: from start to goal in less than optimal distance
	if (distanceMap[startNode->getIndex()] > cost) return 0;

	// make head node
	JMDDNode* head = new JMDDNode();
	head->graphNodes[agentID] = std::vector<int>(1,startNode->getIndex());

	map<int,JMDDNode*> structure[cost+1]; //array[ layer=costLeft ] = map< tile, MDDNodeRepresentingThatTileAtAThatLayer > >

	structure[cost][startNode->getIndex()] = head;

	for (int costLeft = cost; costLeft > 0; --costLeft) { //for every layer of the MDD (working from the head (highest layer) down to tail (0))
		for (auto layer_it = structure[costLeft].begin(); layer_it != structure[costLeft].end(); ++layer_it) { //continue expanding every node of current layer

			GraphNode* currentlyExpandingNode = startNode->getNodeFromGraph((*layer_it).second->graphNodes[agentID][0]);

			//get neighbours, both 0-8 graph neighbours and itself (because wait action)
			vector<GraphNode*> neighbours = currentlyExpandingNode->getNeighbours();
			if (costLeft > 1) {//Clause 1: Don't wait as your last action (elaborated in block comment below)
				neighbours.push_back(currentlyExpandingNode);
			}

			//for every neighbour that the expanding node has ...
			for (auto neighbour_it = neighbours.begin(); neighbour_it != neighbours.end(); ++neighbour_it) {

				// ... (and that is not the goal node even though we're filling second-to-last layer) ...
				if (costLeft == 2 && (*neighbour_it)->getIndex() == endNode->getIndex()) {
					//std::cout << "				(clause 2: no go to goal)" << std::endl;
					continue; //Clause 2 (elaborated in block comment below)
				}

				//... check if it is conform with our optimal goal distance requirements
				if (distanceMap[(*neighbour_it)->getIndex()] <= costLeft-1) {

					//... and if they are see if we have a MDD node for them already
					if (structure[costLeft-1].find((*neighbour_it)->getIndex()) != structure[costLeft-1].end()) {

						// if so, just add the link (in both directions)
						(*layer_it).second->successors.push_back(structure[costLeft-1][(*neighbour_it)->getIndex()]); //forward link
						structure[costLeft-1][(*neighbour_it)->getIndex()]->predecessors.push_back((*layer_it).second); //backward link

					} else {

						//if not, make a new MDDNode
						JMDDNode* newNode = new JMDDNode();
						newNode->graphNodes[agentID] = std::vector<int>(1,(*neighbour_it)->getIndex());

						//link it
						(*layer_it).second->successors.push_back(newNode);
						newNode->predecessors.push_back((*layer_it).second);

						//add it to the structure
						structure[costLeft-1][(*neighbour_it)->getIndex()] = newNode;

					}
				}
			}
		}
	}

	/*  Here I previously sanitized the MDD to remove paths that end by waiting in the goal.
	 *
	 *  The paths represented may not include paths where the last action is waiting at the goal
	 *  because then the actual path length is shorter than demanded (waiting after we are done adds no cost; we are done)
	 *
	 *  At first I thought this needed to be done by sanitizing recursively after the fact but assuming that the goal tile
	 *  has neighbours it is always possible to end a path by getting from the goal to the goal in 2 steps without waiting.
	 *  (and if the goal has no neighbours we have other problems. then the MDD is empty or has cost 0)
	 *  So it is always possible to wait at the goal except in the final 2 steps in which case it is always illegal.
	 *
	 *  In addition, the second-to-last step may not go to the goal node, because going from goal to goal can not happen
	 *  with precisely 1 cost withput waiting which again would be illegal.
	 *
	 *  I think the best way to formulate that as clauses is
	 *  1. Don't wait as your last action (this holds true for all nodes, not just goal node)
	 *  	(not actually necessary if the second-to-last node can't be the goal but does avoid unnecessary nodes bein generated so I kept it)
	 *  2. Don't go to the goal tile as your second-to-last action (either by waiting there or going there from somewhere else)
	 */

	return new JMDD(head, structure[0][endNode->getIndex()], cost); //added the tail parameter later, I think it checks out like that
}



//returns distance, where -1 means "no entry" (and INT_MAX means unreachable)
int MDDFactory::_getDistanceTo(int startTile, int endTile) {
	if (_distanceMap.find(endTile) == _distanceMap.end()) {
		return -1;
	}
	return _distanceMap[endTile][startTile];
}

int MDDFactory::_getDistanceTo(GraphNode* startNode, int endTile) {
	return _getDistanceTo(startNode->getIndex(), endTile);
}

int MDDFactory::_getDistanceTo(int startTile, GraphNode* endNode) {
	return _getDistanceTo(startTile, endNode->getIndex());
}

int MDDFactory::_getDistanceTo(GraphNode* startNode, GraphNode* endNode) {
	return _getDistanceTo(startNode->getIndex(), endNode->getIndex());
}


bool MDDFactory::_isJointTransitionConflicting(std::vector<int> &oA, std::vector<int> &dA, std::vector<int> &oB, std::vector<int> &dB) {
	for (auto it_oA = oA.begin(); it_oA != oA.end(); ++it_oA) {
		for (auto it_dA = dA.begin(); it_dA != dA.end(); ++it_dA) {
			for (auto it_oB = oB.begin(); it_oB != oB.end(); ++it_oB) {
				for (auto it_dB = dB.begin(); it_dB != dB.end(); ++it_dB) { //That... That can't be good... Do I just hope that most of the time we'll find a conflict free pair early?

					//for every possible pair of explicit transitions
					if (!ICTS::_sourceNode->isTransitionPairConflicting(*it_oA, *it_dA, *it_oB, *it_dB)) {
						return false;
					}

				}
			}
		}
	}
	return true;
}


//joins two single-agent MDDs; NEVER ACTUALLY USED IN THE END
JMDD* MDDFactory::_joinMDD(JMDD* firstMDD, JMDD* secondMDD) {
	int firstCost = firstMDD->getCost();
	int secondCost = secondMDD->getCost();
	int cost = (firstCost >= secondCost) ? firstCost : secondCost;

	JMDDNode* tail = 0;

	//variables to keep track of where we are and what's left to process
	set<JMDDNode*>* currentLayer_first;
	set<JMDDNode*>* nextLayer_first;
	set<JMDDNode*>* currentLayer_second;
	set<JMDDNode*>* nextLayer_second;

	currentLayer_first = new set<JMDDNode*>;
	currentLayer_first->insert(firstMDD->getHead());
	nextLayer_first = new set<JMDDNode*>;
	currentLayer_second = new set<JMDDNode*>;
	currentLayer_second->insert(secondMDD->getHead());
	nextLayer_second = new set<JMDDNode*>;

	map<pair<JMDDNode*,JMDDNode*>,JMDDNode*> joiningTable; //saves which pair of MDDNodes (from firstMDD and secondMDD respectively)
	                                                              //were joint into which MDDNode of the new MDD

	vector<int> agents_first; //get agents represented by MDDs so we can check new combinations of transitions for conflicts and avoid re-checking old ones
	vector<int> agents_second;
	for (auto it_agents = firstMDD->getHead()->graphNodes.begin(); it_agents != firstMDD->getHead()->graphNodes.end(); ++it_agents) {
		agents_first.push_back(it_agents->first);
	}
	for (auto it_agents = secondMDD->getHead()->graphNodes.begin(); it_agents != secondMDD->getHead()->graphNodes.end(); ++it_agents) {
		agents_second.push_back(it_agents->first);
	}

	//std:.cout << "Start Loop" << std::endl;
	for (int layer = cost; layer >= 0; --layer) {
		//std::cout << "Layer " << std::to_string(layer) << std::endl;

		for (auto it_first = currentLayer_first->begin(); it_first != currentLayer_first->end(); ++it_first) {
			for (auto it_second = currentLayer_second->begin(); it_second != currentLayer_second->end(); ++it_second) {
				//for every pair of MDD Nodes in this layer

				//std::cout << "	Joint Node " << std::endl;

				//make new MDDNode
				JMDDNode* newNode = new JMDDNode;
				//with both joining node's agent-to-place assignments
				for (auto it_first_nodes = (*it_first)->graphNodes.begin(); it_first_nodes != (*it_first)->graphNodes.end(); ++it_first_nodes) {
					newNode->graphNodes[it_first_nodes->first] = it_first_nodes->second;
				}
				for (auto it_second_nodes = (*it_second)->graphNodes.begin(); it_second_nodes != (*it_second)->graphNodes.end(); ++it_second_nodes) {
					newNode->graphNodes[it_second_nodes->first] = it_second_nodes->second;
				}

				//with proper linking to its predecessors as dictated by the joining table

				std::vector<JMDDNode*> first_predecessors; //relevant predecessor list; may be only itself if we're padding a shorter MDD to join it to a longer one
				if (layer >= cost - firstCost) { //if we are not yet in padding for that joining MDD
					first_predecessors = (*it_first)->predecessors;
				} else {
					first_predecessors = std::vector<JMDDNode*>(1,*it_first);
				}

				std::vector<JMDDNode*> second_predecessors; //relevant predecessor list; may be only itself if we're padding a shorter MDD to join it to a longer one
				if (layer >= cost - secondCost) { //if we are not yet in padding for that joining MDD
					second_predecessors = (*it_second)->predecessors;
				} else {
					second_predecessors = std::vector<JMDDNode*>(1,*it_second);
				}
				for (auto it_first_predecessors = first_predecessors.begin(); it_first_predecessors != first_predecessors.end(); ++it_first_predecessors) {
					for (auto it_second_predecessors = second_predecessors.begin(); it_second_predecessors != second_predecessors.end(); ++it_second_predecessors) {
						//for every pair of predecessors
						//std:.cout << "			new predecessor pair" << std::endl;

						//that has been joint into a node (should always have happened unless it was a sameTileConflict)
						if (joiningTable.find(make_pair(*it_first_predecessors, *it_second_predecessors)) == joiningTable.end()) {
							//std:.cout << "				no entry so this predecessor pair is probably a sameTileConflict" << std::endl;
							continue;
						}

						JMDDNode* currentlyConsideredPredecessor = joiningTable[make_pair(*it_first_predecessors, *it_second_predecessors)];


						//that does not represent a crossover conflict (agents going from a to b and b to a in same time step)
						bool opposingTransitionConflict = false;



						//vector<int> agents; //should be unnecessary for once to calculate every time and otherwise to combine both joining agents into one when you would want to avoid rechecking connections that lie completely within one of the joining MDDs (and are not a new combination introduced by joining)
						//for (auto it_predecessor_agents = currentlyConsideredPredecessor->graphNodes.begin(); it_predecessor_agents != currentlyConsideredPredecessor->graphNodes.end(); ++it_predecessor_agents) {
						//	agents.push_back(it_predecessor_agents->first);
						//}
						//std::cout << "Checking for conflicts while joining MDDs" << std::endl;
						for (int agentIndexA = 0; !opposingTransitionConflict && agentIndexA < (int) agents_first.size(); ++agentIndexA) {
							for (int agentIndexB = 0; !opposingTransitionConflict && agentIndexB < (int) agents_second.size(); ++agentIndexB) { //Way too many loops, that can't be the best way
								if (_isJointTransitionConflicting(
										currentlyConsideredPredecessor->graphNodes[agents_first[agentIndexA]],
										newNode->graphNodes[agents_first[agentIndexA]],
										currentlyConsideredPredecessor->graphNodes[agents_second[agentIndexB]],
										newNode->graphNodes[agents_second[agentIndexB]])) {
									opposingTransitionConflict = true;
								}
							}
						}


						if (opposingTransitionConflict) {
							//std:.cout << "				predecessor represents crossover conflict" << std::endl;
							continue;
						}
						//end of crossover conflict detection


						//link to which node they were joint to
						newNode->predecessors.push_back(currentlyConsideredPredecessor);
						currentlyConsideredPredecessor->successors.push_back(newNode);
					}
				}

				//and if it has genuine predecessors (without crossover conflicts) or else is the head which needs none
				if (layer == cost || newNode->predecessors.size() > 0) {

					//std::cout << "		genuine node" << std::endl;

					//enter it into the joining table so we can later link its successor nodes to it
					joiningTable[make_pair(*it_first, *it_second)] = newNode;

					//and fill next layer of input nodes to be processed
					if (layer > cost - firstCost) { //if there is still a next layer in that joining MDD
						for (auto it_first_successors = (*it_first)->successors.begin(); it_first_successors != (*it_first)->successors.end(); ++it_first_successors) {
							nextLayer_first->insert(*it_first_successors);
							//std:.cout << "		first nextlayer now has " << nextLayer_first->size() << " entries" << std::endl;
						}
					}
					if (layer > cost - secondCost) { //if there is still a next layer in that joining MDD
						for (auto it_second_successors = (*it_second)->successors.begin(); it_second_successors != (*it_second)->successors.end(); ++it_second_successors) {
							nextLayer_second->insert(*it_second_successors);
							//std:.cout << "		second nextlayer now has " << nextLayer_second->size() << " entries" << std::endl;
						}
					}

					if (layer == 0) { //added later for getting the tail. I think that checks out like that.
						tail = newNode;
					}

				} else {

					//std::cout << "		false node" << std::endl;

					//nevermind, not a genuine node because due to crossover conflicts it can not be reached
					delete newNode;

				}
			}

		}//end of iterating through all pairs of MDDNodes in current input layer

		//see if the next layer to process is empty (e.g. there is no joint path possible)
		if (layer > 0 && ( //not the last layer (which needs no successors in any case)
				( (layer > cost - firstCost) && nextLayer_first->size() == 0 ) //and the first layer has no successors even though it needed some
				||
				( (layer > cost - secondCost) && nextLayer_second->size() == 0 ) //or the second layer has no successors though needing some
				)) {
			//std::cout << "	END OF THE LINE" << std::endl;
			delete currentLayer_first;
			delete currentLayer_second;
			delete nextLayer_first;
			delete nextLayer_second;
			return 0;
		}

		//advance to next layer, delete used one
		//std:.cout << "	advance to next layer, delete used one" << std::endl;
		if (layer > cost - firstCost) { //if there is still a next layer in that joining MDD
			delete currentLayer_first;
			currentLayer_first = nextLayer_first;
			nextLayer_first = new set<JMDDNode*>;
		}

		if (layer > cost - secondCost) { //if there is still a next layer in that joining MDD
			delete currentLayer_second;
			currentLayer_second = nextLayer_second;
			nextLayer_second = new set<JMDDNode*>;
		}

	}

	//free variables
	//std::cout << "free variables" << std::endl;
	delete currentLayer_first;
	delete currentLayer_second;
	delete nextLayer_first;
	delete nextLayer_second;

	return new JMDD(joiningTable[make_pair(firstMDD->getHead(), secondMDD->getHead())], tail, cost);

}


//single-agent version of MDD creation.
//duplicate code, it's bad I know
//contains backwards links for MDD nodes
MDD* MDDFactory::_makeMDDFromDistanceMap(GraphNode* startNode, GraphNode* endNode, int cost) {

	//check for impossibility special case: from goal to goal in 1 time step
	if (cost == 1 && startNode == endNode) return 0;

	//check for entry in distance map and if need be make one
	if (_distanceMap.find(endNode->getIndex()) == _distanceMap.end()) {
		_addDistanceMapEntry(endNode);
	}
	int* distanceMap = _distanceMap[endNode->getIndex()];

	//check for impossibility normal case: from start to goal in less than optimal distance
	if (distanceMap[startNode->getIndex()] > cost) {
		return 0;
	}

	// make head node
	EMDDNode* head = new EMDDNode(startNode->getIndex());

	map<int,EMDDNode*> structure[cost+1]; //array[ layer=costLeft ] = map< tile, MDDNodeRepresentingThatTileAtAThatLayer > >

	structure[cost][startNode->getIndex()] = head;

	for (int costLeft = cost; costLeft > 0; --costLeft) { //for every layer of the MDD (working from the head (highest layer) down to tail (0))
		for (auto layer_it = structure[costLeft].begin(); layer_it != structure[costLeft].end(); ++layer_it) { //continue expanding every node of current layer
			//std::cout << "		Entry " << (*layer_it).second->graphNode->toString() << " (" << (*layer_it).second->graphNode->getIndex() << ")" << std::endl;

			// The graphnode we are currently at and whose neighbours we may want to be at in the next time step
			GraphNode* currentlyExpandingNode = startNode->getNodeFromGraph((*layer_it).second->getGraphNode());

			//get neighbours, both 0-8 graph neighbours and itself (because wait action)
			vector<GraphNode*> neighbours = currentlyExpandingNode->getNeighbours();
			if (costLeft > 1) {//Clause 1: Don't wait as your last action
				neighbours.push_back(currentlyExpandingNode);
			}

			//for every neighbour that the expanding node has ...
			for (auto neighbour_it = neighbours.begin(); neighbour_it != neighbours.end(); ++neighbour_it) {

				// ... (and that is not the goal node even though we're filling second-to-last layer) ...
				if (costLeft == 2 && (*neighbour_it)->getIndex() == endNode->getIndex()) {
					continue; //Clause 2
				}

				//... check if it is conform with our optimal goal distance requirements
				if (distanceMap[(*neighbour_it)->getIndex()] <= costLeft-1) {

					//... and if they are see if we have a MDD node for them already
					if (structure[costLeft-1].find((*neighbour_it)->getIndex()) != structure[costLeft-1].end()) {

						// if so, just add the link (in both directions)
						(*layer_it).second->successors.push_back(structure[costLeft-1][(*neighbour_it)->getIndex()]); //forward link
						structure[costLeft-1][(*neighbour_it)->getIndex()]->predecessors.push_back((*layer_it).second); //backward link

					} else {

						//if not, make a new MDDNode
						EMDDNode* newNode = new EMDDNode((*neighbour_it)->getIndex());

						//link it
						(*layer_it).second->successors.push_back(newNode);
						newNode->predecessors.push_back((*layer_it).second);


						//add it to the structure
						structure[costLeft-1][(*neighbour_it)->getIndex()] = newNode;

					}
				}
			}
		}
	}

	MDDNode* tail = structure[0].begin()->second;
	return new MDD(head, tail, cost);
}

MDD* MDDFactory::_copyMDDForAbstraction(MDD* original) {

	std::map<MDDNode*,AMDDNode*> identity; //stores which original MDD nodes are which copied ones.
	identity[original->getHead()] = new AMDDNode(original->getHead());

	std::set<MDDNode*> currentLayer;
	currentLayer.insert(original->getHead());
	std::set<MDDNode*> nextLayer;

	for (int i = 0; i <= original->getCost(); ++i) {
		for (auto layer_it = currentLayer.begin(); layer_it != currentLayer.end(); ++layer_it) {
			for (auto successor_it = (*layer_it)->successors.begin(); successor_it != (*layer_it)->successors.end(); ++successor_it) {
				nextLayer.insert(*successor_it);
			}
			AMDDNode* newNode = new AMDDNode(*layer_it); //make a copy containing the same graph nodes but no links yet
			identity[*layer_it] = newNode;
		}
		currentLayer.clear();
		currentLayer = std::move(nextLayer);
		nextLayer.clear();
	}

	//linking
	for (auto allNodesIterator = identity.begin(); allNodesIterator != identity.end(); ++allNodesIterator) {
		for (auto predecessorIterator = allNodesIterator->first->predecessors.begin(); predecessorIterator != allNodesIterator->first->predecessors.end(); ++predecessorIterator) {
			allNodesIterator->second->predecessors.push_back(identity[*predecessorIterator]);
			identity[*predecessorIterator]->successors.push_back(allNodesIterator->second);
		}
	}

	return new MDD(identity[original->getHead()], identity[original->getTail()], original->getCost());

}

MDDNode* MDDFactory::_mergeTwoMDDNodesInPlace(MDDNode* nodeA, MDDNode* nodeB) {
	set<int> tileUnion;
	set<MDDNode*> successorUnion;
	set<MDDNode*> predecessorUnion;

	AMDDNode* mergedNode = new AMDDNode();

	//tiles
	vector<int> nodeATiles = nodeA->getAllTiles();
	for (auto tileIterator = nodeATiles.begin(); tileIterator != nodeATiles.end(); ++tileIterator) {
		tileUnion.insert(*tileIterator);
	}
	vector<int> nodeBTiles = nodeB->getAllTiles();
	for (auto tileIterator = nodeBTiles.begin(); tileIterator != nodeBTiles.end(); ++tileIterator) {
		tileUnion.insert(*tileIterator);
	}

	//successors
	for (auto successorIterator = nodeA->successors.begin(); successorIterator != nodeA->successors.end(); ++successorIterator) {
		successorUnion.insert(*successorIterator);
	}
	for (auto successorIterator = nodeB->successors.begin(); successorIterator != nodeB->successors.end(); ++successorIterator) {
		successorUnion.insert(*successorIterator);
	}

	//predecessors
	for (auto predecessorIterator = nodeA->predecessors.begin(); predecessorIterator != nodeA->predecessors.end(); ++predecessorIterator) {
		predecessorUnion.insert(*predecessorIterator);
	}
	for (auto predecessorIterator = nodeB->predecessors.begin(); predecessorIterator != nodeB->predecessors.end(); ++predecessorIterator) {
		predecessorUnion.insert(*predecessorIterator);
	}

	//add to node
	for (auto tileIterator = tileUnion.begin(); tileIterator != tileUnion.end(); ++tileIterator) {
		mergedNode->addTile(*tileIterator);
	}
	for (auto successorIterator = successorUnion.begin(); successorIterator != successorUnion.end(); ++successorIterator) {
		//sever old connections
		auto link = std::find((*successorIterator)->predecessors.begin(), (*successorIterator)->predecessors.end(), nodeA);
		if (link != (*successorIterator)->predecessors.end()) {
			(*successorIterator)->predecessors.erase(link);
		}
		link = std::find((*successorIterator)->predecessors.begin(), (*successorIterator)->predecessors.end(), nodeB);
		if (link != (*successorIterator)->predecessors.end()) {
			(*successorIterator)->predecessors.erase(link);
		}

		//add new connections
		(*successorIterator)->predecessors.push_back(mergedNode);
		mergedNode->successors.push_back(*successorIterator);
	}
	for (auto predecessorIterator = predecessorUnion.begin(); predecessorIterator != predecessorUnion.end(); ++predecessorIterator) {
		//sever old connections
		auto link = std::find((*predecessorIterator)->successors.begin(), (*predecessorIterator)->successors.end(), nodeA);
		if (link != (*predecessorIterator)->successors.end()) {
			(*predecessorIterator)->successors.erase(link);
		}
		link = std::find((*predecessorIterator)->successors.begin(), (*predecessorIterator)->successors.end(), nodeB);
		if (link != (*predecessorIterator)->successors.end()) {
			(*predecessorIterator)->successors.erase(link);
		}

		//add new connections
		(*predecessorIterator)->successors.push_back(mergedNode);
		mergedNode->predecessors.push_back(*predecessorIterator);
	}
	//delete old Nodes
	delete nodeA;
	delete nodeB;
	//in case it is needed for other reasons, but merging is already done
	return mergedNode;

}

pair<MDDNode*, MDDNode*> MDDFactory::_refineANodeBySplittingOffOneGraphNode(AMDDNode* node, int graphNode) {
	if (!node->containsTile(graphNode)) return make_pair(nullptr, nullptr);
	if (node->getTileCount() <= 1) return make_pair(node, nullptr);

	AMDDNode* recipientNode = new AMDDNode(graphNode);
	AMDDNode* donorNode = new AMDDNode(node);
	donorNode->removeTile(graphNode);

	set<int> neighbours_recipient;
	set<int> neighbours_donor;

	//transitions to and from this node and the new node are valid if and only if they are to previous predecessors and successors who still contain at least one graph-neighbour of at least one of our graph nodes (=just 'int graphNode' for the new and all but that for the old one).
	vector<GraphNode*> neighbours_tmp = ICTS::_sourceNode->getNodeFromGraph(graphNode)->getNeighbours();

	for (auto neighbourIterator = neighbours_tmp.begin(); neighbourIterator != neighbours_tmp.end(); ++neighbourIterator) {
		neighbours_recipient.insert((*neighbourIterator)->getIndex());
	}
	neighbours_recipient.insert(graphNode); //don't forget, a node should be its own neighbour (even though I did not write that into getNeighbours() and not it's kind of too late)

	vector<int> remainingTiles = donorNode->getAllTiles();
	for (auto tileIterator = remainingTiles.begin(); tileIterator != remainingTiles.end(); ++tileIterator) {
		neighbours_tmp = ICTS::_sourceNode->getNodeFromGraph(*tileIterator)->getNeighbours();
		for (auto neighbourIterator = neighbours_tmp.begin(); neighbourIterator != neighbours_tmp.end(); ++neighbourIterator) {
			neighbours_donor.insert((*neighbourIterator)->getIndex());
		}
		neighbours_donor.insert(*tileIterator);
	}


	vector<MDDNode*> originalPredecessors = node->predecessors;
	vector<MDDNode*> originalSuccessors = node->successors;

	//make links to predecessors
	for (auto predecessorIterator = originalPredecessors.begin(); predecessorIterator != originalPredecessors.end(); ++predecessorIterator) {
		bool validForDonor = false;
		bool validForRecipient = false;
		vector<int> predecessorTiles = (*predecessorIterator)->getAllTiles();
		for (auto predecessorTileIterator = predecessorTiles.begin(); predecessorTileIterator != predecessorTiles.end(); ++predecessorTileIterator) {
			//for any tile in the predecessor, see if it is a neighbour of the old or new node
			if (neighbours_recipient.find(*predecessorTileIterator) != neighbours_recipient.end()) {
				validForRecipient = true;
			}
			if (neighbours_donor.find(*predecessorTileIterator) != neighbours_donor.end()) {
				validForDonor = true;
			}
			if (validForDonor && validForRecipient) break; //nothing left to prove
		}
		if (validForRecipient) {
			//link to the new node
			recipientNode->predecessors.push_back(*predecessorIterator);
			(*predecessorIterator)->successors.push_back(recipientNode);
		}
		if (validForDonor) {
			//link to the new node
			donorNode->predecessors.push_back(*predecessorIterator);
			(*predecessorIterator)->successors.push_back(donorNode);
		}
		if (true) { //in any case
			//sever connection to old node
			auto link = std::find((*predecessorIterator)->successors.begin(), (*predecessorIterator)->successors.end(), node);
			if (link != (*predecessorIterator)->successors.end()) {
				(*predecessorIterator)->successors.erase(link);
			}

			//NOW HERE'S AND IMPORANT EXCEPTION: In the interest of re-refinement we need to know which old nodes had which predecessors before being refined. There is no harm (except a bit more memory) in keeping the old nodes linked to the new ones since no link leads TO them.
			/*
			link = std::find(node->predecessors.begin(), node->predecessors.end(), (*predecessorIterator));
			if (link != node->predecessors.end()) {
				node->predecessors.erase(link);
			}
			*/
		}
	}

	if (recipientNode->predecessors.size() <= 0) {
		//std::cout << "DISQUALIFIED 01" << std::endl;
		//no predecessors -> no node!
		delete recipientNode;
		recipientNode = nullptr;
	}
	if (donorNode->predecessors.size() <= 0) {
		//std::cout << "DISQUALIFIED 02" << std::endl;
		//no predecessors -> no node!
		delete donorNode;
		donorNode = nullptr;
	}

	//make links to successors
	for (auto successorIterator = originalSuccessors.begin(); successorIterator != originalSuccessors.end(); ++successorIterator) {
		bool validForDonor = false;
		bool validForRecipient = false;
		vector<int> predecessorTiles = (*successorIterator)->getAllTiles();
		for (auto predecessorTileIterator = predecessorTiles.begin(); predecessorTileIterator != predecessorTiles.end(); ++predecessorTileIterator) {
			//for any tile in the successors, see if it is a neighbour of the old or new node
			if (recipientNode != nullptr) {
				if (neighbours_recipient.find(*predecessorTileIterator) != neighbours_recipient.end()) {
					validForRecipient = true;
				}
			}
			if (donorNode != nullptr) {
				if (neighbours_donor.find(*predecessorTileIterator) != neighbours_donor.end()) {
					validForDonor = true;
				}
			}
			if (validForDonor && validForRecipient) break; //nothing left to prove
		}
		if (recipientNode != nullptr && validForRecipient) {
			//link to the new node
			recipientNode->successors.push_back(*successorIterator);
			(*successorIterator)->predecessors.push_back(recipientNode);
		}
		if (donorNode != nullptr && validForDonor) {
			//link to the new node
			donorNode->successors.push_back(*successorIterator);
			(*successorIterator)->predecessors.push_back(donorNode);
		}
		if (true) { //in any case
			//sever connection to old node
			auto link = std::find((*successorIterator)->predecessors.begin(), (*successorIterator)->predecessors.end(), node);
			if (link != (*successorIterator)->predecessors.end()) {
				(*successorIterator)->predecessors.erase(link);
			}
			link = std::find(node->successors.begin(), node->successors.end(), (*successorIterator));
			if (link != node->successors.end()) {
				node->successors.erase(link);
			}
		}
	}

	if (recipientNode != nullptr && recipientNode->successors.size() <= 0) {
		//std::cout << "DISQUALIFIED 03" << std::endl;
		//no successors -> no node!

		//sever connections made previously; a bit inefficient to make connections only to delete them later but this is actually a pretty rare case.
		for (auto predecessorIterator = recipientNode->predecessors.begin(); predecessorIterator != recipientNode->predecessors.end(); ++predecessorIterator) {
			auto link = std::find((*predecessorIterator)->successors.begin(), (*predecessorIterator)->successors.end(), recipientNode);
			if (link != (*predecessorIterator)->successors.end()) {
				(*predecessorIterator)->successors.erase(link);
			}
		}

		delete recipientNode;
		recipientNode = nullptr;
	}
	if (donorNode != nullptr && donorNode->successors.size() <= 0) {
		//std::cout << "DISQUALIFIED 04" << std::endl;
		//no successors -> no node!

		//sever connections made previously; a bit inefficient to make connections only to delete them later but this is actually a pretty rare case.
		for (auto predecessorIterator = donorNode->predecessors.begin(); predecessorIterator != donorNode->predecessors.end(); ++predecessorIterator) {
			auto link = std::find((*predecessorIterator)->successors.begin(), (*predecessorIterator)->successors.end(), donorNode);
			if (link != (*predecessorIterator)->successors.end()) {
				(*predecessorIterator)->successors.erase(link);
			}
		}

		delete donorNode;
		donorNode = nullptr;
	}

	//delete node; //This is the important part; May I do this? Because if the same address gets reassigned I'll be in trouble.

	return make_pair((MDDNode*) recipientNode, (MDDNode*) donorNode);

}

void MDDFactory::_modifyMDDToMergeWithHeatMapThreshold(MDD* mdd, double heatThreshold) {
	vector<MDDNode*> mergeCandidates;

	std::set<MDDNode*> currentLayer;
	currentLayer.insert(mdd->getHead());
	std::set<MDDNode*> nextLayer;

	for (int i = 0; i <= mdd->getCost(); ++i) {
		for (auto layer_it = currentLayer.begin(); layer_it != currentLayer.end(); ++layer_it) {
			for (auto successor_it = (*layer_it)->successors.begin(); successor_it != (*layer_it)->successors.end(); ++successor_it) {
				nextLayer.insert(*successor_it);
			}
			vector<int> tiles = (*layer_it)->getAllTiles();
			for (auto tileIterator = tiles.begin(); tileIterator != tiles.end(); ++tileIterator) {
				if (_heatMap[*tileIterator] < heatThreshold) {
					mergeCandidates.push_back(*layer_it);
					break;
				}
			}
		}
		while (mergeCandidates.size() >= 2) {
			mergeCandidates.push_back(_mergeTwoMDDNodesInPlace(mergeCandidates[0], mergeCandidates[1]));
			mergeCandidates.erase(mergeCandidates.begin()); //remove element [0]
			mergeCandidates.erase(mergeCandidates.begin()); //remove new element [0] which before was element [1];
		}
		mergeCandidates.clear();
		currentLayer.clear();
		currentLayer = std::move(nextLayer);
		nextLayer.clear();
	}

}


void MDDFactory::_freeMemory() {
	for (auto iterator = _distanceMap.begin(); iterator != _distanceMap.end(); ++iterator) {
		delete[] iterator->second;
	}
	delete[] _heatMap;
}
