/*
 * ExperimentGenerator.h
 *
 * Made to generate random experiments of mutiple agents traversing a given map concurrently.
 *
 *  Created on: 11.01.2018
 *      Author: Simon
 */

#ifndef EXPERIMENTGENERATOR_H_
#define EXPERIMENTGENERATOR_H_

#include <string>
#include <tuple>
#include <utility>
#include <vector>

using std::vector;
using std::tuple;
using std::pair;
using std::string;

class ExperimentGenerator {

public:
	ExperimentGenerator() {} //don't actually instantiate this class.

	/**
	 * Generates completely random sets of start and end point on the map until numberOfAgents mutually reachable pairs are found.
	 * These experiments are then written to a file as given in fileName.
	 * Will free any previously loaded map and load mapName into MaGraphNode.
	 *
	 * It is ensured that no two agents start or end on the same tile.
	 * Other than that there is no guarantee that the joint problem for all agents will be solvable but for relatively open maps it is exceedingly likely.
	 */
	static void _generateRandomSetOfSizeOnMap(int numberOfAgents, string mapName, string fileName);

	/**
	 * Generates a map with cells randomly being assigned as impassable with given chance
	 * If called with optional parameter as true will determind each cell independently leading to roughly the percentage
	 * If called without optional parameter calculates number of obstructed cells needed and distributes them randomly.
	 */
	static void _generateGridWithRandomObstacles(int width, int height, double obstacleChance, string fileName, bool withVariance = false);

	/**
	 * Creates the first set of benchmarks I used for my tests in the thesis
	 * Designed to mimic tests performed in the original ICTS paper
	 */
	static void _generateSetForSimonsThesis_01_8x8grids();

	/**
	 * Creates the second set of benchmarks I used for my tests in the thesis
	 * Designed to mimic tests performed in the original ICTS paper
	 */
	static void _generateSetForSimonsThesis_02_32x32grids();

	/**
	 * Creates the third set of benchmarks I used for my tests in the thesis
	 * Designed to mimic tests performed in the original ICTS paper
	 */
	static void _generateSetForSimonsThesis_03_dragonagemaps();

	/**
	 * Creates a set of benchmarks not used for my thesis
	 * Designed to mimic tests performed in the original ICTS paper
	 */
	static void _generateSetForSimonsThesis_extra_moonglade();

private:
	/**
	 * Writes a set of experiments to a file.
	 * param is a vector whose size is the number of experiments (/agents that could be run concurrently) in the file
	 *   and whose elements are vectors whose size is either 4 (startX, startY, goalX, goalY) or 5 ([as before], optimal Distance)
	 *   When no fifth entry is given 0 will be written as goal distance.
	 */
	static void _writeExperimentsToFile(vector<vector<int>>& experiments, string& fileName, string& mapName, pair<int,int>& mapSize);

	/**
	 * Uses the MapGraphNode class to figure out optimal distance between coordinates (on the map previously loaded via MapGraphNode::_loadMap() )
	 */
	static int _getDistanceInMapGraph(int startX, int startY, int goalX, int goalY);


};


#endif /* EXPERIMENTGENERATOR_H_ */
