/*
 * ExperimentGenerator.cpp
 *
 *  Created on: 12.01.2018
 *      Author: Simon
 */

#include "ExperimentGenerator.h"

#include <cstdlib>
#include <fstream>
#include <iostream>
#include <stdexcept>

#include "MapGraphNode.h"

using std::cout;
using std::endl;

void ExperimentGenerator::_writeExperimentsToFile(vector<vector<int>>& experiments, string& fileName, string& mapName, pair<int, int>& mapSize) {
	//cout << "Begin writing to file..." << endl;

	std::ofstream experimentFile;
	experimentFile.open(fileName + ".map.scen");
	experimentFile << "version 1.0\n"; //file header


	for (int index = 0; index < (int) experiments.size(); ++index) {
		if (experiments[index].size() < 4 || experiments[index].size() > 5) throw std::runtime_error("ExperimentGenerator: Error while writing experiment file. start and end-position set was not a 4 or 5 integers (" + std::to_string(experiments[index].size()) + " given).");

		experimentFile << "0 "; //bucket number; unused (don't see what that would be useful for).
		experimentFile << mapName << " "; //map name
		experimentFile << mapSize.first << " " << mapSize.second << " "; //map size
		experimentFile << experiments[index][0] << " " << experiments[index][1] << " "; //starting coordinates
		experimentFile << experiments[index][2] << " " << experiments[index][3] << " "; //goal coordinates

		//optimal goal distance; optional (cause I don't see - at least yet - why I would save that to the file. Of course I could also just generate it here by calling GraphNode::getShortestPathTo(), but I'll need to have calculated it beforehand anyway to ensure the position is reachable.
		if (experiments[index].size() >= 5) {
			experimentFile << experiments[index][4];
		} else {
			experimentFile << "0";
		}

		experimentFile << "\n";

	}

	experimentFile.close();

	cout << "Successfully wrote scenario with " << experiments.size() << " agents to file " << fileName << ".map.scen" << endl;

}

int ExperimentGenerator::_getDistanceInMapGraph(int startX, int startY, int goalX, int goalY) {
	GraphNode* startNode = MapGraphNode::_getGraphNodeWithIndex(startX + MapGraphNode::_getMapWidth() * startY);
	return startNode->getDistanceTo(goalX + MapGraphNode::_getMapWidth() * goalY);
}

void ExperimentGenerator::_generateRandomSetOfSizeOnMap(int numberOfAgents, string mapName, string fileName) {
	MapGraphNode::_freeMap();
	MapGraphNode::_loadMap(mapName.c_str());
	MapGraphNode::_setAdjacency(false); //a later addition, so the problem is always also possible with 4-adjacency if needed (for comparison)

	vector<vector<int>> experiments;
	experiments.reserve(numberOfAgents);

	int mapWidth = MapGraphNode::_getMapWidth();
	int mapHeight = MapGraphNode::_getMapHeight();

	int failSafe = 0; //counts up; cancel iterations when it exceeds threshold;
	const int MAX_ITERATIONS = 1048576; //2^20

	int startX, startY, goalX, goalY;
	int goalDistance;
	bool validPointsFlag;

	while ((int) experiments.size() < numberOfAgents) {
		startX = rand() % mapWidth;
		startY = rand() % mapHeight;
		goalX = rand() % mapWidth;
		goalY = rand() % mapHeight;

		//caused problems before. you can move away from an unpassable start tile but then the reverse search will fail and be at a loss as to why and be stuck in an infinite loop
		validPointsFlag = ( (MapGraphNode::_isMapPassableAt(startX + mapWidth*startY)) && (MapGraphNode::_isMapPassableAt(goalX + mapWidth*goalY)) );

		//start checking it it starts or ends on the same tile as a previous iteration.
		if (validPointsFlag) {
			for (int index = 0; index < (int) experiments.size(); ++index) {
				if (experiments[index][0] == startX && experiments[index][1] == startY) {
					validPointsFlag = false;
					break;
				} else if (experiments[index][2] == goalX && experiments[index][3] == goalY) {
					validPointsFlag = false;
					break;
				}
			}
		}

		if (validPointsFlag) {
			goalDistance = _getDistanceInMapGraph(startX, startY, goalX, goalY);
			if (goalDistance >= 0) { //if goal can be reached from start
				experiments.push_back({startX, startY, goalX, goalY, goalDistance});
			}
		}


		if (failSafe++ > MAX_ITERATIONS) throw std::runtime_error("ExperimentGenerator:  Number of iterations to find valid start-goal pair exceeded threshold of " + std::to_string(MAX_ITERATIONS) + " iterations.");
	}

	pair<int,int> mapSize(MapGraphNode::_getMapWidth(), MapGraphNode::_getMapHeight());
	_writeExperimentsToFile(experiments, fileName, mapName, mapSize);

}


void ExperimentGenerator::_generateGridWithRandomObstacles(int width, int height, double obstacleChance, string fileName, bool withVariance) {
	//cout << "Begin writing to file..." << endl;

	std::ofstream mapFile;
	mapFile.open(fileName + ".map");
	mapFile << "type octile\n";
	mapFile << "height " << height << "\n";
	mapFile << "width " << width << "\n";
	mapFile << "map\n";

	int obstacles = 0;

	if (withVariance) {

		for (int y = 0; y < height; ++y) {
			for (int x = 0; x < width; ++x) {
				if ( ((double)rand() / RAND_MAX) < obstacleChance ) {
					mapFile << "T";
					++obstacles;
				} else {
					mapFile << ".";
				}

			}
			mapFile << "\n";
		}

	} else {

		int obstaclesNeeded = (int) (((width * height) * obstacleChance) + 0.5); //number of obstacles needed rounded to nearest integer

		bool obstructionMap[width][height];
		for(int y = 0; y < height; ++y){
			for(int x = 0; x < width; ++x){
				obstructionMap[x][y] = false;
			}
		}


		int failSafe = 0; //counts up; cancel iterations when it exceeds threshold;
		const int MAX_ITERATIONS = 1048576; //2^20

		int obsX, obsY;
		while (obstacles < obstaclesNeeded) {
			obsX = rand() % width;
			obsY = rand() % height;

			if (!obstructionMap[obsX][obsY]) {
				obstructionMap[obsX][obsY] = true;
				++obstacles;
			}

			if (failSafe++ > MAX_ITERATIONS) throw std::runtime_error("ExperimentGenerator:  Number of iterations to find valid obstacle position exceeded threshold of " + std::to_string(MAX_ITERATIONS) + " iterations.");
		}

		for (int y = 0; y < height; ++y) {
			for (int x = 0; x < width; ++x) {
				if ( obstructionMap[x][y] ) {
					mapFile << "T";
				} else {
					mapFile << ".";
				}
			}
			mapFile << "\n";
		}

	}

	mapFile.close();

	cout << "Successfully wrote " << width << "x" << height << " map with " << obstacles << " (" << ((100.0 * obstacles) / (width*height)) << "%) obstacles to " << fileName << ".map" << endl;

}









// ---------------------------------- Past here, very specific functions to generate not general benchmarks but mine specifically (want to keep the code, and better have that code here than as a comment block in the main method I reckon)
// warning: better set up directory structure for files to be written in beforehand, this really isn't meant and wasn't written for general use.

string padWithZeroes(string target, int length) {
	while ((int) target.length() < length) {
		target = "0" + target;
	}
	return target;
}

void ExperimentGenerator::_generateSetForSimonsThesis_01_8x8grids() {
		cout << "Start generating experiment files - set 01 - 8x8 grids." << endl;

		//first the map file
		ExperimentGenerator::_generateGridWithRandomObstacles(8, 8, 0.0, "benchmarks/8x8grid/8x8grid");

		//then the scenario files
		int i;
		for (int agentCount = 4; agentCount <= 16; agentCount += 2) {
			for (i = 1; i <= 100; ++i) {
				ExperimentGenerator::_generateRandomSetOfSizeOnMap(agentCount, "benchmarks/8x8grid/8x8grid.map", "benchmarks/8x8grid/" + std::to_string(agentCount) + "agents/8x8grid_" + std::to_string(agentCount) + "agents_" + padWithZeroes(std::to_string(i),3));
			}
			cout << "8x8 grid: Finished writing " << (i-1) << " files for " << agentCount << " agents." << endl;
		}
}

void ExperimentGenerator::_generateSetForSimonsThesis_02_32x32grids() {
	cout << "Start generating experiment files - set 02 - 32x32 grids." << endl;

	//first the map files
	ExperimentGenerator::_generateGridWithRandomObstacles(32, 32, 0.0, "benchmarks/32x32grid/0obstacles/32x32grid_0obstacles");
	ExperimentGenerator::_generateGridWithRandomObstacles(32, 32, 0.05, "benchmarks/32x32grid/5obstacles/32x32grid_5obstacles");
	ExperimentGenerator::_generateGridWithRandomObstacles(32, 32, 0.10, "benchmarks/32x32grid/10obstacles/32x32grid_10obstacles");
	ExperimentGenerator::_generateGridWithRandomObstacles(32, 32, 0.15, "benchmarks/32x32grid/15obstacles/32x32grid_15obstacles");
	ExperimentGenerator::_generateGridWithRandomObstacles(32, 32, 0.20, "benchmarks/32x32grid/20obstacles/32x32grid_20obstacles");
	ExperimentGenerator::_generateGridWithRandomObstacles(32, 32, 0.25, "benchmarks/32x32grid/25obstacles/32x32grid_25obstacles");


	//then the scenario files
	const int agentCount = 40;
	int i;
	for (int obstaclePercentage = 0; obstaclePercentage <= 25; obstaclePercentage += 5) {
		for (i = 1; i <= 100; ++i) {
			ExperimentGenerator::_generateRandomSetOfSizeOnMap(agentCount, "benchmarks/32x32grid/" + std::to_string(obstaclePercentage) + "obstacles/32x32grid_" + std::to_string(obstaclePercentage) + "obstacles.map", "benchmarks/32x32grid/" + std::to_string(obstaclePercentage) + "obstacles/32x32grid_" + std::to_string(obstaclePercentage) + "obstacles_" + padWithZeroes(std::to_string(i),3));
		}
		cout << "32x32 grid: Finished writing " << (i-1) << " files for " << obstaclePercentage << "% obstacles." << endl;
	}
}

void ExperimentGenerator::_generateSetForSimonsThesis_03_dragonagemaps() {
	cout << "Start generating experiment files - set 03 - dragon age origins maps." << endl;

	//map files by Nathan Sturtevant found at http://movingai.com/benchmarks/
	//have them in place beforehand

	//scenario files
	int i;
	string map;

	map = "den520d";
	for (int agentCount = 10; agentCount <= 50; agentCount += 10) {
		for (i = 1; i <= 100; ++i) {
			ExperimentGenerator::_generateRandomSetOfSizeOnMap(agentCount, "benchmarks/dragonage/" + map + "/" + map + ".map", "benchmarks/dragonage/" + map + "/" + std::to_string(agentCount) + "agents/" + map + "_" + std::to_string(agentCount) + "agents_" + padWithZeroes(std::to_string(i),3));
		}
		cout << "DAO - " + map + ": Finished writing " << (i-1) << " files for " << agentCount << " agents." << endl;
	}

	map = "ost003d";
	for (int agentCount = 10; agentCount <= 30; agentCount += 10) {
		for (i = 1; i <= 100; ++i) {
			ExperimentGenerator::_generateRandomSetOfSizeOnMap(agentCount, "benchmarks/dragonage/" + map + "/" + map + ".map", "benchmarks/dragonage/" + map + "/" + std::to_string(agentCount) + "agents/" + map + "_" + std::to_string(agentCount) + "agents_" + padWithZeroes(std::to_string(i),3));
		}
		cout << "DAO - " + map + ": Finished writing " << (i-1) << " files for " << agentCount << " agents." << endl;
	}

	map = "brc202d";
	for (int agentCount = 5; agentCount <= 20; agentCount += 5) {
		for (i = 1; i <= 100; ++i) {
			ExperimentGenerator::_generateRandomSetOfSizeOnMap(agentCount, "benchmarks/dragonage/" + map + "/" + map + ".map", "benchmarks/dragonage/" + map + "/" + std::to_string(agentCount) + "agents/" + map + "_" + std::to_string(agentCount) + "agents_" + padWithZeroes(std::to_string(i),3));
		}
		cout << "DAO - " + map + ": Finished writing " << (i-1) << " files for " << agentCount << " agents." << endl;
	}
}

void ExperimentGenerator::_generateSetForSimonsThesis_extra_moonglade() {
	cout << "Start generating experiment files - extra set - moonglade." << endl;
	int i;
	for (i = 1; i <= 100; ++i) {
		ExperimentGenerator::_generateRandomSetOfSizeOnMap(100, "benchmarks/moonglade/moonglade.map", "benchmarks/moonglade/moonglade_100agents_" + padWithZeroes(std::to_string(i),3));
	}
	cout << "Extra - moonglade: Finished writing " << (i-1) << " files." << endl;
}


