/*
 * Based on the utils/hash.h of the FastDownward Planner developed by the University of Basel
 */


#ifndef UTILS_HASH_H
#define UTILS_HASH_H

#include <cstddef>
#include <unordered_map>
#include <utility>
#include <vector>

namespace utils {
/*
  Hash a new value and combine it with an existing hash.

  This function should only be called from within this module.
*/
template<typename T>
inline void hash_combine(size_t &hash, const T &value) {
    std::hash<T> hasher;
    /*
      The combination of hash values is based on issue 6.18 in
      http://www.open-std.org/JTC1/SC22/WG21/docs/papers/2005/n1756.pdf.
      Boost combines hash values in the same way.
    */
    hash ^= hasher(value) + 0x9e3779b9 + (hash << 6) + (hash >> 2);
}

template<typename Sequence>
size_t hash_sequence(const Sequence &data, size_t length) {
    size_t hash = 0;
    for (size_t i = 0; i < length; ++i) {
        hash_combine(hash, data[i]);
    }
    return hash;
}

//An addition of mine (Simon)
template<typename T>
struct PointerResolutionHash {
	size_t operator()(const T* k) const {
		return std::hash<T>()(*k);
	}
};

//An addition of mine (Simon)
template<typename T>
struct PointerResolutionEqual {
	bool operator()(const T* k1, const T* k2) const {
		return (*k1) == (*k2);
	}
};

}

namespace std {
template<typename T>
struct hash<std::vector<T>> {
    size_t operator()(const std::vector<T> &vec) const {
        return utils::hash_sequence(vec, vec.size());
    }
};

template<typename TA, typename TB>
struct hash<std::pair<TA, TB>> {
    size_t operator()(const std::pair<TA, TB> &pair) const {
        size_t hash = 0;
        utils::hash_combine(hash, pair.first);
        utils::hash_combine(hash, pair.second);
        return hash;
    }
};
}

#endif
