/*
 * MapGraphNode.h
 *
 * Implementation of a GraphNode that uses a static 2-dimensional boolean array to represent a graph.
 *
 *  Created on: 15.10.2017
 *      Author: Simon
 */

#ifndef MAPGRAPHNODE_H_
#define MAPGRAPHNODE_H_

#include <string>
#include <vector>

#include "GraphNode.h"

class MapGraphNode : public GraphNode {

private:
	static bool* _map; //The map being presented as a graph. Only differentiates traversable (true) and obstructed (false) tiles.
	                   //size is _mapWidth * _mapHeight
	                   //enuerates all tiles of the map from left to right and top to bottom (like english text is read).
	static int _mapWidth; //width of the map;
	static int _mapHeight; //height of the map;
	static bool _eightAdjacency; //if false, use 4 adjacency; if true, use 8-adjacency.

	static MapGraphNode** _nodes;

public:
	static void _loadMap(const char* filename); //Reads a map to fill _map and set the other static variables (width, height, adjacency).
	static void _loadMap(); //Empty test map
	static void _freeMap(); //Releases memory for all hold objects, especially the MapGraphNode objects in _nodes
	static GraphNode* _getGraphNodeWithIndex(int index); //returns the pointer to the GraphNode (MapGraphNode) object with that index.
	static void _setAdjacency(bool adjacency); //probably won't need that outside of testing but sure, why not. false = 4, true = 8 adjacency.
	static bool _isMapPassableAt(int index); //probably won't need that outside of testing but sure, why not.
	static int _getMapWidth();
	static int _getMapHeight();
	static int _getMapSize();

	/**
	 * Constructor
	 */
	MapGraphNode(int index);
	virtual ~MapGraphNode();

	/**
	 *  Lists all adjacent fields in _map that are not obstructed.
	 *  Uses 4-adjacency or 8-adjacency depending on _eightAdjacency
	 */
	virtual std::vector<GraphNode*> getNeighbours();

	virtual string toString();

	virtual int getGraphSize();

	virtual GraphNode* getNodeFromGraph(int index);

	/**
	 * admissible, consistent
	 * returns shortest path length assuming no obstacles
	 * = sum of x and y distance for 4-adjacency (manhattan distance)
	 * = max of x and y distance for 8-adjacency
	 */
	double estimateDistanceTo(GraphNode* destination);
	double estimateDistanceTo(int destinationIndex);

	/**
	 * returns whether a pair of transitions conflicts in this graph or not.
	 * Does not depend on the node calling this function,
	 * which node to call it from only determines the Graph/Implementation.
	 *
	 * In this case it is moving to same tile, swapping positions, and moving over cross with 8-adjacency
	 */
	virtual bool isTransitionPairConflicting(int originA, int destinationA, int originB, int destinationB);

	virtual bool doConflictsRequireOverlappingNodes(){return !_eightAdjacency;};


};

#endif /* MAPGRAPHNODE_H_ */
