/*
 * MDDFactory.h
 *
 *  Created on: 19.10.2017
 *      Author: Simon
 */

#ifndef MDDFACTORY_H_
#define MDDFACTORY_H_

#include <map>
#include <utility>
#include <vector>

#include "MDDNode.h"

class MDDNode;
class AMDDNode;

class GraphNode;
class JMDD;
class MDD;

class MDDFactory {
private:
	/**
	 * For every goal tile it has an entry for (-> agent's goal nodes)
	 * it stores the optimal distance from every other tile to it.
	 */
	static std::map<int,int*> _distanceMap;
	static double* _heatMap;
	static bool _isJointTransitionConflicting(std::vector<int> &oA, std::vector<int> &dA, std::vector<int> &oB, std::vector<int> &dB);

public:
	/**
	 * Can be called for a goal node to generate its entry in the _distanceMap
	 * May be called directly, once for every agent's goal node, but if it was not
	 * the _makeMDDFromDistanceMap() method will take the liberty to call it when needed.
	 */
	static void _addDistanceMapEntry(GraphNode* goalNode);

	/**
	 * Initializes the _heatMap and must be called before _addToHeatMap()
	 */
	static void _initializeHeatMap(int size);

	/**
	 * Adds an agent's optimal and near-optimal paths to the heat-map
	 */
	static void _addToHeatMap(GraphNode* startNode, GraphNode* endNode, int addedBounds);
	static const double * _getHeatMap();

	/**
	 * Makes a new MDD object for a given agent, given start and end point and given costs
	 * Uses its internal _distanceMap field to generate it, for which it will
	 * call _addDistanceMapEntry() for the goal node if it wasn't already.
	 */
	static MDD* _makeMDDFromDistanceMap(GraphNode* startNode, GraphNode* endNode, int cost);
	static JMDD* _makeJMDDFromDistanceMap(int agentID, GraphNode* startNode, GraphNode* endNode, int cost);

	/**
	 * Well, I would know, wouldn't I. So long as there is an entry in _distanceMap of course.
	 * @returns optimal distance or -1 (no entry) or INT_MAX (unreachable)
	 */
	static int _getDistanceTo(int startTile, int endTile);
	static int _getDistanceTo(GraphNode* startNode, int endTile);
	static int _getDistanceTo(int startTile, GraphNode* endNode);
	static int _getDistanceTo(GraphNode* startNode, GraphNode* endNode);

	/**
	 * Returns the joint MDD
	 */
	static JMDD* _joinMDD(JMDD* first, JMDD* second);

	/**
	 * Copies an existing MDD so we can do abstractions in place.
	 * Resulting MDD will already contain AMDD nodes with potentially just one graph node per MDD node
	 */
	static MDD* _copyMDDForAbstraction(MDD* original);

	/**
	 * works in-place and replaces the pair of MDD nodes with a joint abstract node
	 */
	static MDDNode* _mergeTwoMDDNodesInPlace(MDDNode* nodeA, MDDNode* nodeB);

	/**
	 * splits off a graph node from the abstract MDD node given
	 * returns the new node first and abstract node second.
	 * return nullptr for new node if the new node didn't work (can't get there from any predecessor or can't go from there to any successor of the abstract nodes)
	 * also culls transitions of the abstract node which no longer apply, potentially making it disappear.
	 */
	static std::pair<MDDNode*,MDDNode*> _refineANodeBySplittingOffOneGraphNode(AMDDNode* node, int graphNode);

	/*
	 * works in-place and merges every two nodes of a layer which contain 'cold graph nodes' i.e. nodes whose heat map value is below heatThreshold
	 */
	static void _modifyMDDToMergeWithHeatMapThreshold(MDD* mdd, double heatThreshold);

	/**
	 * free stored information (e.g. _distanceMap) when done
	 */
	static void _freeMemory();

};


#endif /* MDDFACTORY_H_ */
