/*
 * ICTS.h
 *
 *  Created on: 03.11.2017
 *      Author: Simon
 */

#ifndef ICTS_H_
#define ICTS_H_

#include <map>
#include <set>
#include <unordered_map>
#include <utility>
#include <vector>

#include "GraphNode.h"
#include "hash.h"
#include "ICTSNode.h"
#include "JMDD.h"
#include "MDD.h"
#include "MDDNode.h"

using std::vector;
using std::pair;
using std::map;
using std::unordered_map;
using std::set;

class ICTS {
private:
	static map<pair<int,int>,MDD*> _storedMDDs; //matches a pair of agent and cost to an MDD (so we can avoid re-creating them).
	static map<pair<int,int>,MDD*> _storedAMDDs;
	static map<pair<int,int>,JMDD*> _storedJMDDs;
	                                    //TODO at what point can we delete MDDs from here? Can we ever?
	static double _heatThreshold;
	static double _heatThresholdMultiplier;
	/**
	 * Returns successors in the joint MDD space (Already pruning conflicts, true successors) along with the expected number of conflicts it will have with other agents not in the joint space (external information, 0 if none given).
	 * Optional parameters:
	 *     externalIllegalMoveTable: Forbids successors from moves that conflict with the provided moves.
	 *     deprecatedTiles: holds tiles already occupied by an agent from a different group in this time step.
	 */
	static vector<pair<vector<MDDNode*>*,int>>* _generateSuccessorsOf(vector<MDDNode*>* currentMetaNode, MDDNode** agentTails, map<int,int>* externalIllegalMoveTable = 0, map<int,int>* deprecatedMoves = 0);
	static vector<vector<GraphNode*>>* _reconstructPath(vector<MDDNode*>* goalMetaNode, unordered_map<vector<MDDNode*>*,vector<vector<MDDNode*>*>,utils::PointerResolutionHash<vector<MDDNode*>>,utils::PointerResolutionEqual<vector<MDDNode*>>>& predecessors, int goalDepth = -1, vector<map<int,int>>* deprecatedMoveTable = 0);
	static pair<int,int> _simulatePathAndCheckForConflicts(const vector<vector<GraphNode*>*>& paths);
	static vector<map<int,int>>* _fillIllegalMoveTables(const vector<vector<GraphNode*>*> &paths, const vector<int> &toAvoidIndices);
	static pair<int,MDDNode*> _getNumberOfNodesInNeedOfRefinement(vector<MDDNode*>* metaNode, vector<MDDNode*>* likelyPredecessorMetaNode); //tells us how many nodes we can not discount the possibility of refinement for, and also gives one of these nodes to refine.
	static int _getBestTileToSplitOffFromMDDNode(vector<map<int,int>>* deprecatedMoveTable, int distanceFromHead, MDDNode* nodeToRefine, MDDNode* likelyPredecessor);

public:
	/**
	 * Performs the high-level search.
	 * Returns a vector of paths (path = vector of Node pointers), one per agent.
	 * These paths have optimal sum-of-costs and no conflicts.
	 */
	static vector<vector<GraphNode*>>* _performIncreasingCostTreeSearch(vector<Agent>* agents, vector<int>* initialCosts, vector<map<int,int>>* deprecatedMoveTable = 0, bool abstract = false);

	/**
	 * Uses the Independence Detection framework to separate the agents,
	 * calling _performIncreasingCostTreeSearch() as its MAPF solver
	 * and returns the joint solution.
	 */
	static vector<vector<GraphNode*>>* _performIncreasingCostTreeSearchWithIndependenceDetection(vector<Agent>* agents, vector<int>* initialCosts, bool abstract = false);

	/**
	 * Performs the low-level search.
	 * Doesn't explicitly create the joint MDD but generated nodes on the fly
	 * as it searches through the joint state space in a best-first (least expected conflicts with tie-break depth-first) order
	 * Returns a joint path or a null pointer if none exists.
	 */
	static vector<vector<GraphNode*>>* _getJointPath(std::vector<MDD*>* singleAgentMDDs, vector<map<int,int>>* illegalMoveTables = 0, vector<map<int,int>>* deprecatedMoveTable = 0);

	/**
	 * Replans (with same costs) for a set of agents while respecting an externally supplied vector of illegalMoveTables (one per time step)
	 * Returns null pointer if no such replan is possible.
	 */
	static vector<vector<GraphNode*>*>* _replanWithIllegalMoveTables(vector<Agent>* agents, vector<int>* costs, vector<map<int,int>>* illegalMoveTables, vector<map<int,int>>* deprecatedMoveTable = 0, bool abstract = false);

	/**
	 * Set the graph by giving an arbitrary node from that graph as a reference point.
	 * MUST be called before any other function in this class.
	 */
	static void _setGraph(GraphNode* sourceNode);


	static MDD* _getMDDForAgentWithCost(Agent& agent, int cost);
	static MDD* _getAMDDForAgentWithCost(Agent& agent, int cost);
	static JMDD* _getJMDDForAgentWithCost(Agent& agent, int cost);

	/**
	 * This makes an explicitly joined MDD for a set of agents and respective costs. Effectively takes the role of low-level search for explicitly joint ICTS
	 */
	static JMDD* _makeJMDDForListOfAgentsAndCosts(vector<int> agentCosts, vector<Agent>* agents);

	static GraphNode* _sourceNode; //In order to make the ICTS independent of the implementation of GraphNode
		                               //I save an (arbitrary) node from the graph I want to use and ask it for members of its graph.

	/**
	 * At the end of the search, free all MDDs generated and still stored in _storedMDDs
	 */
	static void _freeStoredMDDs();

	/**
	 * sets the heat that a node must at least have to be considered "hot enough" to be exempted from abstraction.
	 */
	//static void _setHeatThreshold(double heat);

	/**
	 * sets a multiplier which will be taken into account when calculating the heat threshold. Must be set before the threshold is calculated to take effect.
	 */
	static void _setHeatThresholdMultiplier (double multiplier);

	/**
	 * initializes and fills the heat map with values according to the given agent list (and the graph stored in _sourceNode) as well as the value of _heatThresholdMultiplier
	 */
	static void _buildHeatMap(std::vector<Agent>* agents);

	const static bool _verbose = false; //TODO maybe find a better way to handle verbosity.

};


#endif /* ICTS_H_ */
