/*
 * ICTS.cpp
 *
 *  Created on: 04.11.2017
 *      Author: Simon
 */

#include "ICTS.h"

#include <algorithm> //for std::reverse in low-level search path reconstruction
#include <iostream>
#include <iterator>
#include <queue>
#include <stdexcept>
#include <string>
#include <unordered_set>
#include <climits>

#include "MDDFactory.h"
#include "AMDDNode.h"

using std::pair;
using std::tuple;
using std::make_pair;
using std::make_tuple;
using std::unordered_map;
using std::queue;
using std::priority_queue;
using std::set;
using std::unordered_set;
using std::cout;
using std::endl;

map<pair<int,int>,MDD*> ICTS::_storedMDDs;
map<pair<int,int>,MDD*> ICTS::_storedAMDDs;
map<pair<int,int>,JMDD*> ICTS::_storedJMDDs;
GraphNode* ICTS::_sourceNode = 0;
double ICTS::_heatThreshold = 0.0;
double ICTS::_heatThresholdMultiplier = 1.0;

void ICTS::_setGraph(GraphNode* sourceNode) {
	_sourceNode = sourceNode;
}

MDD* ICTS::_getMDDForAgentWithCost(Agent& agent, int cost) {
	auto iterator = _storedMDDs.find(make_pair(agent.id, cost));
	if (iterator != _storedMDDs.end()) {
		return (*iterator).second; //there was an entry already
	} else {
		if (_sourceNode != 0) {
			MDD* newMDD = MDDFactory::_makeMDDFromDistanceMap(
							_sourceNode->getNodeFromGraph(agent.startTile),
							_sourceNode->getNodeFromGraph(agent.goalTile),
							cost);
			 _storedMDDs[make_pair(agent.id, cost)] = newMDD;
			return newMDD;
		} else {
			throw std::runtime_error("ICTS does not have a graph specified. Please call ICTS::_setGraph() before working with it.");
		}

	}
}


//code duplication from ICTS::_getMDDForAgentWithCost()
MDD* ICTS::_getAMDDForAgentWithCost(Agent& agent, int cost) {
	auto iterator = _storedAMDDs.find(make_pair(agent.id, cost));
	if (iterator != _storedAMDDs.end()) {
		return (*iterator).second; //there was an entry already
	} else {
		if (_sourceNode != 0) {
			//MDD* newMDD = MDDFactory::_makeAMDDFromDistanceMapAndHeatMap( //Old and broken version
			//				_sourceNode->getNodeFromGraph(agent.startTile),
			//				_sourceNode->getNodeFromGraph(agent.goalTile),
			//				cost);
			MDD* newMDD = MDDFactory::_copyMDDForAbstraction(ICTS::_getMDDForAgentWithCost(agent, cost)); //fetch the concrete one (which should already have been made or else will be stored for when its needed)
			MDDFactory::_modifyMDDToMergeWithHeatMapThreshold(newMDD, _heatThreshold);
			_storedAMDDs[make_pair(agent.id, cost)] = newMDD;
			return newMDD;
		} else {
			throw std::runtime_error("ICTS does not have a graph specified. Please call ICTS::_setGraph() before working with it.");
		}

	}
}


//code duplication from ICTS::_getMDDForAgentWithCost()
JMDD* ICTS::_getJMDDForAgentWithCost(Agent& agent, int cost) {
	auto iterator = _storedJMDDs.find(make_pair(agent.id, cost));
	if (iterator != _storedJMDDs.end()) {
		return (*iterator).second; //there was an entry already
	} else {
		if (_sourceNode != 0) {
			JMDD* newMDD = MDDFactory::_makeJMDDFromDistanceMap(
							agent.id,
							_sourceNode->getNodeFromGraph(agent.startTile),
							_sourceNode->getNodeFromGraph(agent.goalTile),
							cost);
			 _storedJMDDs[make_pair(agent.id, cost)] = newMDD;
			return newMDD;
		} else {
			throw std::runtime_error("ICTS does not have a graph specified. Please call ICTS::_setGraph() before working with it.");
		}

	}
}

JMDD* ICTS::_makeJMDDForListOfAgentsAndCosts(vector<int> agentCosts, vector<Agent>* agents) {
	int agentCount = agentCosts.size();
	if (agentCount < 1) {
		return 0;
	} else if (agentCount < 2) {
		return _getJMDDForAgentWithCost((*agents)[0], agentCosts[0]);
	} else {
		JMDD* jointMDD = MDDFactory::_joinMDD(
				_getJMDDForAgentWithCost((*agents)[0], agentCosts[0]),
				_getJMDDForAgentWithCost((*agents)[1], agentCosts[1]));
		for (int i = 2; i < agentCount; ++i) {
			if (jointMDD != 0) {
				//cout << "i = " << i << endl;
				JMDD* intermediaryJMDD = jointMDD;
				JMDD* secondMDD = ICTS::_getJMDDForAgentWithCost((*agents)[i], agentCosts[i]);
				if (secondMDD == 0) throw std::runtime_error("While making big joint MDD the MDD for agent " + std::to_string((*agents)[i].id) + " and cost " + std::to_string(agentCosts[i]) + " was null.");
				jointMDD = MDDFactory::_joinMDD(jointMDD, secondMDD);
				delete intermediaryJMDD;
				//cout << "i = " << i<< " and JMDD = " << jointMDD << endl;
			} else return 0;
		}
		return jointMDD;
	}
	return 0; //unreachable return
}




//Caution: When called on a meta-node of all tails it will faithfully pad all of them to return as the sole successor an identical meta-node of all tails.
//         It will happily keep doing so ad infinitum so don't keep generating the successors' successors expecting it to stop at some point.
//         Check if you've already reached the lowest level in your MDD before calling this method is what I'm getting at.
//vector<pair<vector<MDDNode*>*,int>>* ICTS::_generateSuccessorsOf(vector<MDDNode*>* currentMetaNode, MDDNode** agentTails, map<int,int>* externalIllegalMoveTable, set<int>* deprecatedTiles) {
vector<pair<vector<MDDNode*>*,int>>* ICTS::_generateSuccessorsOf(vector<MDDNode*>* currentMetaNode, MDDNode** agentTails, map<int,int>* externalIllegalMoveTable, map<int,int>* deprecatedMoves) {
	vector<pair<vector<MDDNode*>*,int>>* jointSuccessors = new vector<pair<vector<MDDNode*>*,int>>;

	int agentCount = currentMetaNode->size();
	int successorCount = 1; //how many successors are possible (number of combinations of agents' successors)

	vector<vector<MDDNode*>> agentSuccessors(agentCount);
	for (int agentIndex = 0; agentIndex < agentCount; ++agentIndex) {
		if ((*currentMetaNode)[agentIndex] != agentTails[agentIndex]) {
			agentSuccessors[agentIndex] = (*currentMetaNode)[agentIndex]->successors;
			successorCount = successorCount * agentSuccessors[agentIndex].size();
		}
		// else successorCount = successorCount * 1; //(because exactly 1 successor by padding = waiting at no cost after reaching goal)
	}

	vector<pair<MDDNode*,MDDNode*>> previouslyTakenActions; //NEW CD //holds all actions previously taken for a successor and has to be iterated through in time linear in number of agents
	previouslyTakenActions.reserve(agentCount);
	for (int successorIndex = 0; successorIndex < successorCount; ++successorIndex) {

		vector<MDDNode*>* nextMetaNode = new vector<MDDNode*>(agentCount);
		int expectedNumberOfConflicts = 0;

		int indexLeft = successorIndex;
		int currentAgentSuccessorIndex;

		previouslyTakenActions.clear();
		bool conflict = false;

		//for every combination of agent indices each from 0 to agent's successors.size()-1
		for (int agentIndex = 0; agentIndex < agentCount; ++agentIndex) {

			//calculate the index of this agent's successor for this joint successor
			if ((*currentMetaNode)[agentIndex] != agentTails[agentIndex]) {
				currentAgentSuccessorIndex = indexLeft % agentSuccessors[agentIndex].size();
				indexLeft = indexLeft / agentSuccessors[agentIndex].size();

				//add this agent's successor to the joint successor
				(*nextMetaNode)[agentIndex] = agentSuccessors[agentIndex][currentAgentSuccessorIndex];


			} else {
				//indexLeft = indexLeft / 1 //(because again exactly 1 successor)

				//this MDD is already finished and the tail's only successor is the tail itself (padding).
				(*nextMetaNode)[agentIndex] = agentTails[agentIndex];
			}


			if (externalIllegalMoveTable != 0 && !conflict) {
				vector<int> currentTilesOfThatAgent = (*currentMetaNode)[agentIndex]->getAllTiles();
				vector<int> nextTilesOfThatAgent = (*nextMetaNode)[agentIndex]->getAllTiles();
				bool cascadeOut = false;
				for (auto currentTileIterator = currentTilesOfThatAgent.begin(); !cascadeOut && currentTileIterator != currentTilesOfThatAgent.end(); ++currentTileIterator) {
					for (auto nextTileIterator = nextTilesOfThatAgent.begin(); !cascadeOut && nextTileIterator != nextTilesOfThatAgent.end(); ++nextTileIterator) {
						conflict = false; //base assumption for every new tile combination. Only if every single tile combination anew sets conflict to true will we accept it as true
						for (auto illegalMoveIterator = externalIllegalMoveTable->begin(); illegalMoveIterator != externalIllegalMoveTable->end(); ++illegalMoveIterator) {
							if (_sourceNode->isTransitionPairConflicting(*currentTileIterator, *nextTileIterator, illegalMoveIterator->second, illegalMoveIterator->first)) {
								conflict = true;
								break;
							}
						}
						if (!conflict) {
							//only if a conflict was found we need to check other combinations of tiles to see if we find one without conflict
							cascadeOut = true;
						}
					}
				}
				if (conflict) break;
			}

			//regular conflicts
			for (auto actionIterator = previouslyTakenActions.begin(); actionIterator != previouslyTakenActions.end(); ++actionIterator) {
				if ((*currentMetaNode)[agentIndex]->isMDDNodeTransitionPairConflicting((*currentMetaNode)[agentIndex], (*nextMetaNode)[agentIndex], actionIterator->first, actionIterator->second, _sourceNode)) {
					conflict = true;
					break;
				}
			}
			if (conflict) break;
			previouslyTakenActions.push_back(make_pair((*currentMetaNode)[agentIndex], (*nextMetaNode)[agentIndex]));
			//CD END


			//estimate number of conflicts with agents other than the one being jointly processed right now
			if (deprecatedMoves != 0) {
				bool potentialConflict = false;
				vector<int> currentTilesOfThatAgent = (*currentMetaNode)[agentIndex]->getAllTiles();
				vector<int> nextTilesOfThatAgent = (*nextMetaNode)[agentIndex]->getAllTiles();
				bool cascadeOut = false;
				for (auto currentTileIterator = currentTilesOfThatAgent.begin(); !cascadeOut && currentTileIterator != currentTilesOfThatAgent.end(); ++currentTileIterator) {
					for (auto nextTileIterator = nextTilesOfThatAgent.begin(); !cascadeOut && nextTileIterator != nextTilesOfThatAgent.end(); ++nextTileIterator) {
						potentialConflict = false; //base assumption for every new tile combination. Only if every single tile combination anew sets conflict to true will we accept it as true
						for (auto illegalMoveIterator = deprecatedMoves->begin(); illegalMoveIterator != deprecatedMoves->end(); ++illegalMoveIterator) {
							if (_sourceNode->isTransitionPairConflicting(*currentTileIterator, *nextTileIterator, illegalMoveIterator->second, illegalMoveIterator->first)) {
								potentialConflict = true;
								break;
							}
						}
						if (!potentialConflict) {
							//if a conflict was found we need to check other combinations of tiles to see if we find one without conflict
							cascadeOut = true;
						}
					}
				}
				if (potentialConflict) {
					++expectedNumberOfConflicts;
				}
			}
		}

		//only proceed if there was no conflict
		if (conflict) {
			delete nextMetaNode;
			continue;
		}

		//add to return variable
		jointSuccessors->push_back(make_pair(nextMetaNode, expectedNumberOfConflicts));
	}

	return jointSuccessors;
}

//Only call for valid goal node and predecessor list;
//This reconstructs the path by walking backwards from the goal through the predecessors.
//It can handle abstract nodes if we have the ensureance that picking any position from the abstraction and comitting to it will yield a valid path.
//That is to say, if we could walk one of two ways and have them abstracted together because they are of same length and neither can cause conflicts,
//we may pick one of the two positions where the abstraction begins and can expect to find a neighbour of it in the predecessor abstraction and a neighbour of the neighbour in the predecessors predecessors abstraction etc. until reaching the start.
vector<vector<GraphNode*>>* ICTS::_reconstructPath(vector<MDDNode*>* goalMetaNode, unordered_map<vector<MDDNode*>*,vector<vector<MDDNode*>*>,utils::PointerResolutionHash<vector<MDDNode*>>,utils::PointerResolutionEqual<vector<MDDNode*>>>& predecessors, int goalDepth, vector<map<int,int>>* deprecatedMoveTable) {

	if (_sourceNode == 0) throw std::runtime_error("ICTS does not have a graph specified. Please call ICTS::_setGraph() before working with it.");

	vector<vector<GraphNode*>>* jointPath = new vector<vector<GraphNode*>>(goalMetaNode->size());

	vector<bool> active(goalMetaNode->size(), false); //agents lie dormant until (walking in reverse) they start moving away from the goal towards the start.
	                                                  //Only when they move away from the goal for the first time do they start recording every node they are in until reaching the start

	for (int agentIndex = 0; agentIndex < (int) goalMetaNode->size(); ++agentIndex) {
		(*jointPath)[agentIndex].push_back(
				_sourceNode->getNodeFromGraph(
						(*goalMetaNode)[agentIndex]->getExplicitTile()));
		//The goal is never abstracted, and even if it were we can chose ANY position (even though conflict avoidance table should be consulted in that case).
	}
	//vector<MDDNode*>*predecessorMetaNode = predecessors[goalMetaNode];
	vector<MDDNode*>* predecessorMetaNode = goalMetaNode; //reconstruct path by taking the first option; any option works and they are ordered by expansion order and thus quality like least expected conflicts

	int depth = goalDepth + 1; //+1 because will immediately be decremented

	while (predecessorMetaNode != 0) {

		vector<int> nextTiles;
		--depth;

		vector<MDDNode*>* previousPredecessorMetaNode = predecessorMetaNode;
		if (predecessors[previousPredecessorMetaNode][0] == 0) break;

		bool success = false;
		for (size_t predecessorIndex = 0; predecessorIndex < predecessors[previousPredecessorMetaNode].size(); ++predecessorIndex) {
			nextTiles = vector<int>((int) goalMetaNode->size(), -1);

			predecessorMetaNode = predecessors[previousPredecessorMetaNode][predecessorIndex];

			bool breakOut_nextPredecessorPlease = false;
			for (int agentIndex = 0; agentIndex < (int) goalMetaNode->size(); ++agentIndex) {
				if (breakOut_nextPredecessorPlease) break;
				if (!active[agentIndex]) {
					if (   (*predecessorMetaNode)[agentIndex]
						!= (*goalMetaNode)[agentIndex]) {
						active[agentIndex] = true;
						//Here we rely on the fact that the the tail nodes of the MDD as used in the meta nodes do not only encode the same positions but use the same object to do it.
						//The good thing about that is that we do not need to worry about abstractions, though admittedly that would not be much of a problem as the the goal is never abstract and the layer before that can never include the goal position, so 'getExplicitTile' would have served.
					}
				}
				if (active[agentIndex]) {
					if ((*predecessorMetaNode)[agentIndex]->getTileCount() > 1) {
						vector<int>* encodedPositions = ((AMDDNode*) (*predecessorMetaNode)[agentIndex])->getGraphNodes();
						GraphNode* positionToReach = (*jointPath)[agentIndex].back();
						int graphNodeIndexWeDecideToUseForPath = -1;
						int minPossibleConflicts = INT_MAX;
						double minHeat = 1000000.0;
						const double* heatMap = MDDFactory::_getHeatMap();

						for (auto encodedPositionIterator = encodedPositions->begin(); encodedPositionIterator != encodedPositions->end(); ++encodedPositionIterator) {
							//for all encoded positions...
							vector<GraphNode*> neighbours = _sourceNode->getNodeFromGraph(*encodedPositionIterator)->getNeighbours();
							if (std::find(neighbours.begin(), neighbours.end(), positionToReach) != neighbours.end()) {
								//.. find one that connects to out path


								/* //if we need just ANY one, this will do. Ideally we should try to consult the conflict avoidance table.
								(*jointPath)[agentIndex].push_back(
										_sourceNode->getNodeFromGraph(*encodedPositionIterator));
								success = true;
								break;
								*/

								//with Conflict Avoidance Table
								int currentPossibleConflicts = 0;
								for (auto moveToAvoidIterator = (*deprecatedMoveTable)[depth].begin(); moveToAvoidIterator != (*deprecatedMoveTable)[depth].end(); ++moveToAvoidIterator) {
									if (_sourceNode->isTransitionPairConflicting(*encodedPositionIterator, positionToReach->getIndex(), moveToAvoidIterator->second, moveToAvoidIterator->first)) {
										++currentPossibleConflicts;
									}
								}
								if (currentPossibleConflicts <= minPossibleConflicts) {
									if (currentPossibleConflicts < minPossibleConflicts || heatMap[(*encodedPositionIterator)] < minHeat) {
										minPossibleConflicts = currentPossibleConflicts;
										minHeat = heatMap[(*encodedPositionIterator)];
										graphNodeIndexWeDecideToUseForPath = (*encodedPositionIterator);
									}
								}

							}

						}
						if (graphNodeIndexWeDecideToUseForPath == -1) {
							breakOut_nextPredecessorPlease = true;
							break; //try again with a different predecessor;
						}
						nextTiles[agentIndex] = graphNodeIndexWeDecideToUseForPath;
						//(*jointPath)[agentIndex].push_back(
						//		_sourceNode->getNodeFromGraph(graphNodeIndexWeDecideToUseForPath));
					} else {
						//cout << "we take " << (*predecessorMetaNode)[agentIndex]->getExplicitTile() << endl;
						nextTiles[agentIndex] = (*predecessorMetaNode)[agentIndex]->getExplicitTile();
						//(*jointPath)[agentIndex].push_back(
						//		_sourceNode->getNodeFromGraph(
						//				(*predecessorMetaNode)[agentIndex]->getExplicitTile()));
						//Too easy to do for unabstracted nodes to pass up. Also, then I can safely assume the nodes are abstract for the other method which makes things easier.
					}

				}

			}
			if (breakOut_nextPredecessorPlease) continue;

			//if we got here we've found a way to continue every path
			for (int agentIndex = 0; agentIndex < (int) goalMetaNode->size(); ++agentIndex) {
				if (active[agentIndex]) {
					if (nextTiles[agentIndex] != -1) {
						(*jointPath)[agentIndex].push_back(
								_sourceNode->getNodeFromGraph(nextTiles[agentIndex]));
					}
				}
			}
			success = true;
			break;
		}// end of trying different predecessors
		if (!success) throw std::runtime_error("Path reconstruction failed");
	}
	for (int agentIndex = 0; agentIndex < (int) goalMetaNode->size(); ++agentIndex) {
		std::reverse((*jointPath)[agentIndex].begin(), (*jointPath)[agentIndex].end());
	}
	return jointPath;
}

// compares two pairs by comparing their first members only.
template<typename TA, typename TB>
struct LowLevelSearchQueueComparator {
	bool operator()(const std::pair<TA, TB> &lhs, const std::pair<TA, TB> &rhs) const {
		 return lhs.first < rhs.first;
    }
};

int ICTS::_getBestTileToSplitOffFromMDDNode(vector<map<int,int>>* deprecatedMoveTable, int distanceFromHead, MDDNode* nodeToRefine, MDDNode* likelyPredecessor) {
	map<int,int>* deprecatedMoves = 0;
	if (deprecatedMoveTable != 0 && deprecatedMoveTable->size() > 0) {
		if (distanceFromHead < (int) deprecatedMoveTable->size()) {
			deprecatedMoves = &( (*deprecatedMoveTable)[distanceFromHead] );
		} else {
			deprecatedMoves = &( (*deprecatedMoveTable)[deprecatedMoveTable->size() - 1] );
		}
	}
	//estimate number of conflicts with agents other than the one being jointly processed right now
	int graphNodeToSplitOff = nodeToRefine->getExplicitTile(); //just get one graph node, we have as yet no good way to determine a best order.
	if (deprecatedMoves != 0 && likelyPredecessor != 0) {
		vector<int> currentTilesOfThatAgent = likelyPredecessor->getAllTiles();
		vector<int> nextTilesOfThatAgent = nodeToRefine->getAllTiles();

		int minPossibleConflicts = INT_MAX;
		double minHeat = 1000000.0;
		const double* heatMap = MDDFactory::_getHeatMap();

		for (auto currentTileIterator = currentTilesOfThatAgent.begin(); currentTileIterator != currentTilesOfThatAgent.end(); ++currentTileIterator) {
			for (auto nextTileIterator = nextTilesOfThatAgent.begin(); nextTileIterator != nextTilesOfThatAgent.end(); ++nextTileIterator) {

				int numConflicts = 0;
				for (auto illegalMoveIterator = deprecatedMoves->begin(); illegalMoveIterator != deprecatedMoves->end(); ++illegalMoveIterator) {
					if (_sourceNode->isTransitionPairConflicting(*currentTileIterator, *nextTileIterator, illegalMoveIterator->second, illegalMoveIterator->first)) {
						++numConflicts;
					}
				}
				if (numConflicts <= minPossibleConflicts) {
					if (numConflicts < minPossibleConflicts || heatMap[(*nextTileIterator)] < minHeat) {
						graphNodeToSplitOff = *nextTileIterator;
						minPossibleConflicts = numConflicts;
						minHeat = heatMap[(*nextTileIterator)];
					}
				}
			}
		}
	}
	return graphNodeToSplitOff;
}

pair<int,MDDNode*> ICTS::_getNumberOfNodesInNeedOfRefinement(vector<MDDNode*>* metaNode, vector<MDDNode*>* likelyPredecessorMetaNode) {

	MDDNode* nodeToRefine = nullptr;
	int numberOfNodesToRefine = 0;

	vector<int>* dA;
	vector<int>* dB;
	bool assuredConflictFree = false;
	int minimumDistanceForConflicts = _sourceNode->minimumDestinationDistanceForConflicts();

	for (auto nodeAIterator = metaNode->begin(); nodeAIterator != metaNode->end(); ++nodeAIterator) {
		if ((*nodeAIterator)->getTileCount() > 1) { //for every abstract node in the meta node

			if (likelyPredecessorMetaNode == nullptr) {
				//no guarantees without consulting predecessor
				++numberOfNodesToRefine;
				if (nodeToRefine == nullptr) {
					nodeToRefine = *nodeAIterator;
				}
				continue;
			} else {

				assuredConflictFree = true; //base assumption, needs to be proven false
				for (auto nodeBIterator = metaNode->begin(); nodeBIterator != metaNode->end(); ++nodeBIterator) {
					//I though about letting it start at nodeAIterator + 1, but I can't safely do that when skipping non-abstract nodes with A

					if ((*nodeAIterator) != (*nodeBIterator)) { //checking if it conflicts with itself would of course defeat the purpose; of course it does on paper but of course that does not count.

						dA = ((AMDDNode*) *nodeAIterator)->getGraphNodes();
						dB = ((AMDDNode*) *nodeBIterator)->getGraphNodes();

						assuredConflictFree = true;
						minimumDistanceForConflicts = _sourceNode->minimumDestinationDistanceForConflicts();
						for (auto it_dA = dA->begin(); it_dA != dA->end(); ++it_dA) {
							for (auto it_dB = dB->begin(); it_dB != dB->end(); ++it_dB) {
								//for every possible pair of origin positions
								if (_sourceNode->getNodeFromGraph(*it_dA)->estimateDistanceTo(*it_dB) <= minimumDistanceForConflicts) { //if they are close enough to cause trouble
									assuredConflictFree = false;
								}
								if (!assuredConflictFree) break; //when we have one possible trouble maker, we can stop here.
							}
							if (!assuredConflictFree) break; //when we have one possible trouble maker, we can stop here.
						}

						if (!assuredConflictFree) {
							//this node (nodeAIterator) is a bad apple, we can make no easy guarantees and should better refine it.
							++numberOfNodesToRefine;
							if (nodeToRefine == nullptr) {
								nodeToRefine = *nodeAIterator;
							}
							break;
						}
					}
				}
				if (assuredConflictFree) {
					//ADDITIONAL REQUIREMENT: Predecessor node must contain every position of ours as a neighbour node.
					int agentIndex = std::find(metaNode->begin(), metaNode->end(), (*nodeAIterator)) - metaNode->begin();
					MDDNode* predecessorNode = (*likelyPredecessorMetaNode)[agentIndex];
					vector<int> predecessorTiles = predecessorNode->getAllTiles();
					vector<int> currentTiles = (*nodeAIterator)->getAllTiles();
					set<int> predecessorNeighbourUnion;
					for (auto predecessorTileIterator = predecessorTiles.begin(); predecessorTileIterator != predecessorTiles.end(); ++predecessorTileIterator) {
						vector<GraphNode*> neighbours = _sourceNode->getNodeFromGraph(*predecessorTileIterator)->getNeighbours();
						for (auto neighbourIterator = neighbours.begin(); neighbourIterator != neighbours.end(); ++neighbourIterator) {
							predecessorNeighbourUnion.insert((*neighbourIterator)->getIndex());
						}
					}
					for (auto currentTileIterator = currentTiles.begin(); currentTileIterator != currentTiles.end(); ++currentTileIterator) {
						if (std::find(predecessorNeighbourUnion.begin(), predecessorNeighbourUnion.end(), (*currentTileIterator)) != predecessorNeighbourUnion.end()) {
							//this tile of ours is okay
						} else {
							//this tile is not the neighbour of any predecessor tile, we can not guarantee a path through this abstract node will contain a concrete path.
							++numberOfNodesToRefine;
							if (nodeToRefine == nullptr) {
								nodeToRefine = *nodeAIterator;
							}
							break;
						}
					}
				}
			} //end of checking if this node may remain unrefined

		}
	}

	return make_pair(numberOfNodesToRefine, nodeToRefine);
}



vector<vector<GraphNode*>>* ICTS::_getJointPath(vector<MDD*>* singleAgentMDDs_original, vector<map<int,int>>* illegalMoveTables, vector<map<int,int>>* deprecatedMoveTable) {
	int agentCount = singleAgentMDDs_original->size();

	//trivial cases
	if (agentCount == 0) return nullptr;


	vector<MDD*>* singleAgentMDDs = new vector<MDD*>();
	singleAgentMDDs->reserve(agentCount);

	int maxCost = 0;
	for (auto it = singleAgentMDDs_original->begin(); it != singleAgentMDDs_original->end(); ++it) {

		//getting an empty MDD here can happen, for instance by having an agent that starts in the goal position that then has his cost limit raised to 1. (Can't go from A to A in 1 step since waiting at goal has cost 0)
		if (*it == 0) return nullptr; //return no solution, when an MDD is empty there can be no search and can be no solution.

		if ((*it)->getCost() > maxCost) {
			maxCost = (*it)->getCost();
		}


		// copy all MDDs so we can modify them; not ideal but I can't think of a good way to do it any other way
		singleAgentMDDs->push_back(MDDFactory::_copyMDDForAbstraction(*it));


	}

	//Key type pair<int,int> sorts key Key.first first and uses Key.second as tie-breaker. Does not sort by value (because custom comparator)
	//priority_queue<pair<pair<int,int>,vector<MDDNode*>*>, vector<pair<pair<int,int>,vector<MDDNode*>*>>, LowLevelSearchQueueComparator<pair<int,int>,vector<MDDNode*>*>> open;
	//Key is pair< -NumberOfConflicts , Depth >. the number of conflicts is given as negative so it will prefer fewer conflicts while preferring higher depth.

	priority_queue<pair<tuple<int,int,int>,vector<MDDNode*>*>, vector<pair<tuple<int,int,int>,vector<MDDNode*>*>>, LowLevelSearchQueueComparator<tuple<int,int,int>,vector<MDDNode*>*>> open;
	//Key is tuple<-NumberOfAbstractNodes, -NumberOfConflicts , Depth >. the number of conflicts is given as negative so it will prefer fewer abstract nodes and conflicts while preferring higher depth.



	// "predecessors" serves two purposes:
	// First, a closed list (or rather a "seen before" list)
	// Second, saves how we first got to a node for path reconstruction.
	// This only stores pointers but resolves them for hashing and equality checks as defined in hash.h
	unordered_map<vector<MDDNode*>*,vector<vector<MDDNode*>*>,utils::PointerResolutionHash<vector<MDDNode*>>,utils::PointerResolutionEqual<vector<MDDNode*>>> predecessors;

	vector<MDDNode*>* const head = new vector<MDDNode*>(agentCount);
	MDDNode** agentTails = new MDDNode*[agentCount];
	for (int agentIndex = 0; agentIndex < agentCount; ++agentIndex) {
		(*head)[agentIndex] = (*singleAgentMDDs)[agentIndex]->getHead();
		agentTails[agentIndex] = (*singleAgentMDDs)[agentIndex]->getTail();
	}

	predecessors[head] = vector<vector<MDDNode*>*>(1,nullptr); //vector containing a nullptr now replaces a straight null pointer as "no predecessors" (since we store multiple predecessors now, and consider having no predecessors a vital break criterium during path reconstruction)

	open.push(make_pair(make_tuple(0,0,0),head)); //the head can not have abstract nodes, at least not with value abstracted positions cause there is only one starting position.

	unordered_map<MDDNode*,pair<MDDNode*,MDDNode*>> pastRefinements; //need to keep list of past refinements to redirect outdated nodes. Wish I had thought of that necessity sooner, it des add a lot of memory.


	vector<MDDNode*>* goalMetaNode = 0;

	//initial goal test
	bool isGoalFound = true;
	for (int agentIndex = 0; agentIndex < agentCount; ++agentIndex) {
		if ((*head)[agentIndex] != (*singleAgentMDDs)[agentIndex]->getTail()) {
			isGoalFound = false;
			break;
		}
	}
	if (isGoalFound) {
		goalMetaNode = head; //need this for goal reconstruction
	}

	while (!isGoalFound && open.size() > 0) {
		pair<tuple<int,int,int>,vector<MDDNode*>*> currentPair = open.top();
		open.pop();

		int distanceFromHead = std::get<2>(currentPair.first);
		vector<MDDNode*>* currentMetaNode = currentPair.second;
		int accumulatedConflicts = -std::get<1>(currentPair.first);

		pair<int, MDDNode*> refinementInformation = _getNumberOfNodesInNeedOfRefinement(currentMetaNode, ((predecessors.find(currentMetaNode) != predecessors.end() && predecessors[currentMetaNode].size() > 0) ? predecessors[currentMetaNode][0] : nullptr));
		bool needsRefinement = refinementInformation.first != 0;
		MDDNode* nodeToRefine = refinementInformation.second;

		//BEGIN MDD Refinement
		if (needsRefinement) {
			//The plan: split off one graph node from one abstract node from one MDD and add the resulting new
			//If need be repeat step; multiple abstract MDDs dissolve one after the other, and in decent order. If we have 2 abstract MDD nodes we first split off a concrete one for the first one, then have that as a node with now only 1 abstract MDD which we prefer to split next to get our first concrete specimen after just 2 steps.
			//We cancel the expansion after refining the node and adding the results to the open list. The next node expanded should be one we just deposited, but it is possible that we need to refine again before it is concrete.

			//Find the first MDD with an abstract node.
			int nodeToRefineAgentIndex = std::find(currentMetaNode->begin(), currentMetaNode->end(), nodeToRefine) - currentMetaNode->begin();

			if ((*currentMetaNode)[nodeToRefineAgentIndex] != nodeToRefine) throw std::runtime_error("Agent index calculation wrong");
			if (nodeToRefine == 0) throw std::runtime_error("Tried to refine an abstract joint mdd node but could not find a single abstract MDD node within it");

			//Now that we have the node we need to split our currentMetaNode in two meta nodes, identical except one contains a concrete node split off from nodeToRefine to take the place of nodeToRefine itself while the other contains the rest of nodeToRefine after one was split off.
			//Both copies of currentMetaNode will be added to the open list, one with the concrete node to be popped again and hopefully expanded right away and one with the rest of nodeToRefine to be expanded and possible refined again when needed.

			vector<MDDNode*>* otherMetaNode = new vector<MDDNode*>(*currentMetaNode); //make a copy ; we will modify the entry for nodeToRefine later.

			vector<vector<MDDNode*>*> oldPredecessors = predecessors[currentMetaNode]; //stored for later use before we modify currentMetaNode
			predecessors.erase(currentMetaNode); //before we modify currentMetaNode and thus invalidate it as a key we need to remove it from the 'predecessors' map

			//But first see if it has not been split before
			pair<MDDNode*,MDDNode*> splitNodes;
			if (pastRefinements.find(nodeToRefine) != pastRefinements.end()) {

				splitNodes = pastRefinements[nodeToRefine];

			} else {

				MDDNode* likelyPredecessor = 0;
				if (oldPredecessors.size() > 0) {
					likelyPredecessor = (*(oldPredecessors[0]))[nodeToRefineAgentIndex]; //the first node through which we reached the node to be refined; avoid conflicts under the assumption that's where we're coming from.
				}

				int graphNodeToSplitOff = _getBestTileToSplitOffFromMDDNode(deprecatedMoveTable, distanceFromHead, nodeToRefine, likelyPredecessor);

				splitNodes = MDDFactory::_refineANodeBySplittingOffOneGraphNode((AMDDNode*) nodeToRefine, graphNodeToSplitOff);
				pastRefinements[nodeToRefine] = splitNodes;

			}

			if (splitNodes.first == nullptr) {
				//if splitNode.first resulted in a nullptr, splitting off this graph node from the MDD node has not resulted in a usable new MDDNode; modify otherMetaNode as usual but ignore adding this new one
				delete currentMetaNode;
			}

			if (splitNodes.second == nullptr) {
				//if splitNode.second resulted in a nullptr, splitting off this graph node from the MDD node has left the donor node unusable; modify currentMetaNode as usual but ignore adding this new one
				delete otherMetaNode;
			}

			if (splitNodes.first == nullptr && splitNodes.second == nullptr) {
				//well in that case there's nothing we can do. Not sure this can actually happen though.
				continue;
			}

			if (splitNodes.first != nullptr) {
				//replace the old entry of currentMetaNode with the new, concrete MDD node.
				std::replace(currentMetaNode->begin(), currentMetaNode->end(), nodeToRefine, splitNodes.first);
			}

			if (splitNodes.second != nullptr) {
				//replace the old entry of otherMetaNode with the new and now one graph node smaller MDD node.
				std::replace(otherMetaNode->begin(), otherMetaNode->end(), nodeToRefine, splitNodes.second);
			}


			//Now we need to keep the predecessors straight. Only a meta node that is a predecessor of our original abstract meta node we want to refine is eligible to be a predecessor of the two new nodes.
			//However, it might well be that we lose a predecessor doing the splitting. If we lose all predecessors we can - nay, must - delete the orphaned node. Another predecessor can later re-add it if there is a way to reach it that was just not yet explored at the time of this refinement.
			//Since we are just now expanding our original node we need not worry about its successors - they will shortly be generated with the new information - nor about anyone quoting it as its pedecessors; that too would require having expanded it before.

			//This next bit has a bit of code duplication from the method that generated meta mdd node successors.
			vector<pair<MDDNode*,MDDNode*>> previouslyTakenActions; //holds all actions previously taken for a successor and has to be iterated through in time linear in number of agents
			previouslyTakenActions.reserve(agentCount);
			bool conflict;

			for (auto metaPredecessorIterator = oldPredecessors.begin(); metaPredecessorIterator != oldPredecessors.end(); ++metaPredecessorIterator) {
				//now check if the old predecessor is still a predecessor of currentMetaNode and otherMetaNode

				if (splitNodes.first != nullptr) {
					//first for the one that was split off

					//first see if the single-agent MDD we just changed still supports the connection
					conflict = true;
					for (auto singleAgentPredecessorIterator = splitNodes.first->predecessors.begin(); singleAgentPredecessorIterator != splitNodes.first->predecessors.end(); ++singleAgentPredecessorIterator) {
						if ((*singleAgentPredecessorIterator) == (**metaPredecessorIterator)[nodeToRefineAgentIndex]) {
							conflict = false;
							break;
						}
					}
					if (!conflict) {

						//then see if the multi-agent MDD has conflicts preventing this transition
						previouslyTakenActions.clear();
						conflict = false;
						for (int agentIndex = 0; agentIndex < agentCount; ++agentIndex) {
							for (auto actionIterator = previouslyTakenActions.begin(); actionIterator != previouslyTakenActions.end(); ++actionIterator) {
								if ((*currentMetaNode)[agentIndex]->isMDDNodeTransitionPairConflicting((**metaPredecessorIterator)[agentIndex], (*currentMetaNode)[agentIndex], actionIterator->first, actionIterator->second, _sourceNode)) {
									conflict = true;
									break;
								}
							}
							if (conflict) break;
							previouslyTakenActions.push_back(make_pair((**metaPredecessorIterator)[agentIndex], (*currentMetaNode)[agentIndex]));
						}
						if (!conflict) {
							//yep, predecessor still good
							predecessors[currentMetaNode].push_back(*metaPredecessorIterator);
						}

					}
				}


				if (splitNodes.second != nullptr) {
					//then for the one that was split off from

					//first see if the single-agent MDD we just changed still supports the connection
					conflict = true;
					for (auto singleAgentPredecessorIterator = splitNodes.second->predecessors.begin(); singleAgentPredecessorIterator != splitNodes.second->predecessors.end(); ++singleAgentPredecessorIterator) {

						if ((*singleAgentPredecessorIterator) == (**metaPredecessorIterator)[nodeToRefineAgentIndex]) {
							conflict = false;
							break;
						}
					}
					if (!conflict) {
						//then see if the multi-agent MDD has conflicts preventing this transition
						previouslyTakenActions.clear();
						conflict = false;
						for (int agentIndex = 0; agentIndex < agentCount; ++agentIndex) {
							for (auto actionIterator = previouslyTakenActions.begin(); actionIterator != previouslyTakenActions.end(); ++actionIterator) {
								if ((*otherMetaNode)[agentIndex]->isMDDNodeTransitionPairConflicting((**metaPredecessorIterator)[agentIndex], (*otherMetaNode)[agentIndex], actionIterator->first, actionIterator->second, _sourceNode)) {
									conflict = true;
									break;
								}
							}
							if (conflict) break;
							previouslyTakenActions.push_back(make_pair((**metaPredecessorIterator)[agentIndex], (*otherMetaNode)[agentIndex]));
						}
						if (!conflict) {
							//yep, predecessor still good
							predecessors[otherMetaNode].push_back(*metaPredecessorIterator);
						}
					}

				}

			}

			//Now that we have the predecessors adjusted we can see if the nodes are actually still reachable through any previously explored path
			if (splitNodes.first != nullptr) {
				if (predecessors.count(currentMetaNode) == 0) {
					//no entry on how to reach currentMetaNode; that means we must delete it
					delete currentMetaNode;
				} else {
					//we can reach it and it is time to queue it up
					open.push(make_pair(make_tuple(-_getNumberOfNodesInNeedOfRefinement(currentMetaNode, ((predecessors.find(currentMetaNode) != predecessors.end() && predecessors[currentMetaNode].size() > 0) ? predecessors[currentMetaNode][0] : nullptr)).first, -accumulatedConflicts, distanceFromHead), currentMetaNode));
				}
			}

			if (splitNodes.second != nullptr) {
				if (predecessors.count(otherMetaNode) == 0) {
					//no entry on how to reach currentMetaNode; that means we must delete it
					delete otherMetaNode;
				} else {
					//we can reach it and it is time to queue it up
					open.push(make_pair(make_tuple(-_getNumberOfNodesInNeedOfRefinement(otherMetaNode, ((predecessors.find(otherMetaNode) != predecessors.end() && predecessors[otherMetaNode].size() > 0) ? predecessors[otherMetaNode][0] : nullptr)).first, -accumulatedConflicts, distanceFromHead), otherMetaNode));
				}
			}

			continue; //return to the open list, the next element popped will most likely be one we just added but it might need further refinement.
		}
		//END MDD Refinement

		if (distanceFromHead >= maxCost) {
			continue;
		}

		//extract the illegal moves for this time step from the list of illegal moves for all time steps
		//the last element of illegalMoveTables does not correspond with a specific point in time but rather all time points after the agents have run their course
		map<int,int>* externalIllegalMoveTable = 0;
		if (illegalMoveTables != 0 && illegalMoveTables->size() > 0) {
			if (distanceFromHead + 1 < (int) illegalMoveTables->size()) {
				externalIllegalMoveTable = &( (*illegalMoveTables)[distanceFromHead + 1] ); //it's distance + 1 because
					//it wants to know where to go from t and we tell it where we've been before t and how we got there.
			} else {
				//this is in case we've run out of time-specific instructions on what moves to avoid (because other agents are still moving)
				//and instead avoid dormant agents for all time steps after they've stopped changing
				externalIllegalMoveTable = &( (*illegalMoveTables)[illegalMoveTables->size() - 1] );
			}
		}

		map<int,int>* deprecatedMoves = 0;
		if (deprecatedMoveTable != 0 && deprecatedMoveTable->size() > 0) {
			if (distanceFromHead + 1 < (int) deprecatedMoveTable->size()) {
				deprecatedMoves = &( (*deprecatedMoveTable)[distanceFromHead + 1] );
			} else {
				deprecatedMoves = &( (*deprecatedMoveTable)[deprecatedMoveTable->size() - 1] );
			}
		}

		vector<pair<vector<MDDNode*>*,int>>* successors = _generateSuccessorsOf(currentMetaNode, agentTails, externalIllegalMoveTable, deprecatedMoves);

		for (auto successorIterator = successors->begin(); successorIterator != successors->end(); ++successorIterator) {

			vector<MDDNode*>* successorMetaNode = successorIterator->first;
			//int expectedNumberOfConflicts = successorIterator->second; //old interpretation of conflict avoidance table
			int expectedNumberOfConflicts = successorIterator->second + accumulatedConflicts;

			auto foundIterator = predecessors.find(successorMetaNode);
			if (foundIterator != predecessors.end()) {
				foundIterator->second.push_back(currentMetaNode); //add as an additional successor; needed when we want to consider more than one possible path

				delete successorMetaNode;
				continue;
			}

			predecessors[successorMetaNode] = vector<vector<MDDNode*>*>(1,currentMetaNode); //copy key into map
			//predecessors[successorMetaNode] = currentMetaNode; //copy key into map

			// new node

			//goal check
			isGoalFound = true;
			for (int agentIndex = 0; agentIndex < agentCount; ++agentIndex) {
				if ((*successorMetaNode)[agentIndex] != (*singleAgentMDDs)[agentIndex]->getTail()) {
					isGoalFound = false;
					break;
				}
			}

			if (isGoalFound) {
				goalMetaNode = successorMetaNode; //need this for goal reconstruction

				//delete the unprocessed potential successors still in "successors" by quickly finishing iterating
				//through them before breaking out and deleting the vector that held the pointers
				for (++successorIterator; successorIterator != successors->end(); ++successorIterator) {
					delete successorIterator->first;
				}
				//all meta-nodes generated in this low-level search should now be in predecessors or else deleted
				break;
			}


			//push to open list
			open.push(make_pair(make_tuple(-_getNumberOfNodesInNeedOfRefinement(successorMetaNode, currentMetaNode).first, -expectedNumberOfConflicts, distanceFromHead + 1), successorMetaNode));

		}

		delete successors;


	}
	delete[] agentTails;

	vector<vector<GraphNode*>>* pathToReturn = 0;
	if (isGoalFound) {
		pathToReturn = _reconstructPath(goalMetaNode, predecessors, maxCost, deprecatedMoveTable);
	}

	//Nodes in the singleAgentMDDs are a superset of nodes stored in predecessors, so don't bother freeing them first that oly leads to crashes trying to delete twice 'n finding nodes through already deleted ones.
	//necessary if we work with copies of single agent MDDs so we can modify them

	for (auto mddIterator = singleAgentMDDs->begin(); mddIterator != singleAgentMDDs->end(); ++mddIterator) {
		delete *mddIterator; //all copied and potentially modified MDDs
	}
	delete singleAgentMDDs; //and the container that kept the copies

	//delete the MDD nodes that were refined away
	for (auto refinedNodeIterator = pastRefinements.begin(); refinedNodeIterator != pastRefinements.end(); ++refinedNodeIterator) {
		delete refinedNodeIterator->first;
	}

	return pathToReturn;


}



//performing high-level search in breadth-first manner.
vector<vector<GraphNode*>>* ICTS::_performIncreasingCostTreeSearch(vector<Agent>* agents, vector<int>* initialCosts, vector<map<int,int>>* deprecatedMoveTable, bool abstract) {
	if (_verbose) cout << "Begin ICTS Algorithm" << endl;
	if (_sourceNode == 0) throw std::runtime_error("ICTS does not have a graph specified. Please call ICTS::_setGraph() before working with it.");

	//avoiding duplicate ICTSNodes. Man, that vector hashing in hash.h is really pulling its weight today.
	unordered_map<vector<int>,ICTSNode*>* alreadyGeneratedNodes = new unordered_map<vector<int>,ICTSNode*>;

	queue<ICTSNode*> open;

	ICTSNode* head = new ICTSNode(*initialCosts, agents, false);
	(*alreadyGeneratedNodes)[*initialCosts] = head;
	open.push(head);


	vector<vector<GraphNode*>>* goalPath = 0;

	if (abstract) {
		ICTSNode* abstractHead = new ICTSNode(*initialCosts, agents, true);
		goalPath = ICTS::_getJointPath(& abstractHead->agentMDDs, 0, deprecatedMoveTable);
		if (goalPath != 0) { //possible goal found
			if (_verbose) cout << "Abstraction accepts initial goal and is reliable; accept results" << endl;
		} else {
			if (_verbose) cout << "Abstraction rejects initial goal" << endl;
			goalPath = 0; //not genuine goal
		}
		delete abstractHead;
	} else {
		goalPath = ICTS::_getJointPath(& head->agentMDDs, 0, deprecatedMoveTable);
	}

	if (_verbose) {
		if (goalPath != 0) {
			//Goal Found!
			cout << "INITIAL GOAL! " << endl;
		}
	}


	while (goalPath == 0 && !open.empty()) { //until a goal is found (or we run out of nodes, but short of through a bug in the code that literally can't happen, there are infinite successors)
		if (_verbose) cout << "LOOP  open: " << open.size() << ", closed: " << alreadyGeneratedNodes->size() << endl;

		ICTSNode* next = open.front();
		open.pop();

		if (_verbose) {
			cout << "	costs are (";
			int sum = 0;
			for (int i = 0; i < (int) next->agentCosts.size(); ++i) {
				sum += next->agentCosts[i];
				cout << next->agentCosts[i] << ", ";
			}
			cout << ") for a sum of 	" << sum << endl;
		}


		// Expand Node

		if (abstract) { //CASE WITH ABSTRACTIONS; BEWARE CODE DUPLICATION


			vector<pair<ICTSNode*,ICTSNode*>> successors = next->getSuccessorsWithAbstracts(agents, alreadyGeneratedNodes);
			for (auto successorIterator = successors.begin(); successorIterator != successors.end(); ++successorIterator) {

				if (_verbose) {
					cout << "		performing goal test for (";
					int sum = 0;
					for (int i = 0; i < (int) successorIterator->first->agentCosts.size(); ++i) {
						sum += successorIterator->first->agentCosts[i];
						cout << successorIterator->first->agentCosts[i] << ", ";
					}
					cout << ") with a sum of 	" << sum << endl;
				}
				goalPath = ICTS::_getJointPath(&(successorIterator->first->agentMDDs), 0, deprecatedMoveTable);
				if (goalPath != 0) {
					if (_verbose) cout << "Abstraction accepts possible goal and is reliable; accept results" << endl;
					break;
				} else {
					if (_verbose) cout << "			No goal possible says abstraction" << endl;
				}
				open.push(successorIterator->second);
			}


		} else { //CASE WITHOUT ABSTRACTIONS; BEWARE CODE DUPLICATION


			vector<ICTSNode*> successors = next->getSuccessors(agents, alreadyGeneratedNodes);
			for (auto successorIterator = successors.begin(); successorIterator != successors.end(); ++successorIterator) {

				if (_verbose) {
					cout << "		performing goal test for (";
					int sum = 0;
					for (int i = 0; i < (int) (*successorIterator)->agentCosts.size(); ++i) {
						sum += (*successorIterator)->agentCosts[i];
						cout << (*successorIterator)->agentCosts[i] << ", ";
					}
					cout << ") with a sum of 	" << sum << endl;
				}
				//goal test, low-level search, high complexity, never call twice for one node
				goalPath = ICTS::_getJointPath(&(*successorIterator)->agentMDDs, 0, deprecatedMoveTable);
				if (goalPath != 0) {
					//Goal Found!
					break;
				}
				open.push(*successorIterator);
			}


		}


	}

	//free memory
	for (auto allNodesStillInMemoryIterator = alreadyGeneratedNodes->begin(); allNodesStillInMemoryIterator != alreadyGeneratedNodes->end(); ++allNodesStillInMemoryIterator) {
		delete allNodesStillInMemoryIterator->second;
	}
	delete alreadyGeneratedNodes;

	if (_verbose) cout << "End of ICTS Algorithm" << endl;
	return goalPath;

}

//returns {-1,-1} if no conflict was found and the conflicting agents if one was found
pair<int,int> ICTS::_simulatePathAndCheckForConflicts(const vector<vector<GraphNode*>*>& paths) {
	if (_verbose) cout << "	ID PS: Start" << endl;
	int firstConflictingAgent = -1;
	int secondConflictingAgent = -1;
	bool conflicts = false;

	int agentCount = paths.size();
	int longestPathLength = 0;

	//get the iterators which witch to iterate over the paths
	vector<vector<GraphNode*>::iterator> pathIterators(agentCount);
	vector<vector<GraphNode*>::iterator> pathIteratorsLastElements(agentCount); //one step before end(), the point at which each respective path needs to halt and reiterate the last valid element in the list while the longer paths are still going
	for (int agentIndex = 0; agentIndex < agentCount; ++agentIndex) {
		pathIterators[agentIndex] = paths[agentIndex]->begin();
		pathIteratorsLastElements[agentIndex] = paths[agentIndex]->end() - 1;
		if ((int) paths[agentIndex]->size() > longestPathLength) longestPathLength = (int) paths[agentIndex]->size();
	}

	map<pair<int,int>,int> previouslyTakenActions; //NEW CD
	vector<bool> agentsDone(agentCount, false);
	int currentlyOccupiedTile;
	int previouslyOccupiedTile;

	//go through the paths step-by-step
	for (int pathStep = 0; !conflicts && pathStep < longestPathLength; ++pathStep) {

		previouslyTakenActions.clear(); //NEW CD

		for (int agentIndex = 0; agentIndex < agentCount; ++agentIndex) {

			//CD
			currentlyOccupiedTile = (*pathIterators[agentIndex])->getIndex();
			if (pathStep > 0) { //can't have transition conflicts if we haven't had any transitions yet, assume previous tile = -1 for everyone (which means only same-tile conflicts possible)
				if (!agentsDone[agentIndex]) {
					previouslyOccupiedTile = (*(pathIterators[agentIndex] - 1))->getIndex();
				} else {
					previouslyOccupiedTile = currentlyOccupiedTile;
				}
			} else {
				previouslyOccupiedTile = -1;
			}

			//CD START
			for (auto actionIterator = previouslyTakenActions.begin(); actionIterator != previouslyTakenActions.end(); ++actionIterator) {
				conflicts = _sourceNode->isTransitionPairConflicting(previouslyOccupiedTile,      currentlyOccupiedTile,
																	 actionIterator->first.first, actionIterator->first.second);
				if (conflicts) {
					firstConflictingAgent = actionIterator->second;
					secondConflictingAgent = agentIndex;
					if (_verbose) {
						cout << "		conflict in moving from " << actionIterator->first.first << " to " << actionIterator->first.second << " as " << firstConflictingAgent
								<< " and from " << previouslyOccupiedTile << " to " << currentlyOccupiedTile << " as " << secondConflictingAgent << "." << endl;
					}
					break;
				}
			}
			if (conflicts) break;
			previouslyTakenActions[make_pair(previouslyOccupiedTile, currentlyOccupiedTile)] = agentIndex;
			//CD END

			//advance path unless it already finished
			if (!agentsDone[agentIndex]) {
				if (pathIterators[agentIndex] != pathIteratorsLastElements[agentIndex]) ++pathIterators[agentIndex];
				else agentsDone[agentIndex] = true; //only turns true if the agent failed to proceed meaning that in all future iterations prevTile = currTile can be assumed.
			}

		}
	}
	return make_pair(firstConflictingAgent, secondConflictingAgent);
}


//simulated path similar to _simulatePathAndCheckForConflicts() but for a subset of paths and does not check for conflicts, just fills illegalMoveTables
//illegalMoveTables is a vector of tables of type map<int,int>, wherein vector[timestep] is the element 'map[whereAnAgentIsAtThatTimeStep] = whereThatSameAgentWasInTheTimeStepBefore]'
vector<map<int,int>>* ICTS::_fillIllegalMoveTables(const vector<vector<GraphNode*>*> &paths, const vector<int> &toAvoidIndices) {
	int agentCount = toAvoidIndices.size();
	size_t longestPathLength = 0;

	//get the iterators with which to iterate over the paths
	vector<vector<GraphNode*>::iterator> pathIterators(agentCount);
	for (int agentIndex = 0; agentIndex < agentCount; ++agentIndex) {
		pathIterators[agentIndex] = paths[toAvoidIndices[agentIndex]]->begin();
		if (paths[toAvoidIndices[agentIndex]]->size() > longestPathLength) longestPathLength = paths[toAvoidIndices[agentIndex]]->size();
	}

	vector<map<int,int>>* illegalMoveTables = new vector<map<int,int>>(longestPathLength+1);

	for (size_t pathStep = 0; pathStep < longestPathLength + 1; ++pathStep) { //the +1 is so we have one final entry where all agents lie dormant. We can loop that when we run out of illegal move instructions (which means the agents we're avoiding with the table have run their course but the agents that are avoding them have not yet and need further instructions on where not to tread).

		for (int agentIndex = 0; agentIndex < agentCount; ++agentIndex) {

			GraphNode* currentlyOccupiedTile = *pathIterators[agentIndex];

			if (pathStep == 0) {
				(*illegalMoveTables)[pathStep][currentlyOccupiedTile->getIndex()] = -1;
			} else {

				GraphNode* previouslyOccupiedTile;
				if (pathStep < paths[toAvoidIndices[agentIndex]]->size()) {
					previouslyOccupiedTile = *(pathIterators[agentIndex] - 1); //for when we haven't reached the end yet
				} else {
					previouslyOccupiedTile = *pathIterators[agentIndex]; //for when we have
				}
				(*illegalMoveTables)[pathStep][currentlyOccupiedTile->getIndex()] = previouslyOccupiedTile->getIndex();

			}

			//advance path unless it already finished
			if (pathStep < paths[toAvoidIndices[agentIndex]]->size() - 1) ++pathIterators[agentIndex];
		}
	}

	return illegalMoveTables;
}

void ICTS::_buildHeatMap(vector<Agent>* agents) {
	MDDFactory::_initializeHeatMap(_sourceNode->getGraphSize());
	for (size_t agentIndex = 0; agentIndex < agents->size(); ++agentIndex) {
		GraphNode* startNode = _sourceNode->getNodeFromGraph(
				(*agents)[agentIndex].startTile);
		GraphNode* goalNode = _sourceNode->getNodeFromGraph(
				(*agents)[agentIndex].goalTile);
		MDDFactory::_addDistanceMapEntry(goalNode);
		MDDFactory::_addToHeatMap(startNode, goalNode, 3);
	}
	//_heatThreshold = (200.0 * agents->size()) / _sourceNode->getGraphSize();
	_heatThreshold = _heatThresholdMultiplier * (0.2 * agents->size()); //super simple function; needs refinement
}

vector<vector<GraphNode*>>* ICTS::_performIncreasingCostTreeSearchWithIndependenceDetection(vector<Agent>* agents, vector<int>* initialCosts, bool abstract) {

	int agentCount = agents->size();

	//BUILDING HEAT MAP (if necessary)
	if (abstract) {
		_buildHeatMap(agents);
		cout << "setting heat threshold at " << _heatThreshold << endl;
	}
	//END OF HEAT MAP BUILDING


	vector<int> groups(agentCount); //groups[agentID] = groupAgentbelongsTo (as integer identifier)
	vector<vector<GraphNode*>*> paths(agentCount); //paths[agentID] = pathThatAgentTakes (as pointer)

	groups.reserve(agentCount);
	paths.reserve(agentCount);

	int longestPathLength = 0;
	
	//map<int,vector<set<int>>>* conflictAvoidanceTable = new map<int,vector<set<int>>>;

	vector<int> previouslyUsedIndices; //used for conflict avoidance during initial /single-agent) path generation.

	for (int agentIndex = 0; agentIndex < agentCount; ++agentIndex) {
		Agent agent = (*agents)[agentIndex];

		//Line 1: assign each agent to a group
		groups[agentIndex] = agentIndex;

		//Line 2: plan a path for each group
		//Old Version: this was done with A* for each agent independently.
		//This gave random initial paths and thus could influence k', making comparisons between methods impossible
		/*
		paths[agentIndex] = _sourceNode->getNodeFromGraph(agent.startTile)->getShortestPathTo(agent.goalTile);
		if (paths[agentIndex] == 0) throw std::runtime_error("Encountered impossible path for agent " + std::to_string(agent.id) + " going from " + std::to_string(agent.startTile) + " to " + std::to_string(agent.goalTile));
		if ((int) paths[agentIndex]->size() < longestPathLength) longestPathLength = paths[agentIndex]->size();
		*/

		//New Version. Just use the pre-existing MAPF solver I also use for replanning during Independence Detection for groups of 1 agent, and use the conflict avoidance table to maybe get good starting paths. No guarantees but if we have to settle for one thing we always use it might as well be more sensible than randomness.
		vector<map<int,int>>* deprecatedMoveTable = _fillIllegalMoveTables(paths, previouslyUsedIndices);
		vector<Agent>* agentVector = new vector<Agent>(1,agent);
		vector<int>* costVector = new vector<int>(1,(*initialCosts)[agentIndex]);
		vector<vector<GraphNode*>*>* jointPath = _replanWithIllegalMoveTables(agentVector, costVector, 0, deprecatedMoveTable);
		paths[agentIndex] = jointPath->front(); //the one and only element, for the one and only agent
		if (paths[agentIndex] == 0) throw std::runtime_error("Encountered impossible path for agent " + std::to_string(agent.id) + " going from " + std::to_string(agent.startTile) + " to " + std::to_string(agent.goalTile));
		if ((int) paths[agentIndex]->size() < longestPathLength) longestPathLength = paths[agentIndex]->size();
		delete jointPath;
		delete costVector;
		delete agentVector;
		delete deprecatedMoveTable;
		previouslyUsedIndices.push_back(agentIndex);

	}
	
	previouslyUsedIndices.clear();

	if (_verbose) cout << "ID: Entering Loop" << endl;

	set<pair<int,int>> pastConflicts; //to avoid getting stuck in a loop trying to remedy conflicts without merging.

	bool conflicts = true;
	while (conflicts) { //Line 4/19: Repeat until no conflicts occur
		conflicts = false;

		//Line 5: simulate execution of all paths until a conflict between two groups G1 and G2 occurs
		pair<int,int> conflict = _simulatePathAndCheckForConflicts(paths);
		if (conflict.first != -1) {
			conflicts = true;

			vector<Agent>* newGroupToReplan = 0;
			vector<int>* newGroupInitialCosts = 0;
			vector<int> newGroupIndices;
			vector<vector<GraphNode*>*>* newGroupPaths = 0;


			//CONFLICT RESOLUTION STARTS HERE
			bool wasConflictResolved = false;
			if (pastConflicts.find(conflict) == pastConflicts.end()) {
				if (_verbose) cout << "	ID: Attempting Conflict Resolution between " << conflict.first << " and " << conflict.second << endl;

				newGroupToReplan = new vector<Agent>();
				newGroupInitialCosts = new vector<int>();

				vector<int> toAvoidIndices;
				toAvoidIndices.clear();

				vector<int> uninvolvedIndices;
				uninvolvedIndices.clear();

				//First try replanning group A to respect Group B

				int groupToReplan = groups[conflict.first];
				int groupToAvoidConflictsWith = groups[conflict.second];
				for (int agentIndex = 0; agentIndex < agentCount; ++agentIndex) {
					if (groups[agentIndex] == groupToReplan) {
						newGroupToReplan->push_back((*agents)[agentIndex]);
						newGroupInitialCosts->push_back(paths[agentIndex]->size() - 1); //keep same costs for agents
						newGroupIndices.push_back(agentIndex);
					} else if (groups[agentIndex] == groupToAvoidConflictsWith) {
						toAvoidIndices.push_back(agentIndex);
					} else {
						uninvolvedIndices.push_back(agentIndex);
					}
				}

				//replan Group A with illegalMoveTables supplies by Group B
				vector<map<int,int>>* illegalMoveTables = _fillIllegalMoveTables(paths, toAvoidIndices);
				vector<map<int,int>>* deprecatedMoveTable = _fillIllegalMoveTables(paths, uninvolvedIndices);
				newGroupPaths = _replanWithIllegalMoveTables(newGroupToReplan, newGroupInitialCosts, illegalMoveTables, deprecatedMoveTable, false); //replanning with abstract nodes leads to potentially faulty paths leads to impossibilities to respect independence detection; abstract nodes not at that level
				delete illegalMoveTables;
				delete deprecatedMoveTable;

				if (newGroupPaths != 0) {
					if (_verbose) cout << "		ID: Replan group " << conflict.first << " (" << newGroupPaths->size() << " agents) with respect to " << conflict.second << " succeeded" << endl;
					for (int newGroupIndex = 0; newGroupIndex < (int) newGroupPaths->size(); ++newGroupIndex) {
						int agentIndex = newGroupIndices[newGroupIndex]; //valid for paths, groups, agents; not valid for 'newGroup.*'

						if (_verbose) {
							cout << "			path[" << agentIndex << "] = ";
							for (size_t i = 0; i < paths[agentIndex]->size(); ++i) {
								cout << (*paths[agentIndex])[i]->getIndex() << ", ";
							}
							cout << " after" << endl;
						}
						vector<GraphNode*>* toDelete = paths[agentIndex]; //remember so we can remove the now unused old path;
						paths[agentIndex] = (*newGroupPaths)[newGroupIndex]; //switch out the path for that agent by changing the pointer to the new path for that agent just created on the heap and then deleting the old path on the heap previously stored in 'paths'.
						delete toDelete;

						if (_verbose) {
							cout << "			path[" << agentIndex << "] = ";
							for (size_t i = 0; i < paths[agentIndex]->size(); ++i) {
								cout << (*paths[agentIndex])[i]->getIndex() << ", ";
							}
							cout << " after" << endl;
						}

					}
					wasConflictResolved = true;
				} else {
					if (_verbose) cout << "		ID: Replan " << conflict.first << " with respect to " << conflict.second << " failed" << endl;

					//Then try replanning B to respect A

					//cleanup
					delete newGroupToReplan;
					delete newGroupInitialCosts;
					delete newGroupPaths; //while its elements are pointers, they are now all stored in 'paths' which had a corresponding number of old entries deleted.
					newGroupIndices.clear();
					toAvoidIndices.clear();
					uninvolvedIndices.clear();

					newGroupToReplan = new vector<Agent>();
					newGroupInitialCosts = new vector<int>();

					//duplicate code because at this point I don't care
					groupToReplan = groups[conflict.second];
					groupToAvoidConflictsWith = groups[conflict.first];
					for (int agentIndex = 0; agentIndex < agentCount; ++agentIndex) {
						if (groups[agentIndex] == groupToReplan) {
							newGroupToReplan->push_back((*agents)[agentIndex]);
							newGroupInitialCosts->push_back(paths[agentIndex]->size() - 1); //keep same costs for agents
							newGroupIndices.push_back(agentIndex);
						} else if (groups[agentIndex] == groupToAvoidConflictsWith) {
							toAvoidIndices.push_back(agentIndex);
						} else {
							uninvolvedIndices.push_back(agentIndex);
						}
					}

					//replan Group B with illegalMoveTables supplies by Group A
					vector<map<int,int>>* illegalMoveTables = _fillIllegalMoveTables(paths, toAvoidIndices);
					vector<map<int,int>>* deprecatedMoveTable = _fillIllegalMoveTables(paths, uninvolvedIndices);
					newGroupPaths = _replanWithIllegalMoveTables(newGroupToReplan, newGroupInitialCosts, illegalMoveTables, deprecatedMoveTable, false); //replanning with abstract nodes leads to potentially faulty paths leads to impossibilities to respect independence detection; abstract nodes not at that level
					delete illegalMoveTables;
					delete deprecatedMoveTable;

					if (newGroupPaths != 0) {
						if (_verbose) cout << "		ID: Replan group " << conflict.second << " (" << newGroupPaths->size() << " agents) with respect to " << conflict.first << " succeeded" << endl;
						for (int newGroupIndex = 0; newGroupIndex < (int) newGroupPaths->size(); ++newGroupIndex) {
							int agentIndex = newGroupIndices[newGroupIndex]; //valid for paths, groups, agents; not valid for 'newGroup.*'

							if (_verbose) {
								cout << "			path[" << agentIndex << "] = ";
								for (size_t i = 0; i < paths[agentIndex]->size(); ++i) {
									cout << (*paths[agentIndex])[i]->getIndex() << ", ";
								}
								cout << endl;
							}

							vector<GraphNode*>* toDelete = paths[agentIndex]; //remember so we can remove the now unused old path;
							paths[agentIndex] = (*newGroupPaths)[newGroupIndex]; //switch out the path for that agent by changing the pointer to the new path for that agent just created on the heap and then deleting the old path on the heap previously stored in 'paths'.
							delete toDelete;

							if (_verbose) {
								cout << "			path[" << agentIndex << "] = ";
								for (size_t i = 0; i < paths[agentIndex]->size(); ++i) {
									cout << (*paths[agentIndex])[i]->getIndex() << ", ";
								}
								cout << endl;
							}

						}
						wasConflictResolved = true;
					} else {

						if (_verbose) cout << "		ID: Replan " << conflict.second << " with respect to " << conflict.first << " failed" << endl;
					}

				}

				delete newGroupToReplan;
				delete newGroupInitialCosts;
				delete newGroupPaths; //while its elements are pointers, they are now all stored in 'paths' which had a corresponding number of old entries deleted.
				newGroupIndices.clear();

				//done with conflict amendment
				pastConflicts.insert(std::move(conflict));
			} else {
				if (_verbose) cout << "	ID: Trying to resolve " << conflict.first << " and " << conflict.second << " again; deducing cycle (either that or conflict resolution thought it worked but didn't)" << endl;
			}
			if (_verbose) cout << "	ID: Conflict Resolution between " << conflict.first << " and " << conflict.second << (wasConflictResolved? " Successful" : " Failed") << endl;


			if (wasConflictResolved == false) {

				if (_verbose) cout << "	ID: Merging groups " << groups[conflict.first] << " and " << groups[conflict.second] << endl;

				if (_verbose) cout << "		ID: With joint group having agents ";

				//Line 15: merge G1 and G2 into a single group
				vector<int> uninvolvedIndices;
				newGroupToReplan = new vector<Agent>();
				newGroupInitialCosts = new vector<int>();
				int groupToBeMergedInto = groups[conflict.first];
				int groupToBeClearedOut = groups[conflict.second];
				for (int agentIndex = 0; agentIndex < agentCount; ++agentIndex) {
					if (groups[agentIndex] == groupToBeClearedOut) {
						groups[agentIndex] = groupToBeMergedInto;

					} else if (groups[agentIndex] != groupToBeMergedInto) {
						uninvolvedIndices.push_back(agentIndex);
						continue;
					}
					newGroupToReplan->push_back((*agents)[agentIndex]);
					newGroupInitialCosts->push_back((*initialCosts)[agentIndex]);
					newGroupIndices.push_back(agentIndex);
					if (_verbose) cout << agentIndex << ", ";
				}

				if (_verbose) cout << endl;


				vector<map<int,int>>* deprecatedMoveTable = _fillIllegalMoveTables(paths, uninvolvedIndices);
				//--- get new paths from ICTS but in a different format where each path is individually on the heap
				vector<vector<GraphNode*>>* inconvenientFormatNewGroupPaths = _performIncreasingCostTreeSearch(newGroupToReplan, newGroupInitialCosts, deprecatedMoveTable, abstract);
				newGroupPaths = new vector<vector<GraphNode*>*>;
				for (auto iterator = inconvenientFormatNewGroupPaths->begin(); iterator != inconvenientFormatNewGroupPaths->end(); ++iterator) {
					vector<GraphNode*>* singleAgentPath = new vector<GraphNode*>(*iterator);
					newGroupPaths->push_back(singleAgentPath);
				}
				delete inconvenientFormatNewGroupPaths; //calls deconstructor on elements; have at this point been copied
				//--- end of format change, 'newGroupPaths' now available and filled with new vectors for each of which one vector has to be deleted to make room (the old ones from 'paths')
				delete deprecatedMoveTable;

				for (int newGroupIndex = 0; newGroupIndex < (int) newGroupPaths->size(); ++newGroupIndex) {
					int agentIndex = newGroupIndices[newGroupIndex]; //valid for paths, groups, agents; not valid for 'newGroup.*'

					if (_verbose) {
						cout << "			path[" << agentIndex << "] = ";
						for (size_t i = 0; i < paths[agentIndex]->size(); ++i) {
							cout << (*paths[agentIndex])[i]->getIndex() << ", ";
						}
						cout << endl;
					}

					//if (_verbose) cout << "			path[" << agentIndex << "] = " << paths[agentIndex] << " before and ";
					vector<GraphNode*>* toDelete = paths[agentIndex]; //remember so we can remove the now unused old path;
					paths[agentIndex] = (*newGroupPaths)[newGroupIndex]; //switch out the path for that agent by changing the pointer to the new path for that agent just created on the heap and then deleting the old path on the heap previously stored in 'paths'.
					delete toDelete;
					//if (_verbose) cout << paths[agentIndex] << " after." << endl;

					if (_verbose) {
						cout << "			path[" << agentIndex << "] = ";
						for (size_t i = 0; i < paths[agentIndex]->size(); ++i) {
							cout << (*paths[agentIndex])[i]->getIndex() << ", ";
						}
						cout << endl;
					}

				}

				if (_verbose) {
					cout << "		ID: after merging paths have lengths ";
					for (auto pathIterator = paths.begin(); pathIterator != paths.end(); ++pathIterator) {
						cout << (*pathIterator)->size() << ", ";
					}
					cout << endl;
				}

				//free temporary newGroup variables on the heap
				delete newGroupToReplan;
				delete newGroupInitialCosts;
				delete newGroupPaths;
				newGroupIndices.clear();
			}

		} //end of conflict case

	}

	if (_verbose) cout << "ID: Reconstructing Path" << endl;

	//knit together paths
	vector<vector<GraphNode*>>* jointPathToReturn = new vector<vector<GraphNode*>>(agentCount);
	for (int agentIndex = 0; agentIndex < agentCount; ++agentIndex) {
		vector<GraphNode*>* toDelete = paths[agentIndex]; //remember so we can remove the now unused old path;
		(*jointPathToReturn)[agentIndex] = *(paths[agentIndex]); //copy constructor
		delete toDelete;
	}


	//result information
	cout << "k = " << agentCount << endl;
	int maxGroupSize = 0;
	std::vector<int> groupSizes(agentCount, 0);
	for (int agentIndex = 0; agentIndex < agentCount; ++agentIndex) { //find largest subgroup
		++groupSizes[groups[agentIndex]];
		if (groupSizes[groups[agentIndex]] > maxGroupSize) {
			maxGroupSize = groupSizes[groups[agentIndex]];
		}
	}
	cout << "k' = " << maxGroupSize << endl;

	return jointPathToReturn;
}


vector<vector<GraphNode*>*>* ICTS::_replanWithIllegalMoveTables(vector<Agent>* agents, vector<int>* costs, vector<map<int,int>>* illegalMoveTables, vector<map<int,int>>* deprecatedMoveTable, bool abstract) {
	if (_sourceNode == 0) throw std::runtime_error("ICTS does not have a graph specified. Please call ICTS::_setGraph() before working with it.");
	ICTSNode mddSupplierNode(*costs, agents, abstract);
	vector<vector<GraphNode*>>* goalPath = ICTS::_getJointPath(&mddSupplierNode.agentMDDs, illegalMoveTables, deprecatedMoveTable);

	if (goalPath == 0) {
		//replanning failed, no path with such specifications possible
		return 0;
	} else {
		//replanning succeeded, now copy stuff around to get a more serviceable format (which is somewhat inefficient but I decided that is worth it)
		vector<vector<GraphNode*>*>* goalPathNewFormat = new vector<vector<GraphNode*>*>; //makes it more straight-forward to switch out individual agent's paths with this.
		for (auto iterator = goalPath->begin(); iterator != goalPath->end(); ++iterator) {
			vector<GraphNode*>* singleAgentPath = new vector<GraphNode*>(*iterator); //copy constructor
			goalPathNewFormat->push_back(singleAgentPath);
		}
		delete goalPath; //calls deconstructor on it's elements (vector<GraphNode*>)
		return goalPathNewFormat;
	}

	return 0; //unreachable

}

void ICTS::_setHeatThresholdMultiplier(double multiplier) {
	_heatThresholdMultiplier = multiplier;
}

void ICTS::_freeStoredMDDs() {
	for (auto mddIterator = _storedMDDs.begin(); mddIterator != _storedMDDs.end(); ++mddIterator) {
		delete mddIterator->second;
	}
	_storedMDDs.clear();
}
