/*
 * GraphNode.h
 *
 * Abstract Class that represents a node in the graph traversed by the agents.
 * As for the actual implementation I will probably use MapGraphNode most of the time.
 *
 *  Created on: 15.10.2017
 *      Author: Simon
 */

#ifndef GRAPHNODE_H_
#define GRAPHNODE_H_

#include <string>
#include <vector>

using std::string;

class GraphNode {

protected:
	int index = 0; // Every node must have an (integer) identifier.
	               // I'm sure every implementation can find some numbering scheme.

public:

	virtual ~GraphNode(){};

	/**
	 * Compares two nodes based whether their identifiers are equal
	 */
	bool operator==(const GraphNode& rhs);
	bool operator!=(const GraphNode& rhs);

	/**
	 * Checks if the nodes identifier equals the parameter
	 */
	bool operator==(int rhs);
	bool operator!=(int rhs);

	/**
	 * Returns this node's unique identifier.
	 */
	int getIndex();

	/**
	 * Returns the number of nodes in the Graph.
	 */
	virtual int getGraphSize() = 0;

	/**
	 *  Lists all nodes in the graph adjacent to this node.
	 */
	virtual std::vector<GraphNode*> getNeighbours() = 0;

	virtual string toString(); //for easier testing

	/**
	 * Calculates the shortest path to another node in the graph.
	 * For now uses uninformed unit-cost breadth-first search.
	 */
	std::vector<GraphNode*>* getShortestPathTo(GraphNode* destination);
	std::vector<GraphNode*>* getShortestPathTo(int destinationIndex);

	/**
	 * Calculates the distance to another node in the graph.
	 * Actually just returns the length of the shortest path
	 * as gathered from getShortestPathTo()
	 * (length -1 actually because the path also includes the start node)
	 */
	int getDistanceTo(GraphNode* destination);
	int getDistanceTo(int destinationIndex);

	/**
	 * Gets an entirely different node of given index from the same graph.
	 * Basically allows me to get an arbitrary node from the same graph/implementation as this one
	 */
	virtual GraphNode* getNodeFromGraph(int index) = 0;

	/**
	 * A heuristic for distance between this node and other given node.
	 * Should be kept admissible.
	 * If not otherwise defined by subclasses is the uninformed heuristic (= constant 0)
	 */
	virtual double estimateDistanceTo(GraphNode* destination);
	virtual double estimateDistanceTo(int destinationIndex);

	/**
	 * returns whether a pair of transitions conflicts in this graph or not.
	 * Does not depend on the node calling this function,
	 * which node to call it from only determines the Graph/Implementation.
	 *
	 * In this case it is moving to same tile and swapping positions.
	 */
	virtual bool isTransitionPairConflicting(int originA, int destinationA, int originB, int destinationB);

	/**
	 * Returns how far origin positions of transitions have to be at minimum for a conflict to be possible.
	 * For a substantial optimization where we make some tests for abstract nodes first to see if any transition in them can be conflicting
	 *
	 * For all conflict definitions I have this is 2.
	 */
	int inline minimumOriginDistanceForConflicts(){return 2;};

	/**
	 * Returns how far destination positions of transitions have to be at minimum for a conflict to be possible.
	 * For a substantial optimization where we make some tests for abstract nodes first to see if any transition in them can be conflicting
	 *
	 * For all conflict definitions I have this is 1.
	 */
	int inline minimumDestinationDistanceForConflicts(){return 1;};

	/**
	 * Returns whether or not the conflict definition as we have it here requires overlap in the nodes for a conflict.
	 * For a small optimization where we make some tests for abstract nodes first to see if any transition in them can be conflicting
	 *
	 * For MapGraphNodes that is true for 4-adjacency but not true for 8-adjacency (diagonal crossover conflicts)
	 */
	virtual bool doConflictsRequireOverlappingNodes(){return false;};

	//Maybe I'll need a "std::vector<GraphTransition> GetTransitions()" later (for costs and stuff)
	//-> Probably not, because MAPF agents pretty much always move at one transition per time step (->unit cost).

};


#endif /* GRAPHNODE_H_ */
