#ifndef CEGAR_ABSTRACT_SEARCH_H
#define CEGAR_ABSTRACT_SEARCH_H

#include "transition.h"
#include "types.h"
#include "h_updater.h"

#include "../algorithms/priority_queues.h"

#include <deque>
#include <memory>
#include <vector>

namespace cegar {
using Solution = std::deque<Transition>;

/*
  Find abstract solutions using A*.
*/
class AbstractSearch {
    class AbstractSearchInfo {
        int g;
        int h;
        Transition incoming_transition;
public:
        AbstractSearchInfo()
            : h(0),
              incoming_transition() {
            reset();
        }

        void reset() {
            g = std::numeric_limits<int>::max();
            incoming_transition = Transition();
        }

        void decrease_g_value_to(int new_g) {
            assert(new_g <= g);
            g = new_g;
        }

        int get_g_value() const {
            return g;
        }

        void increase_h_value_to(int new_h) {
            assert(new_h >= h);
            h = new_h;
        }

        int get_h_value() const {
            return h;
        }

        void set_incoming_transition(const Transition &transition) {
            incoming_transition = transition;
        }

        const Transition &get_incoming_transition() const {
            assert(incoming_transition.op_id != UNDEFINED &&
                   incoming_transition.target_id != UNDEFINED);
            return incoming_transition;
        }
    };
    
    //helper class: vector-based tree structure with check against loops
    class ShortestPathTree : public std::vector<Transition> {
    public:
        ShortestPathTree(int n) : std::vector<Transition>(n) {}
        /*
        // only if (!isAncestor(a, b)), a may be inserted as a child of b without introducing a loop
        bool isAncestor(size_t anc, size_t desc) const {
            assert(anc <= size());
            assert(desc <= size());
            size_t curr = desc;
            if((*this)[curr].target_id != UNDEFINED) return true;
            while((*this)[curr].target_id != UNDEFINED) {
                curr = (*this)[curr].target_id; 
                if (curr == anc) return true;
            };
            return false;
        }
        */
    };

    const std::vector<int> operator_costs;

    // Keep data structures around to avoid reallocating them.
    priority_queues::AdaptiveQueue<int> candidate_queue;
    priority_queues::AdaptiveQueue<int> open_queue;
    std::vector<AbstractSearchInfo> search_info;
    ShortestPathTree shortest_path;
    
    const HUpdateStrategy hupd;
    const bool greedy;
    
    float orphan_ratio = 1;

    void reset(int num_states);
    void set_h_value(int state_id, int h);
    std::unique_ptr<Solution> extract_solution(int init_id, int goal_id) const;
    
    
    int astar_search(
        const std::vector<Transitions> &transitions,
        const Goals &goals);
    std::unique_ptr<Solution> forward_search(
        int init_id,
        const Goals &goals);

public:
    explicit AbstractSearch(const std::vector<int> &operator_costs, 
                            const HUpdateStrategy hupd,
                            const bool greedy);

    std::unique_ptr<Solution> find_solution(
        const std::vector<Transitions> &transitions,
        int init_id,
        const Goals &goal_ids);
    
    float get_orphan_ratio() {return orphan_ratio;}
    int get_h_value(int state_id) const;
    void set_h_values(std::vector<int> values);
    void copy_h_value_to_children(int v, int v1, int v2);
    void update_perfect_h(const std::vector< cegar::Transitions >&,
                          std::pair<int,int>);
    void update_perfect_h_alt(
        const std::vector< cegar::Transitions >&, 
        const std::vector< cegar::Transitions >&, 
        std::pair<int,int>);
    void update_alg2(
        const std::vector< cegar::Transitions >&, 
        const std::vector< cegar::Transitions >&, 
        std::pair<int,int>);
    void update_goal_distances(const Solution &solution);
    void compute_full_distances(
        const std::vector<Transitions> &transitions,
        const std::unordered_set<int> &goals);
    bool test_distances(
        const std::vector<Transitions> &in,
        const std::unordered_set<int> &goals);
};

std::vector<int> compute_distances(
    const std::vector<Transitions> &transitions,
    const std::vector<int> &costs,
    const std::unordered_set<int> &start_ids);
}

#endif
