#! /usr/bin/env python3
# -*- coding: utf-8 -*-

import argparse
from collections import defaultdict
import json
import sys


HELP = """Convert a suite name to a list of domains.

A suite named "*-strips" means that the output file generated by the
Fast Downward translator is SAS+ plus action costs, while the "adl"
suffix means that it may also contain conditional effects and/or axioms.
These names are not ideal for distinguishing the domains, but they are
the way they are for historical reasons.

For some domains, there are several formulations. Suites with "ipc" in
their names (and their combinations) only contain one formulation of
every domain. If there are other formulations, they are in separate
suites which are not part of the regular IPC suites. They are only part
of special suites such as "suite_alternative_formulations" and
"suite_all"."""

_PREFIX = "suite_"


def suite_alternative_formulations():
    return ['airport-adl']


def suite_ipc98_to_ipc04_adl():
    return [
        'assembly', 'miconic-fulladl', 'miconic-simpleadl',
        'optical-telegraphs', 'philosophers', 'psr-large',
        'psr-middle', 'schedule',
    ]


def suite_ipc98_to_ipc04_strips():
    return [
        'airport', 'blocks', 'depot', 'driverlog', 'freecell', 'grid',
        'gripper', 'logistics00', 'logistics98', 'miconic', 'movie',
        'mprime', 'mystery', 'pipesworld-notankage', 'psr-small',
        'satellite', 'zenotravel',
    ]


def suite_ipc98_to_ipc04():
    # All IPC1-4 domains, including the trivial Movie.
    return sorted(suite_ipc98_to_ipc04_adl() + suite_ipc98_to_ipc04_strips())


def suite_ipc06_adl():
    return [
        'openstacks',
        'trucks',
    ]


def suite_ipc06_strips_compilations():
    return [
        'openstacks-strips',
        'trucks-strips',
    ]


def suite_ipc06_strips():
    return [
        'pathways',
        'pipesworld-tankage',
        'rovers',
        'storage',
        'tpp',
    ]


def suite_ipc06():
    return sorted(suite_ipc06_adl() + suite_ipc06_strips())


def suite_ipc08_common_strips():
    return [
        'parcprinter-08-strips',
        'pegsol-08-strips',
        'scanalyzer-08-strips',
    ]


def suite_ipc08_opt_adl():
    return ['openstacks-opt08-adl']


def suite_ipc08_opt_strips():
    return sorted(suite_ipc08_common_strips() + [
        'elevators-opt08-strips',
        'openstacks-opt08-strips',
        'sokoban-opt08-strips',
        'transport-opt08-strips',
        'woodworking-opt08-strips',
    ])


def suite_ipc08_opt():
    return sorted(suite_ipc08_opt_strips() + suite_ipc08_opt_adl())


def suite_ipc08_sat_adl():
    return ['openstacks-sat08-adl']


def suite_ipc08_sat_strips():
    return sorted(suite_ipc08_common_strips() + [
        # Note: cyber-security is missing.
        'elevators-sat08-strips',
        'openstacks-sat08-strips',
        'sokoban-sat08-strips',
        'transport-sat08-strips',
        'woodworking-sat08-strips',
    ])


def suite_ipc08_sat():
    return sorted(suite_ipc08_sat_strips() + suite_ipc08_sat_adl())


def suite_ipc08():
    return sorted(set(suite_ipc08_opt() + suite_ipc08_sat()))


def suite_ipc11_opt():
    return [
        'barman-opt11-strips',
        'elevators-opt11-strips',
        'floortile-opt11-strips',
        'nomystery-opt11-strips',
        'openstacks-opt11-strips',
        'parcprinter-opt11-strips',
        'parking-opt11-strips',
        'pegsol-opt11-strips',
        'scanalyzer-opt11-strips',
        'sokoban-opt11-strips',
        'tidybot-opt11-strips',
        'transport-opt11-strips',
        'visitall-opt11-strips',
        'woodworking-opt11-strips',
    ]


def suite_ipc11_sat():
    return [
        'barman-sat11-strips',
        'elevators-sat11-strips',
        'floortile-sat11-strips',
        'nomystery-sat11-strips',
        'openstacks-sat11-strips',
        'parcprinter-sat11-strips',
        'parking-sat11-strips',
        'pegsol-sat11-strips',
        'scanalyzer-sat11-strips',
        'sokoban-sat11-strips',
        'tidybot-sat11-strips',
        'transport-sat11-strips',
        'visitall-sat11-strips',
        'woodworking-sat11-strips',
    ]


def suite_ipc11():
    return sorted(suite_ipc11_opt() + suite_ipc11_sat())


def suite_ipc14_agl_adl():
    return [
        'cavediving-14-adl',
        'citycar-sat14-adl',
        'maintenance-sat14-adl',
    ]


def suite_ipc14_agl_strips():
    return [
        'barman-sat14-strips',
        'childsnack-sat14-strips',
        'floortile-sat14-strips',
        'ged-sat14-strips',
        'hiking-agl14-strips',
        'openstacks-agl14-strips',
        'parking-sat14-strips',
        'tetris-sat14-strips',
        'thoughtful-sat14-strips',
        'transport-sat14-strips',
        'visitall-sat14-strips',
    ]


def suite_ipc14_agl():
    return sorted(suite_ipc14_agl_adl() + suite_ipc14_agl_strips())


def suite_ipc14_mco_adl():
    return [
        'cavediving-14-adl',
        'citycar-sat14-adl',
        'maintenance-sat14-adl',
    ]


def suite_ipc14_mco_strips():
    return [
        'barman-mco14-strips',
        'childsnack-sat14-strips',
        'floortile-sat14-strips',
        'ged-sat14-strips',
        'hiking-sat14-strips',
        'openstacks-sat14-strips',
        'parking-sat14-strips',
        'tetris-sat14-strips',
        'thoughtful-mco14-strips',
        'transport-sat14-strips',
        'visitall-sat14-strips',
    ]


def suite_ipc14_mco():
    return sorted(suite_ipc14_mco_adl() + suite_ipc14_mco_strips())


def suite_ipc14_opt_adl():
    return [
        'cavediving-14-adl',
        'citycar-opt14-adl',
        'maintenance-opt14-adl',
    ]


def suite_ipc14_opt_strips():
    return [
        'barman-opt14-strips',
        'childsnack-opt14-strips',
        'floortile-opt14-strips',
        'ged-opt14-strips',
        'hiking-opt14-strips',
        'openstacks-opt14-strips',
        'parking-opt14-strips',
        'tetris-opt14-strips',
        'tidybot-opt14-strips',
        'transport-opt14-strips',
        'visitall-opt14-strips',
    ]


def suite_ipc14_opt():
    return sorted(suite_ipc14_opt_adl() + suite_ipc14_opt_strips())


def suite_ipc14_sat_adl():
    return [
        'cavediving-14-adl',
        'citycar-sat14-adl',
        'maintenance-sat14-adl',
    ]


def suite_ipc14_sat_strips():
    return [
        'barman-sat14-strips',
        'childsnack-sat14-strips',
        'floortile-sat14-strips',
        'ged-sat14-strips',
        'hiking-sat14-strips',
        'openstacks-sat14-strips',
        'parking-sat14-strips',
        'tetris-sat14-strips',
        'thoughtful-sat14-strips',
        'transport-sat14-strips',
        'visitall-sat14-strips',
    ]


def suite_ipc14_sat():
    return sorted(suite_ipc14_sat_adl() + suite_ipc14_sat_strips())


def suite_ipc14():
    return sorted(set(
        suite_ipc14_agl() + suite_ipc14_mco() +
        suite_ipc14_opt() + suite_ipc14_sat()))


def suite_ipc18_opt_adl():
    return [
        'caldera-opt18-adl',
        'caldera-split-opt18-adl',
        'nurikabe-opt18-adl',
        'settlers-opt18-adl',
    ]


def suite_ipc18_opt_strips():
    return [
        'agricola-opt18-strips',
        # Conditional effects are removed by the translator.
        'data-network-opt18-strips',
        'organic-synthesis-opt18-strips',
        # Conditional effects are removed by the translator.
        'organic-synthesis-split-opt18-strips',
        'petri-net-alignment-opt18-strips',
        'snake-opt18-strips',
        'spider-opt18-strips',
        'termes-opt18-strips',
    ]


def suite_ipc18_opt():
    return sorted(suite_ipc18_opt_adl() + suite_ipc18_opt_strips())


def suite_ipc18_sat_adl():
    return [
        'caldera-sat18-adl',
        'caldera-split-sat18-adl',
        'flashfill-sat18-adl',
        'nurikabe-sat18-adl',
        'settlers-sat18-adl',
    ]


def suite_ipc18_sat_strips():
    return [
        'agricola-sat18-strips',
        # Conditional effects are removed by the translator.
        'data-network-sat18-strips',
        'organic-synthesis-sat18-strips',
        # Conditional effects are removed by the translator.
        'organic-synthesis-split-sat18-strips',
        'snake-sat18-strips',
        'spider-sat18-strips',
        'termes-sat18-strips',
    ]


def suite_ipc18_sat():
    return sorted(suite_ipc18_sat_adl() + suite_ipc18_sat_strips())


def suite_ipc18():
    return sorted(set(
        suite_ipc18_opt() + suite_ipc18_sat()))


def suite_ipc23_opt_adl():
    return [
        'folding-opt23-adl',
        'labyrinth-opt23-adl',
        'recharging-robots-opt23-adl',
        'ricochet-robots-opt23-adl',
        'rubiks-cube-opt23-adl',
        'slitherlink-opt23-adl',
    ]


def suite_ipc23_opt_strips():
    return [
        'quantum-layout-opt23-strips',
    ]


def suite_ipc23_opt():
    return sorted(suite_ipc23_opt_adl() + suite_ipc23_opt_strips())


def suite_ipc23_sat_adl():
    return [
        'folding-sat23-adl',
        'labyrinth-sat23-adl',
        'recharging-robots-sat23-adl',
        'ricochet-robots-sat23-adl',
        'rubiks-cube-sat23-adl',
        'slitherlink-sat23-adl',
    ]


def suite_ipc23_sat_strips():
    return [
        'quantum-layout-sat23-strips',
    ]


def suite_ipc23_sat():
    return sorted(suite_ipc23_sat_adl() + suite_ipc23_sat_strips())


def suite_ipc23():
    return sorted(set(
        suite_ipc23_opt() + suite_ipc23_sat()))


def suite_unsolvable():
    return sorted(
        ['mystery:prob%02d.pddl' % index
         for index in [4, 5, 7, 8, 12, 16, 18, 21, 22, 23, 24]] +
        ['miconic-fulladl:f21-3.pddl', 'miconic-fulladl:f30-2.pddl'])


def suite_optimal_adl():
    return sorted(
        suite_ipc98_to_ipc04_adl() + suite_ipc06_adl() +
        suite_ipc08_opt_adl() + suite_ipc14_opt_adl() +
        suite_ipc18_opt_adl() + suite_ipc23_opt_adl())


def suite_optimal_strips():
    return sorted(
        suite_ipc98_to_ipc04_strips() + suite_ipc06_strips() +
        suite_ipc06_strips_compilations() + suite_ipc08_opt_strips() +
        suite_ipc11_opt() + suite_ipc14_opt_strips() +
        suite_ipc18_opt_strips() + suite_ipc23_opt_strips())


def suite_optimal():
    return sorted(suite_optimal_adl() + list(
        set(suite_optimal_strips()) - set(suite_ipc06_strips_compilations())))


def suite_satisficing_adl():
    return sorted(
        suite_ipc98_to_ipc04_adl() + suite_ipc06_adl() +
        suite_ipc08_sat_adl() + suite_ipc14_sat_adl() +
        suite_ipc18_sat_adl() + suite_ipc23_sat_adl())


def suite_satisficing_strips():
    return sorted(
        suite_ipc98_to_ipc04_strips() + suite_ipc06_strips() +
        suite_ipc06_strips_compilations() + suite_ipc08_sat_strips() +
        suite_ipc11_sat() + suite_ipc14_sat_strips() +
        suite_ipc18_sat_strips() + suite_ipc23_sat_strips())


def suite_satisficing():
    return sorted(suite_satisficing_adl() + suite_satisficing_strips())


def suite_all():
    return sorted(
        suite_ipc98_to_ipc04() + suite_ipc06() +
        suite_ipc06_strips_compilations() + suite_ipc08() +
        suite_ipc11() + suite_ipc14() + suite_ipc18() + suite_ipc23() +
        suite_alternative_formulations())


# Applies to domains where all actions cost 1.
TAG_HAS_ONLY_UNIT_COST_ACTIONS = 'unit-cost'
# Applies to domains with at least one action of cost 0.
TAG_HAS_ZERO_COST_ACTIONS = 'zero-cost'
# Applies to domains with action costs of 0 or 1.
TAG_HAS_ONLY_BINARY_COST_ACTIONS = 'binary-cost'


TAGS = [
    TAG_HAS_ONLY_UNIT_COST_ACTIONS,
    TAG_HAS_ZERO_COST_ACTIONS,
    TAG_HAS_ONLY_BINARY_COST_ACTIONS,
]


IMPLIED_TAGS = {
    TAG_HAS_ONLY_BINARY_COST_ACTIONS: [TAG_HAS_ZERO_COST_ACTIONS],
}


DOMAIN_TO_TAGS = {
    'agricola-opt18-strips': [],
    'agricola-sat18-strips': [],
    'airport': [TAG_HAS_ONLY_UNIT_COST_ACTIONS],
    'airport-adl': [TAG_HAS_ONLY_UNIT_COST_ACTIONS],
    'assembly': [TAG_HAS_ONLY_UNIT_COST_ACTIONS],
    'barman-mco14-strips': [TAG_HAS_ONLY_UNIT_COST_ACTIONS],
    'barman-opt11-strips': [],
    'barman-opt14-strips': [TAG_HAS_ONLY_UNIT_COST_ACTIONS],
    'barman-sat11-strips': [],
    'barman-sat14-strips': [TAG_HAS_ONLY_UNIT_COST_ACTIONS],
    'blocks': [TAG_HAS_ONLY_UNIT_COST_ACTIONS],
    'caldera-opt18-adl': [TAG_HAS_ONLY_UNIT_COST_ACTIONS],
    'caldera-sat18-adl': [TAG_HAS_ONLY_UNIT_COST_ACTIONS],
    'caldera-split-opt18-adl': [],
    'caldera-split-sat18-adl': [],
    'cavediving-14-adl': [],
    'childsnack-opt14-strips': [TAG_HAS_ONLY_UNIT_COST_ACTIONS],
    'childsnack-sat14-strips': [TAG_HAS_ONLY_UNIT_COST_ACTIONS],
    'citycar-opt14-adl': [],
    'citycar-sat14-adl': [],
    'data-network-opt18-strips': [TAG_HAS_ZERO_COST_ACTIONS],
    'data-network-sat18-strips': [TAG_HAS_ZERO_COST_ACTIONS],
    'depot': [TAG_HAS_ONLY_UNIT_COST_ACTIONS],
    'driverlog': [TAG_HAS_ONLY_UNIT_COST_ACTIONS],
    'elevators-opt08-strips': [TAG_HAS_ZERO_COST_ACTIONS],
    'elevators-opt11-strips': [TAG_HAS_ZERO_COST_ACTIONS],
    'elevators-sat08-strips': [TAG_HAS_ZERO_COST_ACTIONS],
    'elevators-sat11-strips': [TAG_HAS_ZERO_COST_ACTIONS],
    'flashfill-sat18-adl': [TAG_HAS_ZERO_COST_ACTIONS],
    'floortile-opt11-strips': [],
    'floortile-opt14-strips': [],
    'floortile-sat11-strips': [],
    'floortile-sat14-strips': [],
    'folding-opt23-adl': [TAG_HAS_ONLY_BINARY_COST_ACTIONS],
    'folding-sat23-adl': [TAG_HAS_ONLY_BINARY_COST_ACTIONS],
    'freecell': [TAG_HAS_ONLY_UNIT_COST_ACTIONS],
    'ged-opt14-strips': [TAG_HAS_ZERO_COST_ACTIONS],
    'ged-sat14-strips': [TAG_HAS_ZERO_COST_ACTIONS],
    'grid': [TAG_HAS_ONLY_UNIT_COST_ACTIONS],
    'gripper': [TAG_HAS_ONLY_UNIT_COST_ACTIONS],
    'hiking-agl14-strips': [TAG_HAS_ONLY_UNIT_COST_ACTIONS],
    'hiking-opt14-strips': [TAG_HAS_ONLY_UNIT_COST_ACTIONS],
    'hiking-sat14-strips': [TAG_HAS_ONLY_UNIT_COST_ACTIONS],
    'labyrinth-opt23-adl': [TAG_HAS_ONLY_UNIT_COST_ACTIONS],
    'labyrinth-sat23-adl': [TAG_HAS_ONLY_UNIT_COST_ACTIONS],
    'logistics00': [TAG_HAS_ONLY_UNIT_COST_ACTIONS],
    'logistics98': [TAG_HAS_ONLY_UNIT_COST_ACTIONS],
    'maintenance-opt14-adl': [TAG_HAS_ONLY_UNIT_COST_ACTIONS],
    'maintenance-sat14-adl': [TAG_HAS_ONLY_UNIT_COST_ACTIONS],
    'miconic': [TAG_HAS_ONLY_UNIT_COST_ACTIONS],
    'miconic-fulladl': [TAG_HAS_ONLY_UNIT_COST_ACTIONS],
    'miconic-simpleadl': [TAG_HAS_ONLY_UNIT_COST_ACTIONS],
    'movie': [TAG_HAS_ONLY_UNIT_COST_ACTIONS],
    'mprime': [TAG_HAS_ONLY_UNIT_COST_ACTIONS],
    'mystery': [TAG_HAS_ONLY_UNIT_COST_ACTIONS],
    'nomystery-opt11-strips': [TAG_HAS_ONLY_UNIT_COST_ACTIONS],
    'nomystery-sat11-strips': [TAG_HAS_ONLY_UNIT_COST_ACTIONS],
    'nurikabe-opt18-adl': [TAG_HAS_ONLY_UNIT_COST_ACTIONS],
    'nurikabe-sat18-adl': [TAG_HAS_ONLY_UNIT_COST_ACTIONS],
    'openstacks': [TAG_HAS_ONLY_UNIT_COST_ACTIONS],
    'openstacks-agl14-strips': [TAG_HAS_ONLY_BINARY_COST_ACTIONS],
    'openstacks-opt08-adl': [TAG_HAS_ONLY_BINARY_COST_ACTIONS],
    'openstacks-opt08-strips': [TAG_HAS_ONLY_BINARY_COST_ACTIONS],
    'openstacks-opt11-strips': [TAG_HAS_ONLY_BINARY_COST_ACTIONS],
    'openstacks-opt14-strips': [TAG_HAS_ONLY_BINARY_COST_ACTIONS],
    'openstacks-sat08-adl': [TAG_HAS_ONLY_BINARY_COST_ACTIONS],
    'openstacks-sat08-strips': [TAG_HAS_ONLY_BINARY_COST_ACTIONS],
    'openstacks-sat11-strips': [TAG_HAS_ONLY_BINARY_COST_ACTIONS],
    'openstacks-sat14-strips': [TAG_HAS_ONLY_BINARY_COST_ACTIONS],
    'openstacks-strips': [TAG_HAS_ONLY_UNIT_COST_ACTIONS],
    'optical-telegraphs': [TAG_HAS_ONLY_UNIT_COST_ACTIONS],
    'organic-synthesis-opt18-strips': [TAG_HAS_ONLY_UNIT_COST_ACTIONS],
    'organic-synthesis-sat18-strips': [TAG_HAS_ONLY_UNIT_COST_ACTIONS],
    'organic-synthesis-split-opt18-strips': [],
    'organic-synthesis-split-sat18-strips': [],
    'parcprinter-08-strips': [TAG_HAS_ZERO_COST_ACTIONS],
    'parcprinter-opt11-strips': [TAG_HAS_ZERO_COST_ACTIONS],
    'parcprinter-sat11-strips': [TAG_HAS_ZERO_COST_ACTIONS],
    'parking-opt11-strips': [TAG_HAS_ONLY_UNIT_COST_ACTIONS],
    'parking-opt14-strips': [TAG_HAS_ONLY_UNIT_COST_ACTIONS],
    'parking-sat11-strips': [TAG_HAS_ONLY_UNIT_COST_ACTIONS],
    'parking-sat14-strips': [TAG_HAS_ONLY_UNIT_COST_ACTIONS],
    'pathways': [TAG_HAS_ONLY_UNIT_COST_ACTIONS],
    'pegsol-08-strips': [TAG_HAS_ONLY_BINARY_COST_ACTIONS],
    'pegsol-opt11-strips': [TAG_HAS_ONLY_BINARY_COST_ACTIONS],
    'pegsol-sat11-strips': [TAG_HAS_ONLY_BINARY_COST_ACTIONS],
    'petri-net-alignment-opt18-strips': [TAG_HAS_ONLY_BINARY_COST_ACTIONS],
    'philosophers': [TAG_HAS_ONLY_UNIT_COST_ACTIONS],
    'pipesworld-notankage': [TAG_HAS_ONLY_UNIT_COST_ACTIONS],
    'pipesworld-tankage': [TAG_HAS_ONLY_UNIT_COST_ACTIONS],
    'psr-large': [TAG_HAS_ONLY_UNIT_COST_ACTIONS],
    'psr-middle': [TAG_HAS_ONLY_UNIT_COST_ACTIONS],
    'psr-small': [TAG_HAS_ONLY_UNIT_COST_ACTIONS],
    'quantum-layout-opt23-strips': [TAG_HAS_ONLY_UNIT_COST_ACTIONS],
    'quantum-layout-sat23-strips': [TAG_HAS_ONLY_UNIT_COST_ACTIONS],
    'recharging-robots-opt23-adl': [TAG_HAS_ONLY_UNIT_COST_ACTIONS],
    'recharging-robots-sat23-adl': [TAG_HAS_ONLY_UNIT_COST_ACTIONS],
    'ricochet-robots-opt23-adl': [TAG_HAS_ONLY_BINARY_COST_ACTIONS],
    'ricochet-robots-sat23-adl': [TAG_HAS_ONLY_BINARY_COST_ACTIONS],
    'rovers': [TAG_HAS_ONLY_UNIT_COST_ACTIONS],
    'rubiks-cube-opt23-adl': [TAG_HAS_ONLY_UNIT_COST_ACTIONS],
    'rubiks-cube-sat23-adl': [TAG_HAS_ONLY_UNIT_COST_ACTIONS],
    'satellite': [TAG_HAS_ONLY_UNIT_COST_ACTIONS],
    'scanalyzer-08-strips': [],
    'scanalyzer-opt11-strips': [],
    'scanalyzer-sat11-strips': [],
    'schedule': [TAG_HAS_ONLY_UNIT_COST_ACTIONS],
    'settlers-opt18-adl': [TAG_HAS_ZERO_COST_ACTIONS],
    'settlers-sat18-adl': [TAG_HAS_ZERO_COST_ACTIONS],
    'slitherlink-opt23-adl': [TAG_HAS_ONLY_UNIT_COST_ACTIONS],
    'slitherlink-sat23-adl': [TAG_HAS_ONLY_UNIT_COST_ACTIONS],
    'snake-opt18-strips': [TAG_HAS_ONLY_UNIT_COST_ACTIONS],
    'snake-sat18-strips': [TAG_HAS_ONLY_UNIT_COST_ACTIONS],
    'sokoban-opt08-strips': [TAG_HAS_ONLY_BINARY_COST_ACTIONS],
    'sokoban-opt11-strips': [TAG_HAS_ONLY_BINARY_COST_ACTIONS],
    'sokoban-sat08-strips': [TAG_HAS_ONLY_BINARY_COST_ACTIONS],
    'sokoban-sat11-strips': [TAG_HAS_ONLY_BINARY_COST_ACTIONS],
    'spider-opt18-strips': [TAG_HAS_ONLY_BINARY_COST_ACTIONS],
    'spider-sat18-strips': [TAG_HAS_ONLY_BINARY_COST_ACTIONS],
    'storage': [TAG_HAS_ONLY_UNIT_COST_ACTIONS],
    'termes-opt18-strips': [TAG_HAS_ONLY_UNIT_COST_ACTIONS],
    'termes-sat18-strips': [TAG_HAS_ONLY_UNIT_COST_ACTIONS],
    'tetris-opt14-strips': [],
    'tetris-sat14-strips': [],
    'thoughtful-mco14-strips': [TAG_HAS_ONLY_UNIT_COST_ACTIONS],
    'thoughtful-sat14-strips': [TAG_HAS_ONLY_UNIT_COST_ACTIONS],
    'tidybot-opt11-strips': [TAG_HAS_ONLY_UNIT_COST_ACTIONS],
    'tidybot-opt14-strips': [TAG_HAS_ONLY_UNIT_COST_ACTIONS],
    'tidybot-sat11-strips': [TAG_HAS_ONLY_UNIT_COST_ACTIONS],
    'tpp': [TAG_HAS_ONLY_UNIT_COST_ACTIONS],
    'transport-opt08-strips': [],
    'transport-opt11-strips': [],
    'transport-opt14-strips': [],
    'transport-sat08-strips': [],
    'transport-sat11-strips': [],
    'transport-sat14-strips': [],
    'trucks': [TAG_HAS_ONLY_UNIT_COST_ACTIONS],
    'trucks-strips': [TAG_HAS_ONLY_UNIT_COST_ACTIONS],
    'visitall-opt11-strips': [TAG_HAS_ONLY_UNIT_COST_ACTIONS],
    'visitall-opt14-strips': [TAG_HAS_ONLY_UNIT_COST_ACTIONS],
    'visitall-sat11-strips': [TAG_HAS_ONLY_UNIT_COST_ACTIONS],
    'visitall-sat14-strips': [TAG_HAS_ONLY_UNIT_COST_ACTIONS],
    'woodworking-opt08-strips': [],
    'woodworking-opt11-strips': [],
    'woodworking-sat08-strips': [],
    'woodworking-sat11-strips': [],
    'zenotravel': [TAG_HAS_ONLY_UNIT_COST_ACTIONS],
}


TAGS_TO_DOMAINS = defaultdict(set)
for domain, tags in DOMAIN_TO_TAGS.items():
    for tag in tags:
        TAGS_TO_DOMAINS[tag].add(domain)
        if tag in IMPLIED_TAGS:
            for implied_tag in IMPLIED_TAGS[tag]:
                TAGS_TO_DOMAINS[implied_tag].add(domain)


def apply_tags(suite, with_tag, without_tag):
    allowed_tags = with_tag or []
    forbidden_tags = without_tag or []
    domains = set(suite)
    for tag in allowed_tags:
        if tag in forbidden_tags:
            sys.exit("Tag {} is both allowed and forbidden".format(tag))
        domains = domains.intersection(TAGS_TO_DOMAINS.get(tag, set()))
    for tag in forbidden_tags:
        domains = domains.difference(TAGS_TO_DOMAINS.get(tag, set()))
    return sorted(domains)


def get_suite_names():
    return [
        name[len(_PREFIX):] for name in sorted(globals().keys())
        if name.startswith(_PREFIX)]


def get_suite(name):
    suite_func = globals()[_PREFIX + name]
    return suite_func()


def _parse_args():
    parser = argparse.ArgumentParser(description=HELP)
    parser.add_argument("suite", choices=get_suite_names(), help="suite name")
    parser.add_argument("--with-tag", nargs='+', choices=TAGS,
                        help="specify tags that domains of the chosen suite "
                        "must have")
    parser.add_argument("--without-tag", nargs='+', choices=TAGS,
                        help="specify tags that domains of the chosen suite "
                        "cannot have")
    return parser.parse_args()


def main():
    args = _parse_args()
    suite = get_suite(args.suite)
    suite = apply_tags(suite, args.with_tag, args.without_tag)
    # Use json module to print double-quote strings.
    print(json.dumps(suite))


if __name__ == "__main__":
    main()
