#! /usr/bin/env python
# -*- coding: utf-8 -*-

import itertools
import os

from lab.environments import LocalEnvironment, BaselSlurmEnvironment

from downward.experiment import FastDownwardExperiment
from downward.reports.absolute import AbsoluteReport
from downward.reports.compare import ComparativeReport
from downward.reports.scatter import ScatterPlotReport

import common_setup
from common_setup import IssueConfig, IssueExperiment

DIR = os.path.dirname(os.path.abspath(__file__))
SCRIPT_NAME = os.path.splitext(os.path.basename(__file__))[0]

REPO = os.environ["DOWNWARD_REPO"]
BENCHMARKS_DIR = os.environ["DOWNWARD_BENCHMARKS"]
REVISION_CACHE = os.path.expanduser('~/lab/revision-cache')
REVISIONS = ['default']
BUILDS = ['releasecustom']
configs = [
    ('seq', 'state_equation_constraints()'),
    ('lmcut', 'lmcut_constraints()'),
    ('pho_normal', 'pho_constraints()'),
    ('pho_one', 'pho_constraints(cost_type=ONE)'),
    ('pho_random', 'pho_constraints(cost_type=RANDOM)')
    ]

CONFIG_NICKS = [
    (name, ['--search', 'astar(operatorcounting([{parameter}]))'.format(**locals())]) for (name, parameter) in configs
] + [
    ('comb({nick1}, {nick2})'.format(**locals()),
     ['--search', 'astar(operatorcounting([{parameter1}, {parameter2}]))'.format(**locals())])
    for (nick1, parameter1), (nick2, parameter2) in itertools.combinations(configs, 2)
] + [
    ('comb({nick1}, {nick2}, {nick3})'.format(**locals()),
     ['--search', 'astar(operatorcounting([{parameter1}, {parameter2}, {parameter3}]))'.format(**locals())])
    for (nick1, parameter1), (nick2, parameter2), (nick3, parameter3) in itertools.combinations(configs, 3)
] + [
    ('comb({nick1}, {nick2}, {nick3}, {nick4})'.format(**locals()),
     ['--search', 'astar(operatorcounting([{parameter1}, {parameter2}, {parameter3}, {parameter4}]))'.format(**locals())])
    for (nick1, parameter1), (nick2, parameter2), (nick3, parameter3), (nick4, parameter4) in itertools.combinations(configs, 4)
] + [
    ('comb({nick1}, {nick2}, {nick3}, {nick4}, {nick5})'.format(**locals()),
     ['--search', 'astar(operatorcounting([{parameter1}, {parameter2}, {parameter3}, {parameter4}, {parameter5}]))'.format(**locals())])
    for (nick1, parameter1), (nick2, parameter2), (nick3, parameter3), (nick4, parameter4), (nick5, parameter5) in itertools.combinations(configs, 5)
]

CONFIGS = [
    IssueConfig(
        "{config_nick}".format(**locals()),
        config,
        build_options=[build],
        driver_options=["--build", build])
    for rev in REVISIONS
    for build in BUILDS
    for config_nick, config in CONFIG_NICKS
]

#SUITE = common_setup.DEFAULT_OPTIMAL_SUITE
COST_SUITE = [ 'agricola-opt18-strips', 'barman-opt11-strips',
               'data-network-opt18-strips', 'elevators-opt08-strips',
               'elevators-opt11-strips', 'floortile-opt14-strips',
               'ged-opt14-strips', 'nomystery-opt11-strips',
               'openstacks-agl14-strips', 'openstacks-opt08-strips',
               'openstacks-opt11-strips', 'openstacks-opt14-strips',
               'organic-synthesis-split-opt18-strips', 'parcprinter-08-strips',
               'parcprinter-opt11-strips', 'parking-opt11-strips',
               'parking-opt14-strips', 'pegsol-08-strips', 'pegsol-opt11-strips',
               'petri-net-alignment-opt18-strips', 'scanalyzer-08-strips',
               'scanalyzer-opt11-strips', 'sokoban-opt08-strips',
               'sokoban-opt11-strips', 'spider-opt18-strips',
               'tetris-opt14-strips', 'transport-opt08-strips',
               'transport-opt11-strips', 'transport-opt14-strips',
               'woodworking-opt08-strips', 'woodworking-opt11-strips']
SUITE = COST_SUITE
ENVIRONMENT = BaselSlurmEnvironment(
    partition="infai_1",
    email="a.thuering@unibas.ch",
    export=["PATH", "DOWNWARD_BENCHMARKS"])

if common_setup.is_test_run():
    SUITE = IssueExperiment.DEFAULT_TEST_SUITE
    ENVIRONMENT = LocalEnvironment(processes=4)

exp = IssueExperiment(
    revisions=REVISIONS,
    configs=CONFIGS,
    environment=ENVIRONMENT,
)

exp.add_suite(BENCHMARKS_DIR, SUITE)

exp.add_parser(exp.EXITCODE_PARSER)
exp.add_parser(exp.TRANSLATOR_PARSER)
exp.add_parser(exp.SINGLE_SEARCH_PARSER)
exp.add_parser(exp.ANYTIME_SEARCH_PARSER)
exp.add_parser(exp.PLANNER_PARSER)

exp.add_step('build', exp.build)

exp.add_step('start', exp.start_runs)

exp.add_fetcher(name='fetch')

ATTRIBUTES = IssueExperiment.DEFAULT_TABLE_ATTRIBUTES

exp.add_absolute_report_step(attributes=ATTRIBUTES + ['initial_h_value'])

# algorithm_pairs = [
#     ("{rev}-{nick1}".format(**locals()),
#      "{rev}-{nick2}".format(**locals()),
#      "Diff ({rev}-{nick2} - {rev}-{nick1})".format(**locals()))
#     for rev in REVISIONS
#     for (nick1, _), (nick2, _) in itertools.combinations(CONFIG_NICKS, 2)
# ]

algorithm_pairs = [
    ("{rev}-{nick1}".format(**locals()),
     "{rev}-{nick2}".format(**locals()),
     "Diff ({rev}-{nick2} - {rev}-{nick1})".format(**locals()))
    for rev in REVISIONS
    for (nick1, nick2) in [('pho_normal', 'pho_one'), ('pho_normal', 'pho_random'), ('pho_one', 'pho_random'), ('pho_normal', 'comb(pho_normal,pho_one)'), ('comb(pho_normal,pho_one,seq)', 'comb(pho_normal,seq)')]
]

exp.add_report(
    ComparativeReport(algorithm_pairs, attributes=['cost', 'coverage',  'expansions']))

# Add scatter plot report step.
def domain_as_category(run1, run2):
    return run1['domain']
exp.add_report(
    ScatterPlotReport(
        attributes=["expansions_until_last_jump"],
        filter_algorithm=["default-pho_normal", "default-pho_one"],
        #get_category=domain_as_category,
        xlabel='pho_Norm',
        ylabel='pho_One'),
    outfile='scatterplot_norm_one.png')

exp.add_report(
    ScatterPlotReport(
        attributes=["expansions"],
        filter_algorithm=["default-pho_normal", "default-pho_random"],
        #get_category=domain_as_category,
        xlabel='pho_Norm',
        ylabel='pho_One'),
    outfile='scatterplot_norm_rand.png')


# Add tex table step.
algorithms = ['default-pho_normal', 'default-pho_one', 'default-pho_random', 'default-lmcut', 'default-seq', 'default-comb(lmcut, pho_one)', 'default-comb(lmcut, pho_random)', 'default-comb(lmcut, pho_normal)', 'default-comb(seq, lmcut)']
exp.add_report(
    AbsoluteReport(
        attributes = ['coverage'],
        filter_algorithm = algorithms,
        format = 'tex'),
    outfile='test.tex')

# initial-h scatter plot report step.

def low_init_h(run):
    return run['initial_h_value'] > 0
exp.add_report(
    ScatterPlotReport(
        attributes=["initial_h_value"],
        filter_algorithm=["default-pho_normal", "default-pho_one"]),
    outfile='initial_h_scatterplot1.png')

# exp.add_report(
#     AbsoluteReport(
#         attributes=ATTRIBUTES,
#         filter=rename_algorithms,
#         filter_algorithm=algorithms),
#     outfile='test.html')
# Parse the commandline and show or run experiment steps.
exp.run_steps()
