#include "state_equation_constraints.h"

#include "lp_constraint_collection.h"
#include "operator_count_lp.h"

#include "../globals.h"
#include "../option_parser.h"
#include "../plugin.h"

using namespace std;

namespace pho {
void StateEquationConstraints::build_propositions() {
    propositions.resize(g_variable_domain.size());
    for (size_t var = 0; var < g_variable_domain.size(); var++) {
        propositions[var].resize(g_variable_domain[var]);
    }
    for (size_t op_id = 0; op_id < g_operators.size(); ++op_id) {
        const ::Operator &op = g_operators[op_id];
        const vector<PrePost> &pre_posts = op.get_pre_post();
        for (size_t i = 0; i < pre_posts.size(); i++) {
            int var = pre_posts[i].var;
            int pre = pre_posts[i].pre;
            int post = pre_posts[i].post;
            // NOTE without axioms or conditional effects the translator guarantees
            //      that pre_post cannot contain two tuples with the same variable.
            //      Also, pre_post cannot contain a tule for a variable with a
            //      prevail condition.
            if (pre == -1) {
                for (size_t j = 0; j < pre_posts.size(); j++) {
                    assert(j == i || pre_posts[j].var != var);
                }
                for (size_t j = 0; j < op.get_prevail().size(); j++) {
                    assert(op.get_prevail()[j].var != var);
                }
            }
            assert(post != -1);
            assert(pre != post);

            if (pre != -1) {
                propositions[var][post].always_produced_by.insert(op_id);
                propositions[var][pre].always_consumed_by.insert(op_id);
            } else {
                one_safe[var] = false;
                propositions[var][post].sometimes_produced_by.insert(op_id);
                for (size_t val = 0; val < propositions[var].size(); ++val) {
                    if (val != post) {
                        propositions[var][val].sometimes_consumed_by.insert(op_id);
                    }
                }
            }
        }
    }
}

void StateEquationConstraints::add_indices_to_constraints(LPConstraint &constraint,
                                                          const set<int> &indices,
                                                          double coefficient) const {
    for (set<int>::iterator it = indices.begin(); it != indices.end(); ++it) {
        constraint.insert(*it, coefficient);
    }
}

void StateEquationConstraints::add_constraints(LPConstraintCollection &constraint_collection) {
    vector<LPConstraint> constraints;
    for (size_t var = 0; var < propositions.size(); ++var) {
        for (size_t value = 0; value < propositions[var].size(); ++value) {
            Proposition &prop = propositions[var][value];
            if (use_lower_bound_constraints) {
                LPConstraint constraint;
                add_indices_to_constraints(constraint, prop.always_produced_by, 1.0);
                add_indices_to_constraints(constraint, prop.sometimes_produced_by, 1.0);
                add_indices_to_constraints(constraint, prop.always_consumed_by, -1.0);
                if (!constraint.empty()) {
                    prop.lower_bound_constraint_index = constraints.size();
                    constraints.push_back(constraint);
                }
            }
            if (use_upper_bound_constraints) {
                LPConstraint constraint;
                add_indices_to_constraints(constraint, prop.always_produced_by, 1.0);
                add_indices_to_constraints(constraint, prop.always_consumed_by, -1.0);
                add_indices_to_constraints(constraint, prop.sometimes_consumed_by, -1.0);
                if (!constraint.empty()) {
                    prop.upper_bound_constraint_index = constraints.size();
                    constraints.push_back(constraint);
                }
            }
        }
    }
    constraint_offset = constraint_collection.add_constraints(constraints);
}

StateEquationConstraints::StateEquationConstraints(const ::Options &opts)
    : use_lower_bound_constraints(opts.get<bool>("use_lower_bound_constraints")),
      use_upper_bound_constraints(opts.get<bool>("use_upper_bound_constraints")),
      use_1safe_information(opts.get<bool>("use_1safe_information")) {

    // Atom can be consumed and produced again but it can be consumed at most once
    // more than it is produced.
    net_change_lower_bound[VAR_UNDEFINED_IN_GOAL][VAR_HAS_VAL_IN_STATE] = -1.0;
    net_change_upper_bound[VAR_UNDEFINED_IN_GOAL][VAR_HAS_VAL_IN_STATE] = 0.0;
    one_safe_upper_bound[VAR_UNDEFINED_IN_GOAL][VAR_HAS_VAL_IN_STATE] = 0.0;

    // Atom can be produced and consumed again but it can be produced at most once
    // more than it is consumed.
    net_change_lower_bound[VAR_UNDEFINED_IN_GOAL][VAR_HAS_OTHER_VAL_IN_STATE] = 0.0;
    net_change_upper_bound[VAR_UNDEFINED_IN_GOAL][VAR_HAS_OTHER_VAL_IN_STATE] = 1.0;
    one_safe_upper_bound[VAR_UNDEFINED_IN_GOAL][VAR_HAS_OTHER_VAL_IN_STATE] = numeric_limits<double>::infinity();

    // Whenever the atom is consumed, it has to be produced again.
    net_change_lower_bound[VAR_HAS_VAL_IN_GOAL][VAR_HAS_VAL_IN_STATE] = 0.0;
    net_change_upper_bound[VAR_HAS_VAL_IN_GOAL][VAR_HAS_VAL_IN_STATE] = 0.0;
    one_safe_upper_bound[VAR_HAS_VAL_IN_GOAL][VAR_HAS_VAL_IN_STATE] = 0.0;

    // The atom has to be produced and hold in the end.
    net_change_lower_bound[VAR_HAS_VAL_IN_GOAL][VAR_HAS_OTHER_VAL_IN_STATE] = 1.0;
    net_change_upper_bound[VAR_HAS_VAL_IN_GOAL][VAR_HAS_OTHER_VAL_IN_STATE] = 1.0;
    one_safe_upper_bound[VAR_HAS_VAL_IN_GOAL][VAR_HAS_OTHER_VAL_IN_STATE] = 1.0;

    // The atom has to be consumed and must not hold in the end.
    net_change_lower_bound[VAR_HAS_OTHER_VAL_IN_GOAL][VAR_HAS_VAL_IN_STATE] = -1.0;
    net_change_upper_bound[VAR_HAS_OTHER_VAL_IN_GOAL][VAR_HAS_VAL_IN_STATE] = -1.0;
    one_safe_upper_bound[VAR_HAS_OTHER_VAL_IN_GOAL][VAR_HAS_VAL_IN_STATE] = -1.0;

    // Whenever the atom is produced, it has to be consumed again.
    net_change_lower_bound[VAR_HAS_OTHER_VAL_IN_GOAL][VAR_HAS_OTHER_VAL_IN_STATE] = 0.0;
    net_change_upper_bound[VAR_HAS_OTHER_VAL_IN_GOAL][VAR_HAS_OTHER_VAL_IN_STATE] = 0.0;
    one_safe_upper_bound[VAR_HAS_OTHER_VAL_IN_GOAL][VAR_HAS_OTHER_VAL_IN_STATE] = 0.0;
}

StateEquationConstraints::~StateEquationConstraints() {
}

void StateEquationConstraints::initialize_constraints(LPConstraintCollection &constraint_collection, vector<bool> &filter) {
    cout << "Initializing constraints from state equation:" << endl
         << "   using lower bound net change constraints = " << (use_lower_bound_constraints ? 1 : 0) << endl
         << "   using upper bound net change constraints = " << (use_upper_bound_constraints ? 1 : 0) << endl
         << "   using 1-safe information=" << (use_1safe_information ? 1 : 0) << endl;
    ::verify_no_axioms_no_cond_effects();
    one_safe = vector<bool>(g_variable_domain.size(), true);
    build_propositions();

    // Initialize goal state.
    goal_state = vector<state_var_t>(g_variable_domain.size(), numeric_limits<state_var_t>::max());
    for (int i = 0; i < g_goal.size(); ++i) {
        goal_state[g_goal[i].first] = g_goal[i].second;
    }

    add_constraints(constraint_collection);
    cout << "finished with initialization" << endl;
}

bool StateEquationConstraints::update_constraints(const State &state, OperatorCountLP &lp) {
    // Compute the bounds for the rows in the LP.
    for (int var = 0; var < propositions.size(); ++var) {
        for (int value = 0; value < propositions[var].size(); ++value) {
            const Proposition &prop = propositions[var][value];
            VariableSituationInGoal in_goal;
            if (goal_state[var] == numeric_limits<state_var_t>::max()) {
                in_goal = VAR_UNDEFINED_IN_GOAL;
            } else if (goal_state[var] == value) {
                in_goal = VAR_HAS_VAL_IN_GOAL;
            } else {
                in_goal = VAR_HAS_OTHER_VAL_IN_GOAL;
            }
            VariableSituationInState in_state;
            if (state[var] == value) {
                in_state = VAR_HAS_VAL_IN_STATE;
            } else {
                in_state = VAR_HAS_OTHER_VAL_IN_STATE;
            }
            // Set row bounds.
            if (prop.lower_bound_constraint_index >= 0) {
                double lower_bound = net_change_lower_bound[in_goal][in_state];
                lp.set_permanent_constraint_lower_bound(
                    constraint_offset + prop.lower_bound_constraint_index, lower_bound);
                if (use_1safe_information && one_safe[var]) {
                    double upper_bound = one_safe_upper_bound[in_goal][in_state];
                    lp.set_permanent_constraint_upper_bound(
                        constraint_offset + prop.lower_bound_constraint_index, upper_bound);
                }
            }
            if (prop.upper_bound_constraint_index >= 0) {
                double upper_bound = net_change_upper_bound[in_goal][in_state];
                lp.set_permanent_constraint_upper_bound(
                    constraint_offset + prop.upper_bound_constraint_index, upper_bound);
            }
        }
    }
    return false;
}

ConstraintGenerator *_parse(OptionParser &parser) {
    parser.add_option<bool>("use_lower_bound_constraints",
                            "enable use of lower bound net-change constraints",
                            "true");
    parser.add_option<bool>("use_upper_bound_constraints",
                            "enable use of upper bound net-change constraints",
                            "false");
    parser.add_option<bool>("use_1safe_information",
                            "enable use of 1-safe information",
                            "false");

    Heuristic::add_options_to_parser(parser);
    Options opts = parser.parse();
    if (parser.help_mode())
        return 0;
    if (!opts.get<bool>("use_lower_bound_constraints") &&
        !opts.get<bool>("use_upper_bound_constraints")) {
        parser.error("must use at least one type of constraints");
    }
    if (opts.get<bool>("use_1safe_information") && opts.get<bool>("use_upper_bound_constraints")) {
        parser.error("Using one-safe information is only supported for lower bound constraints. "
                     "Using both upper and lower bound constraints dominates the use of "
                     "one-safe information.");
    }
    if (parser.dry_run())
        return 0;
    return new StateEquationConstraints(opts);
}

Plugin<ConstraintGenerator> _plugin("state_equation_constraints", _parse);
}
