#ifndef PHO_PDB_OPTIMAL_CONSTRAINTS_H
#define PHO_PDB_OPTIMAL_CONSTRAINTS_H

#include "constraint_generator.h"

class PDBHeuristic;

namespace pho {
/*
   The optimal cost partitioning is computed by the following LP

   Variables:
     (a) heuristic[p] for each p in PDBs
     (b) operator_cost[p][o] for each p in PDBs and each operator o
     (c) distance[p][s'] for each p in PDBs and each s' in the abstract states of p
     All varaibles must be non-negative. We could use operator_cost[p][o] <= cost(o)
     but this information is already contained in the constraints.

   Objective Function: MAX sum_{p in PDBs} heuristic[p]

   Constraints:
    * For p in PDBs
       (1) distance[p][s'] <= 0 if the abstraction of s in PDB p is s'
       (2) distance[p][s'] <= distance[p][s''] + operator_cost[p][o]
             for each abstract transition <s'', o, s'> of PDB p
       (3) heuristic[p] <= distance[p][s*]
             for each abstract goal state s* of PDB p
    * For o in operators
       (4) sum_{p in PDBs} operator_cost[p][o] <= cost(o)

  We calculate the dual of this LP because this better fits the other kinds of
  constraints. To generate the dual, we introduce one variable for each
  primal constraint.

  Dual Variables:
    (1) slack_current[p] for each pattern
    (2) selected_transition[p][t] for each pattern p and each
          abstract transition t = <s'', o, s'> of PDB p
    (3) selected_goal[p][s*] for each pattern p and each
          abstract goal state s* of PDB p
    (4) operator_count[o] for each operator o
    Again, all variables must be non-negative.

   The objective function of the dual is defined by the bounds on the
   primal constraints and a minimization instead of a maximization.
   Only the primal constraints (4) have non-zero bounds.

   Dual Objective Function:
     MIN sum_{o in O} (cost(o)*operator_count[o])

   One dual constraint is generated for each primal variable v. Its lower bound
   is defined by the coefficient of v in the primal objective function,
   the coefficient for dual variable i is the coefficient of v in primal
   constraint i.

   Dual Constraints:
     For each pattern p
     (a) sum_{s* abstract goal in p} selected_goal[p][s*] >= 1
           Intuition: "At least one selected goal per PDB"
     (b) operator_count[o] >= sum_{t in T} selected_transition[p][t]
             where T = transitions of p labeled with o
           Intuition: "If n transitions are selected that use operators o,
                       the number of times o is used globally must be at least n."
     (c) One constraint for every abstract state s' of PDB p. All of them
         contain 1 * selected_transition[p][t] if t is a transition into s' and
         -1 * selected_transition[p][t] if t is a transition out of s'.
         Let
             D = sum_{t in IN(s')} selected_transition[p][t] -
                 sum_{t in OUT(s')} selected_transition[p][t]

           Intuition: "D keeps track of how often we enter and exit the
                       abstract state s'. D > 0 corresponds to reaching a new
                       state, D = 0 corresponds to staying in or out of a state,
                       D < 0 corresponds to leaving the current state."

         The constraint depends on whether s' is an abstract goal s* (s' = s*)
         and on whether s' is the abstraction of s in PDB p (s' = alpha(s)).

           (c1)  D >= 0                    if s' != s* and s' != alpha(s)
           (c2)  D >= selected_goal[p][s*] if s' == s* and s' != alpha(s)
           (c3)  D >= - slack_current[p]   if s' != s* and s' == alpha(s)
           (c4)  D >= selected_goal[p][s*] - slack_current[p]
                                           if s' == s* and s' == alpha(s)
         The variable slack_current[p] is only mentioned in this constraint and
         not in the objective function. It can always be set high enough to
         satisfy the constraint in which it occurs so we can merge (c1) with (c3)
         and (c2) with (c4) to

            (c13) D >= X                        if s' != s*
            (c24) D - selected_goal[p][s*] >= X if s' == s*

         The value of X is changed in each evaluation to 0 if s' != alpha(s)
         and -infinity if s' = alpha(s). This way, we avoid also having to
         introduce variables of type (1).
         The rest of the LP does not depend on the current state and can remain
         fixed.
*/

class PDBOptimalConstraints : public ConstraintGenerator {
    OperatorCost cost_type;
    std::vector<std::vector<int> > patterns;
    std::vector<PDBHeuristic *> heuristics;

    int constraint_offset;

    // Start of type (c) constraints for each PDB
    std::vector<int> entry_count_contraint_offsets;
    // Cache the constraint ids corresponding to the current state in all pdbs.
    // This makes it easier to reset the bounds in each step.
    std::vector<int> current_abstract_state_constraint_ids;
    void add_pattern(const std::vector<int> &pattern,
                     LPConstraintCollection &constraint_collection);
public:
    PDBOptimalConstraints(const Options &opts);
    ~PDBOptimalConstraints();
    virtual void initialize_constraints(LPConstraintCollection &constraint_collection, std::vector<int> &ignore_list);
    virtual bool reach_state(const State &parent_state, const Operator &op,
                             const State &state);
    virtual bool update_constraints(const State &state, OperatorCountLP &lp);
};
}

#endif
