#include "lp_constraint_collection.h"

#include <CoinPackedMatrix.hpp>
#include <CoinPackedVector.hpp>

#include <algorithm>

using namespace std;

namespace pho {
double translate_infinity(double value, OsiSolverInterface *lp_solver) {
    if (value == numeric_limits<double>::infinity())
        return lp_solver->getInfinity();
    else if (value == -numeric_limits<double>::infinity())
        return -lp_solver->getInfinity();
    return value;
}

LPConstraint::LPConstraint()
    : lower_bound(-numeric_limits<double>::infinity()),
      upper_bound(numeric_limits<double>::infinity()) {
}

LPConstraint::~LPConstraint() {
}

bool LPConstraint::empty() const {
    return indices.empty();
}

void LPConstraint::insert(int index, double coefficient) {
    assert(find(indices.begin(), indices.end(), index) == indices.end());
    indices.push_back(index);
    coefficients.push_back(coefficient);
}

int LPConstraintCollection::add_variable(LPVariable var) {
    int offset = variables.size();
    variables.push_back(var);
    return offset;
}

int LPConstraintCollection::add_variables(int n, LPVariable var) {
    int offset = variables.size();
    variables.resize(variables.size() + n, var);
    return offset;
}

int LPConstraintCollection::add_constraints(const vector<LPConstraint> &new_constraints) {
    int offset = constraints.size();
    constraints.insert(constraints.end(), new_constraints.begin(), new_constraints.end());
    return offset;
}

CoinRows::CoinRows(const vector<LPConstraint> &constraints,
                   OsiSolverInterface *lp_solver) {
    num_rows = constraints.size();
    lower_bounds = new double[num_rows];
    upper_bounds = new double[num_rows];
    rows = new CoinPackedVectorBase *[num_rows];
    for (size_t i = 0; i < num_rows; ++i) {
        const LPConstraint &constraint = constraints[i];
        rows[i] = create_coin_vector(constraint, lp_solver);
        lower_bounds[i] = translate_infinity(constraint.get_lower_bound(), lp_solver);
        upper_bounds[i] = translate_infinity(constraint.get_upper_bound(), lp_solver);
    }
    owns_data = true;
}

void CoinRows::release_ownership() {
    owns_data = false;
}

CoinRows::~CoinRows() {
    if (owns_data) {
        delete[] lower_bounds;
        delete[] upper_bounds;
        for (size_t i = 0; i < num_rows; ++i) {
            delete rows[i];
        }
        delete[] rows;
    }
}

CoinPackedVector *CoinRows::create_coin_vector(const LPConstraint &constraint,
                                               OsiSolverInterface *lp_solver) const {
    const vector <int> &indices = constraint.get_indices();
    const vector <double> &coefficients = constraint.get_coefficients();
    int num_entries = indices.size();
    assert(coefficients.size() == num_entries);
    CoinPackedVector *coin_vector = new CoinPackedVector(false);
    coin_vector->reserve(num_entries);
    for (size_t i = 0; i < num_entries; ++i) {
        coin_vector->insert(indices[i], translate_infinity(coefficients[i], lp_solver));
    }
    return coin_vector;
}

CoinColumns::CoinColumns(const vector<LPVariable> &variables,
                         OsiSolverInterface *lp_solver) {
    num_cols = variables.size();
    lower_bounds = new double[num_cols];
    upper_bounds = new double[num_cols];
    objective = new double[num_cols];
    for (size_t i = 0; i < num_cols; ++i) {
        const LPVariable &variable = variables[i];
        lower_bounds[i] = translate_infinity(variable.lower_bound, lp_solver);
        upper_bounds[i] = translate_infinity(variable.upper_bound, lp_solver);
        objective[i] = translate_infinity(variable.objective, lp_solver);
    }
    owns_data = true;
}

void CoinColumns::release_ownership() {
    owns_data = false;
}

CoinColumns::~CoinColumns() {
    if (owns_data) {
        delete[] lower_bounds;
        delete[] upper_bounds;
        delete[] objective;
    }
}
}
