#ifndef PDBS_PDB_HEURISTIC_H
#define PDBS_PDB_HEURISTIC_H

#include "../heuristic.h"

#include <vector>

class AbstractOperator {
    /*
    This class represents an abstract operator how it is needed for the regression search performed during the
    PDB-construction. As all abstract states are represented as a number, abstract operators don't have "usual"
    effects but "hash effects", i.e. the change (as number) the abstract operator implies on a given abstract state.
    */

    int cost;
    int op_id;

    // Preconditions for the regression search, corresponds to normal effects and prevail of concrete operators
    std::vector<std::pair<int, int> > regression_preconditions;

    // Effect of the operator during regression search on a given abstract state number
    size_t hash_effect;
public:
    /* Abstract operators are built from concrete operators. The parameters follow the usual name convention of SAS+
       operators, meaning prevail, preconditions and effects are all related to progression search. */
    AbstractOperator(const std::vector<std::pair<int, int> > &prevail,
                     const std::vector<std::pair<int, int> > &preconditions,
                     const std::vector<std::pair<int, int> > &effects, int cost, int op_id,
                     const std::vector<size_t> &hash_multipliers);
    ~AbstractOperator();

    // Returns variable value pairs which represent the preconditions of the abstract operator in a regression search
    const std::vector<std::pair<int, int> > &get_regression_preconditions() const {return regression_preconditions; }

    // Returns the effect of the abstract operator in form of a value change (+ or -) to an abstract state index
    size_t get_hash_effect() const {return hash_effect; }

    // Returns the cost of the abstract operator (same as the cost of the original concrete operator)
    int get_cost() const {return cost; }
    int get_op_id() const {return op_id; }
    void dump(const std::vector<int> &pattern) const;
};

// TODO: some duplication with AbstractTransition from M&S code
class AbstractPDBTransition {
public:
    int op_id;
    int from_state_index;
    int to_state_index;

    AbstractPDBTransition(int op, int from, int to)
        : op_id(op),
          from_state_index(from),
          to_state_index(to) {
    }
};

// Implements a single PDB
class Operator;
class State;
class PDBHeuristic : public Heuristic {
    bool store_transition_system;
    std::vector<int> *abstract_goal_states;
    std::vector<AbstractPDBTransition> *abstract_transitions;

    std::vector<int> pattern;

    // can be specified to be different from the normal operator costs. this is useful for action cost partitioning
    std::vector<int> operator_costs;

    std::vector<bool> relevant_operators; // stores for all operators whether they are relevant to this PDB or not
    size_t num_states; // size of the PDB

    // concrete variable are mapped to abstract variables in the order they appear in pattern
    std::vector<int> variable_to_index;

    // final h-values for abstract-states. dead-ends are represented by numeric_limits<int>::max()
    std::vector<int> distances;

    std::vector<size_t> hash_multipliers; // multipliers for each variable for perfect hash function
    void verify_no_axioms_no_cond_effects() const; // we support SAS+ tasks only

    /* Recursive method; called by build_abstract_operators.
       In the case of a precondition with value = -1 in the conrete operator, all mutliplied out abstract
       operators are computed, i.e. for all possible values of the variable (with precondition = -1),
       one abstract operator with a conrete value (!= -1) is computed. */
    void multiply_out(int pos, int op_no, int cost, std::vector<std::pair<int, int> > &prev_pairs,
                      std::vector<std::pair<int, int> > &pre_pairs,
                      std::vector<std::pair<int, int> > &eff_pairs,
                      const std::vector<std::pair<int, int> > &effects_without_pre,
                      std::vector<AbstractOperator> &operators);

    /* Computes all abstract operators for a given concrete operator (by its global operator number). Initializes
       datastructures for initial call to recursive method multiyply_out. */
    void build_abstract_operators(int op_no, std::vector<AbstractOperator> &operators);

    /* Computes all abstract operators, builds the match tree (successor generator) and then does a Dijkstra regression
       search to compute all final h-values (stored in distances). */
    void create_pdb();

    // Sets the pattern for the PDB and initializes hash_multipliers and num_states.
    void set_pattern(const std::vector<int> &pattern);

    /* For a given abstract state (given as index), the according values for each variable in the state are computed
       and compared with the given pairs of goal variables and values. Returns true iff the state is a goal state. */
    bool is_goal_state(const size_t state_index, const std::vector<std::pair<int, int> > &abstract_goal) const;
protected:
    virtual void initialize();
    virtual int compute_heuristic(const State &state);
public:
    /* Important: It is assumed that the pattern (passed via Options) is small enough so that the number of
                  abstract states is below numeric_limits<int>::max()
       Parameters:
       dump:     If set to true, prints the construction time.
       op_costs: Can specify individual operator costs for each operator. This is useful for action cost
                 partitioning. If left empty, default operator costs are used.
       store_transition_system_: If set to true, the abstract transition sytem is kept in memory and
                 can be accessed by get_abstract_transitions() and get_abstract_goal_states() */
    PDBHeuristic(const Options &opts,
                 bool dump = true,
                 const std::vector<int> &op_costs = std::vector<int>(),
                 bool store_transition_system_ = false);
    virtual ~PDBHeuristic();

    // Returns the pattern (i.e. all variables used) of the PDB
    const std::vector<int> &get_pattern() const {return pattern; }

    // Returns the size (number of abstract states) of the PDB
    size_t get_size() const {return num_states; }

    // Estimate current memory usage
    size_t get_memory_usage() const;
    
    /* Returns the average h-value over all states, where dead-ends are ignored (they neither increase
       the sum of all h-values nor the total number of entries for the mean value calculation). If pattern
       is empty or all states are dead-ends, infinity is returned.
       Note: This is only calculated when called; avoid repeated calls to this method! */
    double compute_mean_finite_h() const;

    // Returns all operators affecting this PDB
    const std::vector<bool> &get_relevant_operators() const {return relevant_operators; }

    // HACK made public only for optimal cost partitioning of pdb heuristics
    /* The given concrete state is used to calculate the index of the according abstract state. This is only used
       for table lookup (distances) during search. */
    size_t hash_index(const State &state) const;
    const std::vector<int> *get_abstract_goal_states() const {return abstract_goal_states; }
    const std::vector<AbstractPDBTransition> *get_abstract_transitions() const {return abstract_transitions; }
    void clear_transition_system() {
        delete abstract_goal_states;
        abstract_goal_states = 0;
        delete abstract_transitions;
        abstract_transitions = 0;
    }
};

#endif
