#ifndef PDBS_PATTERN_GENERATION_HASLUM_H
#define PDBS_PATTERN_GENERATION_HASLUM_H

#include "../operator_cost.h"
#include "../option_parser.h"
#include "../timer.h"
#include "../globals.h"
#include "../causal_graph.h"

#include <map>
#include <vector>
#include <set>

class Options;
class CanonicalPDBsHeuristic;
class PDBHeuristic;
class State;

struct PatternOrder {

    bool operator()(const std::vector<int> &p1, const std::vector<int> &p2) const {
        // Prefer smaller patterns 
        if (p1.size() != p2.size()) {
            return p1.size() < p2.size();
        }

        // Prefer patterns with higher connectivity
        int c1 = 0, c2 = 0;
        for (int i = 0; i < p1.size(); ++i) {
            c1 += g_causal_graph->get_predecessors(p1[i]).size();
            c1 += g_causal_graph->get_successors(p1[i]).size();
        }
        for (int i = 0; i < p2.size(); ++i) {
            c2 += g_causal_graph->get_predecessors(p2[i]).size();
            c2 += g_causal_graph->get_successors(p2[i]).size();
        }
        if (c1 != c2) {
            return c1 > c2;
        }

        // Enforce strict order
        return p1 < p2;
    }
};

// Implementation of the pattern generation algorithm by Haslum et al.

class PatternGenerationHaslum {
    const int pdb_max_size; // maximum number of states for each pdb
    const int collection_max_size; // maximum added size of all pdbs
    const int num_samples;
    const int min_improvement; // minimal improvement required for hill climbing to continue search
    const OperatorCost cost_type;
    const bool dominance_pruning;
    const int time_limit;
    const int iteration_limit;
    const int candidate_var_diff;
    const float auto_limit;
    const int min_low;
    const Options opt;
    const bool prune_old_patterns, remove_bad_looking_patterns, bad_time_limit, ffsampling;
    CanonicalPDBsHeuristic *current_heuristic;

    // For each variable v, store a list of goal variables g where some action has an effect on both v and g.
    std::vector<std::vector<int> > goal_common_action_effects;

    int num_rejected; // for stats only

    /* For the given pattern, all possible extensions of the pattern by one relevant variable
       are inserted into candidate_patterns. This may generate duplicated patterns. */
    void generate_candidate_patterns(const std::vector<int> &pattern,
            std::set<std::vector<int>, PatternOrder > &candidate_patterns, int recurse = 0, int last_var = -1);

    /* Performs num_samples random walks with a lenght (different for each random walk) chosen
       according to a binomial distribution with n = 4 * solution depth estimate and p = 0.5,
       starting from the initial state. In each step of a random walk, a random operator is taken
       and applied to the current state. If a dead end is reached or no more operators are
       applicable, the walk starts over again from the initial state. At the end of each random
       walk, the last state visited is taken as a sample state, thus totalling exactly
       num_samples of sample states. */
    void sample_states(std::vector<State> &samples, double average_operator_costs);

    /* Returns true iff the h-value of the new pattern (from pdb_heuristic) plus the h-value of all
       maximal additive subsets from the current pattern collection heuristic if the new pattern was
       added to it is greater than the the h-value of the current pattern collection. */
    bool is_heuristic_improved(PDBHeuristic *pdb_heuristic, const State &sample,
            const std::vector<std::vector<PDBHeuristic *> > &max_additive_subsets);

    /* This is the core algorithm of this class. As soon as after an iteration, the improvement (according
       to the "counting approximation") is smaller than the minimal required improvement, the search is
       stopped. This method uses a vector to store PDBs to avoid recomputation of the same PDBs later.
       This is quite a large time gain, but may use too much memory. Also a set is used to store all
       patterns in their "normal form" for duplicate detection.
       TODO: This method computes all PDBs already for candidate iteration, but for each call of
       add_pattern for the current CanonicalPDBsHeuristic, only the pattern is passed as an argument
       and in CanonicalPDBsHeuristic, the PDB is *again* built. One could possibly avoid this by
       passing the PDB and adapt CanonicalPDBsHeuristic accordingly. */
    void hill_climbing(double average_operator_costs, std::set<std::vector<int>, PatternOrder > &initial_candidate_patterns);

    /* Initializes everything for the hill climbing algorithm. Note that the initial pattern collection
       (consisting of exactly one PDB for each goal variable) may break the maximum collection size limit,
       if the latter is set too small or if there are many goal variables with a large domain. */
    void initialize();

    bool is_time_up(Timer &timer);

public:
    PatternGenerationHaslum(const Options &opts);
    virtual ~PatternGenerationHaslum();

    /* Returns the CanonicalPDBsHeuristic created by PatternGenerationHaslum.
       Important: caller owns the returned pointer and has to take care of its deletion. */
    CanonicalPDBsHeuristic *get_pattern_collection_heuristic() const {
        return current_heuristic;
    }
    static void create_options(OptionParser &parser);
    static void sanity_check_options(OptionParser &parser, Options &opts);
};

#endif
