#include "pdb_lp_heuristic.h"
#include "globals.h"
#include "lm_cut_heuristic.h"
#include "operator.h"
#include "operator_cost.h"
#include "pdbs/pattern_generation_haslum.h"
#include "pdbs/pattern_generation_systematic.h"
#include "pdbs/canonical_pdbs_heuristic.h"
#include "pdbs/pdb_heuristic.h"
#include "plugin.h"

using namespace std;

PDBLPHeuristic::PDBLPHeuristic(const Options &opts)
    : Heuristic(opts), lmcut(0), merge_operators(opts.get<bool>("merge_operators")), canonical(0), size(0) {
    vector<vector<int> > patterns;
    lp_solver = create_lp_solver(LPSolverType(opts.get_enum("lpsolver")));
    lp_solver->messageHandler()->setLogLevel(0);

    // All of the following options for the lp solver are not yet completely evaluated
    // but some initial tests have shown no evidence of improvement.
/*
    OsiClpSolverInterface * osiclp = dynamic_cast< OsiClpSolverInterface*> (lp_solver_interface);
    if (osiclp) {
        osiclp->setupForRepeatedUse(0,1);
        ClpSolve solve_options;
        solve_options.setSolveType(ClpSolve::useDual);
        osiclp->setSolveOptions(solve_options);
    }
*/
    if (opts.get<bool>("use_lmcut")) {
        lmcut = new LandmarkCutHeuristic(opts);
    }
    if (opts.contains("patterns")) {
        patterns = opts.get<vector<vector<int> > >("patterns");
    } else if (opts.contains("systematic") && opts.get<int>("systematic")) {
        Options generator_opts;
        generator_opts.set<int>("pattern_max_size", opts.get<int>("systematic"));
        generator_opts.set<bool>("dominance_pruning", opts.get<bool>("dominance_pruning"));
        if (opts.contains("prune_irrelevant_patterns") && opts.get<bool>("prune_irrelevant_patterns")) {
            PatternGenerationSystematic pattern_generator(generator_opts);
            patterns = pattern_generator.get_patterns();
        } else {
            PatternGenerationSystematicNaive pattern_generator(generator_opts);
            patterns = pattern_generator.get_patterns();
        }
    }

    if (!patterns.empty()) {
        for (size_t i = 0; i < patterns.size(); ++i) {
            if (i % 1000 == 0 && i > 0) {
                cout << "Generated " << i << "/" << patterns.size() << " PDBs" << endl;
            }
            Options pdb_opts;
            pdb_opts.set<int>("cost_type", cost_type);
            pdb_opts.set<vector<int> >("pattern", patterns[i]);
            PDBHeuristic *pdb = new PDBHeuristic(pdb_opts, false);
            size += pdb->get_size();
            heuristics.push_back(pdb);
            borrowed_heuristics = false;
        }
    } else {
        // compute pattern collection
        PatternGenerationHaslum pgh(opts);
        canonical = pgh.get_pattern_collection_heuristic();
        heuristics = canonical->get_pattern_databases();
        borrowed_heuristics = true;
        size = canonical->get_size();
    }
    generateLP();
}

PDBLPHeuristic::~PDBLPHeuristic() {
    delete lp_solver;
    clear_pdbs();
}

void PDBLPHeuristic::initialize() {
}


void PDBLPHeuristic::clear_pdbs() {
    delete canonical;
    canonical = 0;
    if (!borrowed_heuristics) {
        for (size_t i = 0; i < heuristics.size(); ++i) {
            delete heuristics[i];
        }
    }
    heuristics.clear();
    size = 0;
}


void PDBLPHeuristic::set_pdbs(std::vector<PDBHeuristic *> heuristics_, bool borrowed_heuristics_) {
    clear_pdbs();
    borrowed_heuristics = borrowed_heuristics_;
    heuristics.insert(heuristics.end(), heuristics_.begin(), heuristics_.end());
    for (size_t i = 0; i < heuristics.size(); ++i) {
        size += heuristics[i]->get_size();
    }
    generateLP();
}

void PDBLPHeuristic::generateLP() {
    // compute operator partitioning
    int num_partitions = 0;
    vector<int> partitioning = vector<int>(g_operators.size(), 0);
    if (merge_operators) {
        for (size_t i = 0; i < heuristics.size(); ++i) {
            vector<int> block_mapping_rel = vector<int>(g_operators.size(), -1);
            vector<int> block_mapping_not_rel = vector<int>(g_operators.size(), -1);
            PDBHeuristic *h = heuristics[i];
            num_partitions = 0;
            const std::vector<bool> &rel_ops = h->get_relevant_operators();
            for (size_t op_id = 0; op_id < g_operators.size(); ++op_id) {
                if (rel_ops[op_id]) {
                    int curr_block = partitioning[op_id];
                    if (block_mapping_rel[curr_block] == -1) {
                        block_mapping_rel[curr_block] = num_partitions;
                        ++num_partitions;
                    }
                    partitioning[op_id] = block_mapping_rel[curr_block];
                } else {
                    int curr_block = partitioning[op_id];
                    if (block_mapping_not_rel[curr_block] == -1) {
                        block_mapping_not_rel[curr_block] = num_partitions;
                        ++num_partitions;
                    }
                    partitioning[op_id] = block_mapping_not_rel[curr_block];
                }
            }
        }
    } else {
        for (size_t i = 0; i < g_operators.size(); ++i) {
            partitioning[i] = i;
        }
        num_partitions = g_operators.size();
    }

    int num_cols = num_partitions;
    int num_rows = heuristics.size();
    double *objective = new double[num_cols];    //objective coefficients
    double *col_lb = new double[num_cols];       //column lower bounds
    double *col_ub = new double[num_cols];       //column upper bounds

    // Minimize x_0 + ... + x_{n_cols - 1}
    fill(objective, objective + num_cols, 1);

    // Variable lower/upper bounds
    fill(col_lb, col_lb + num_cols, 0);
    fill(col_ub, col_ub + num_cols, lp_solver->getInfinity());

    //Define the constraint matrix.
    CoinPackedMatrix *matrix = new CoinPackedMatrix(false, 0, 0);
    matrix->setDimensions(0, num_cols);
    double *row_entries = new double[num_cols];
    for (size_t i = 0; i < num_rows; ++i) {
        PDBHeuristic *h = heuristics[i];
        const std::vector<bool> &rel_ops = h->get_relevant_operators();
        CoinPackedVector row;
        fill(row_entries, row_entries + num_cols, 0.0);
        for (size_t op_id = 0; op_id < g_operators.size(); ++op_id) {
            if (rel_ops[op_id])
                row_entries[partitioning[op_id]] = 1.0;
        }
        row.setFullNonZero(num_cols, row_entries);
        matrix->appendRow(row);
    }

    double *row_ub = new double[num_rows]; //the row upper bounds
    fill(row_ub, row_ub + num_rows, lp_solver->getInfinity());

    double *row_lb = new double[num_rows]; //the row lower bounds
    fill(row_lb, row_lb + num_rows, 0);

    landmark_count = 0;
    lp_solver->loadProblem(*matrix, col_lb, col_ub, objective,
                           row_lb, row_ub);
    // after an initial solve we can always use resolve for solving a modified
    // version of the LP
    lp_solver->initialSolve();

    // Clean up
    delete[] objective;
    delete[] col_lb;
    delete[] col_ub;
    delete[] row_entries;
    delete[] row_ub;
    delete[] row_lb;
    delete matrix;
    
    cout << "Cols:       " << lp_solver->getNumCols() << endl;
    cout << "Rows:        " << lp_solver->getNumRows() << endl;
    cout << "Elements:   " << lp_solver->getNumElements() << endl;
}

void PDBLPHeuristic::add_pattern(const std::vector<int> &pattern) {
    Options pdb_opts;
    pdb_opts.set<int>("cost_type", cost_type);
    pdb_opts.set<vector<int> >("pattern", pattern);
    PDBHeuristic *pdb = new PDBHeuristic(pdb_opts, false);
    size += pdb->get_size();
    heuristics.push_back(pdb);
    generateLP();
}

int PDBLPHeuristic::compute_heuristic(const State &state) {
    int heuristic_rows = heuristics.size();
    for (size_t i = 0; i < heuristic_rows; ++i) {
        PDBHeuristic *h = heuristics[i];
        h->evaluate(state);
        if (h->is_dead_end())
            return DEAD_END;

        int h_val = h->get_heuristic();
        lp_solver->setRowLower(i, h_val);
    }

    if (lmcut) {
        lmcut->evaluate(state);
        if (lmcut->is_dead_end()) {
            return DEAD_END;
        }
        if (landmark_count > 0) {
            int *old_landmark_rows = new int[landmark_count];
            for (int i = 0; i < landmark_count; ++i) {
                old_landmark_rows[i] = heuristic_rows + i;
            }
            lp_solver->deleteRows(landmark_count, old_landmark_rows);
            delete old_landmark_rows;
        }
        const vector<pair<vector<const Operator *>, int> > &landmarks = lmcut->get_last_landmarks();
        int num_rows = landmarks.size();
        int num_cols = g_operators.size();
        landmark_count = num_rows;
        CoinPackedVectorBase **rows = new CoinPackedVectorBase *[num_rows];
        double *row_entries = new double[num_cols];
        double *row_lbs = new double[num_rows];
        double *row_ubs = new double[num_rows];
        fill(row_ubs, row_ubs + num_rows, lp_solver->getInfinity());
        for (int lm = 0; lm < num_rows; ++lm) {
            const vector<const Operator *> &landmark_ops = landmarks[lm].first;
            int cost = landmarks[lm].second;
            CoinPackedVector *row = new CoinPackedVector();
            fill(row_entries, row_entries + num_cols, 0.0);
            for (size_t op_id = 0; op_id < landmark_ops.size(); ++op_id) {
                const Operator *op = landmark_ops[op_id];
                row_entries[op->get_operator_index()] = cost / (double)op->get_cost();
            }
            row->setFullNonZero(num_cols, row_entries);
            row_lbs[lm] = cost;
            rows[lm] = row;
        }
        lp_solver->addRows(landmark_count, rows, row_lbs, row_ubs);
        for (int lm = 0; lm < landmark_count; ++lm) {
            delete rows[lm];
        }
        delete[] rows;
        delete[] row_entries;
        delete[] row_lbs;
        delete[] row_ubs;
    }
    
    lp_solver->resolve();
    double h_val = lp_solver->getObjValue();
    // TODO avoid code duplication with landmark count heuristic
    double epsilon = 0.01;
    return ceil(h_val - epsilon);
}

static ScalarEvaluator *_parse(OptionParser &parser) {
    PatternGenerationHaslum::create_options(parser);
    parser.add_option<int>("systematic",
                           "Systematically generate all patterns with up to n variables instead of using PatternGenerationHaslum.",
                           "0");
    parser.add_option<bool>("prune_irrelevant_patterns",
                            "Prune irrelevant patterns before building the LP.",
                            "true");
    parser.add_option<bool>("merge_operators",
                            "Merge operators in the same equivalence class into one variable (not possible if landmarks are used)",
                            "true");
    parser.add_option<bool>("use_lmcut",
                            "Compute landmarks with lmcut in each state and include them in the LP",
                            "false");

    add_lp_solver_option_to_parser(parser);
    Heuristic::add_options_to_parser(parser);
    Options opts = parser.parse();
    if (parser.help_mode())
        return 0;
    if (opts.get<bool>("use_lmcut") && opts.get<bool>("merge_operators")) {
        cerr << "Merging operators is not possible when landmarks are used." << endl;
        abort();
    }

    PatternGenerationHaslum::sanity_check_options(parser, opts);

    if (parser.dry_run())
        return 0;

    return new PDBLPHeuristic(opts);
}

static Plugin<ScalarEvaluator> _plugin("pdb_lp", _parse);
