#include "operator_count_lp.h"

#include "lp_constraint_collection.h"

#include "../equivalence_relation.h"
#include "../globals.h"
#include "../operator.h"
#include "../exact_timer.h"
#include "../utilities.h"

#include <CoinPackedMatrix.hpp>
#include <CoinPackedVector.hpp>

using namespace std;

namespace pho {

	OperatorCountLP::OperatorCountLP(const LPConstraintCollection &constraint_collection,
		bool merge_lp_variables, LPSolverType solver_type,
		OperatorCost cost_type_)
	: cost_type(cost_type_),
	is_solved(false),
	is_objective_function_modified(false),
	has_temporary_constraints(false),
	merged_variables(0)
	{
		lp_solver = create_lp_solver(solver_type);
		lp_solver->passInMessageHandler(new ErrorCatchingCoinMessageHandler());
		//    lp_solver->messageHandler()->setLogLevel(0);

		// Create columns (LP variables)
		const vector<LPVariable> &variables = constraint_collection.get_variables();
		CoinColumns coin_columns(variables, lp_solver);
		for (size_t i = 0; i < variables.size(); ++i) {
			int expected_objective = 0;
			if (i < g_operators.size()) {
				expected_objective = get_adjusted_action_cost(g_operators[i], cost_type);
			}
			if (variables[i].objective != expected_objective) {
				is_objective_function_modified = true;
				cout << "Objective function is modified" << endl;
				break;
			}
		}
		// Create rows (constraints)
		const vector<LPConstraint> &constraints = constraint_collection.get_constraints();
		num_permanent_constraints = constraints.size();
		CoinRows coin_rows(constraints, lp_solver);

		// Create matrix and add constraints
		CoinPackedMatrix *matrix = new CoinPackedMatrix(false, 0, 0);
		matrix->setDimensions(0, coin_columns.num_cols);
		matrix->appendRows(coin_rows.num_rows, coin_rows.rows);

		try {
			lp_solver->setObjSense(1);
			lp_solver->assignProblem(matrix,
				coin_columns.lower_bounds, coin_columns.upper_bounds,
				coin_columns.objective,
				coin_rows.lower_bounds, coin_rows.upper_bounds);
			// NOTE Using assignProblem instead of loadProblem, the ownership of the
			// data is transfered to the LP solver. It will delete the rows, columns
			// and the coefficient matrix.
			coin_columns.release_ownership();
			coin_rows.release_ownership();
		} catch (CoinError &error) {
			handle_coin_error(error);
		}

		print_statistics("at creation");
		if (merge_lp_variables) {
			ExactTimer merge_timer;
			merge_variables(constraint_collection);
			cout << "LP variable merging took " << merge_timer << endl;
			print_statistics("after merge");
		}

		// After an initial solve we can always use resolve for solving a modified
		// version of the LP.
		ExactTimer initial_solve_timer;
		try {
			lp_solver->initialSolve();
		} catch (CoinError &error) {
			handle_coin_error(error);
		}
		is_solved = true;
		cout << "LP initial solve took " << initial_solve_timer << endl;

		/*constraint_bound = new int[lp_solver->getNumRows()];
		for (int i = 0; i < lp_solver->getNumRows(); ++i) {
			constraint_bound[i] = 0;
		}*/
	}

	OperatorCountLP::~OperatorCountLP()
	{
		print_statistics("at destruction");
		delete merged_variables;
		delete lp_solver;
	}

	int OperatorCountLP::get_num_variables() const
	{
		return lp_solver->getNumCols();
	}

	int OperatorCountLP::get_num_zero_variables() const
	{
		if (merged_variables) {
			return merged_variables->get_num_explicit_elements() - merged_variables->get_num_explicit_blocks();
		}
		return 0;
	}

	int OperatorCountLP::get_num_constraints() const
	{
		return lp_solver->getNumRows();
	}

	void OperatorCountLP::print_statistics(string location) const
	{
		cout << "LP variables (" << location << "): " << get_num_variables() << endl;
		if (merged_variables) {
			cout << "LP variables restricted to 0 by merging (" << location << "): "
				<< get_num_zero_variables() << endl;
		}
		cout << "LP constraints (" << location << "): " << get_num_constraints() << endl;
	}

	void OperatorCountLP::merge_variables(const LPConstraintCollection &constraint_collection)
	{
		/*
		   Detect LP variables x_1, ..., x_k that always occur as a multiple of the
		   same weighted sum d*(a_1*x_1 + ... + a_k*x_k). The factor d can vary
		   between different constraints, but the weighted sum must be the same in
		   every constraint that mentions any of the x_i and the objective function
		   if it mentions any of the x_i.

		   The first occurrence of every x_i in either the objective function or
		   a constraint is used as a reference value. If the reference for a column
		   is 0, this means that the column has no entries and can be removed.

		   Two columns are equivalent (i.e. can be merged), according to a row
		   if the relation of the coefficients in the row and their reference
		   value is the same in both columns. If columns 1, ..., k are equivalent
		   according to all rows and the objective function, they can be replaced by
		   a new variable that represents reference[1]*x_1 + ... + reference[1]*x_k
		   The coefficient in row r for this new variable is then
			   d = coefficient[r][1] / reference[1]

		   We only consider variables with bounds (0, infinity) for merging and can
		   merge variables by setting the upper bounds of all but one variable per
		   equivalence class to 0.
		 */
		const vector<LPVariable> &variables = constraint_collection.get_variables();
		const vector<LPConstraint> &constraints = constraint_collection.get_constraints();
		int num_vars = variables.size();
		merged_variables = new EquivalenceRelation(num_vars);
		reference_values.resize(num_vars, 0);
		refine_merged_variables(variables);
		refine_merged_variables(constraints);
	}

	// The reference value for all variables mentioned in the objective is their
	// value in the objective function, so all these variables are equivalent
	// according to the objective. All other variables will have a ratio of 0
	// with their reference and are also equivalent.
	// EquivalenceRelation can handle sparse definition, so we only need to specify
	// one class.

	void OperatorCountLP::refine_merged_variables(const vector<LPVariable> &variables)
	{
		assert(merged_variables);
		Block mentioned_variables;
		for (size_t i = 0; i < variables.size(); ++i) {
			reference_values[i] = variables[i].objective;
			if (variables[i].objective != 0) {
				mentioned_variables.insert(i);
			}
		}
		list<Block> classes;
		classes.insert(classes.end(), mentioned_variables);
		EquivalenceRelation objective_equivalence(variables.size(), classes);
		merged_variables->refine(objective_equivalence);
		has_modified_merge_representatives = true;
	}

	void OperatorCountLP::refine_merged_variables(const vector<LPConstraint> &constraints)
	{
		assert(merged_variables);
		int num_blocks_before = merged_variables->get_num_blocks();
		for (size_t i = 0; i < constraints.size(); ++i) {
			const LPConstraint &constraint = constraints[i];
			const vector<int> &indices = constraint.get_indices();
			const vector<double> &coefficients = constraint.get_coefficients();
			vector<pair<double, int> > row_class_labels;
			for (size_t j = 0; j < indices.size(); ++j) {
				int var_id = indices[j];
				double coefficient = coefficients[j];
				double class_label;
				if (coefficient == 0) {
					class_label = 0;
				} else if (reference_values[var_id] == 0) {
					reference_values[var_id] = coefficient;
					class_label = 1;
				} else {
					// NOTE that we use the reciprocal here because in the LPs we
					// currently consider reference_values can be large and
					// coefficients are usually small.
					class_label = reference_values[var_id] / coefficient;
				}
				row_class_labels.push_back(make_pair(class_label, var_id));
			}
			int num_vars = merged_variables->get_num_elements();
			EquivalenceRelation constraint_equivalence =
				EquivalenceRelation::from_labels<double, DoubleEpsilonEquality>(num_vars, row_class_labels);
			merged_variables->refine(constraint_equivalence);
		}
		if (merged_variables->get_num_blocks() > num_blocks_before) {
			has_modified_merge_representatives = true;
		}
	}

	void OperatorCountLP::set_merged_variable_bounds()
	{
		assert(merged_variables);
		try {
			for (BlockListConstIter it_block = merged_variables->begin();
				it_block != merged_variables->end(); ++it_block) {
				const Block &block = *it_block;
				bool representative = true;
				for (ElementListConstIter it_element = block.begin();
					it_element != block.end(); ++it_element) {
					if (representative) {
						lp_solver->setColUpper(*it_element, lp_solver->getInfinity());
						representative = false;
					} else {
						lp_solver->setColUpper(*it_element, 0);
					}
				}
			}
			has_modified_merge_representatives = false;
		} catch (CoinError &error) {
			handle_coin_error(error);
		}
	}

	void OperatorCountLP::set_permanent_constraint_lower_bound(int constraint_id, double lower_bound)
	{
		assert(constraint_id < num_permanent_constraints);
		try {
			lp_solver->setRowLower(constraint_id, translate_infinity(lower_bound, lp_solver));
		} catch (CoinError &error) {
			handle_coin_error(error);
		}
		is_solved = false;
	}

	void OperatorCountLP::set_permanent_constraint_upper_bound(int constraint_id, double upper_bound)
	{
		assert(constraint_id < num_permanent_constraints);
		try {
			lp_solver->setRowUpper(constraint_id, translate_infinity(upper_bound, lp_solver));
		} catch (CoinError &error) {
			handle_coin_error(error);
		}
		is_solved = false;
	}

	void OperatorCountLP::add_temporary_constraints(const std::vector<LPConstraint> &constraints)
	{
		if (constraints.empty()) {
			return;
		}
		if (merged_variables) {
			refine_merged_variables(constraints);
		}
		CoinRows coin_rows(constraints, lp_solver);
		try {
			lp_solver->addRows(coin_rows.num_rows, coin_rows.rows,
				coin_rows.lower_bounds, coin_rows.upper_bounds);
		} catch (CoinError &error) {
			handle_coin_error(error);
		}
		has_temporary_constraints = true;
		is_solved = false;
	}

	void OperatorCountLP::solve()
	{
		if (merged_variables && has_modified_merge_representatives) {
			set_merged_variable_bounds();
		}
		try {
			lp_solver->resolve();
			if (lp_solver->isAbandoned()) {
				// The documentation of OSI is not very clear here but memory seems
				// to be the most common cause for this in our case.
				cerr << "Abandoned LP during resolve. "
					<< "Reasons include \"numerical difficulties\" and running out of memory." << endl;
				exit_with(EXIT_CRITICAL_ERROR);
			}
			is_solved = true;
		} catch (CoinError &error) {
			handle_coin_error(error);
		}
	}
	
	size_t OperatorCountLP::update_binding_count(vector<pair<int,int> > &activityCounts) const{
		size_t num = lp_solver->getNumRows();
		const double *row_activity = lp_solver->getRowActivity();
		const double *row_rhs = lp_solver->getRightHandSide();
		size_t bound_count = 0;
		for (size_t i = 0; i < num; ++i) {
			if (row_activity[i] == row_rhs[i]) {
				++activityCounts[i].second;
				++bound_count;
			}
		}
		return bound_count;
	}
	
	void OperatorCountLP::remove_constraints(const int num, const int *rowIndices){
		lp_solver->deleteRows(num, rowIndices);
		num_permanent_constraints -= num;
	}

	bool OperatorCountLP::has_feasible_solution() const
	{
		assert(is_solved);
		bool has_feasible_solution = false;
		try {
			has_feasible_solution = !lp_solver->isProvenPrimalInfeasible();
			// NOTE We do not check isProvenOptimal in release code because the LP
			//      should never be unbounded.
			assert(lp_solver->isProvenOptimal());
		} catch (CoinError &error) {
			handle_coin_error(error);
		}
		return has_feasible_solution;
	}

	int OperatorCountLP::get_heuristic_value() const
	{
		assert(is_solved);
		double heuristic_value = 0.0;
		try {
			if (is_objective_function_modified) {
				heuristic_value = 0.0;
				const double *solution = lp_solver->getColSolution();
				for (size_t i = 0; i < g_operators.size(); ++i) {
					int op_cost = get_adjusted_action_cost(g_operators[i], cost_type);
					heuristic_value += solution[i] * op_cost;
				}
			} else {
				heuristic_value = lp_solver->getObjValue();
			}
		} catch (CoinError &error) {
			handle_coin_error(error);
		}
		// TODO avoid code duplication with landmark count heuristic
		double epsilon = 0.01;
		return ceil(heuristic_value - epsilon);
	}

	void OperatorCountLP::remove_temporary_constraints()
	{
		if (has_temporary_constraints) {
			try {
				lp_solver->restoreBaseModel(num_permanent_constraints);
			} catch (CoinError &error) {
				handle_coin_error(error);
			}
			has_temporary_constraints = false;
		}
	}
}
