#include "pattern_generation_haslum.h"

#include "canonical_pdbs_heuristic.h"
#include "pdb_heuristic.h"

#include "../globals.h"
#include "../legacy_causal_graph.h"
#include "../causal_graph.h"
#include "../operator.h"
#include "../option_parser.h"
#include "../plugin.h"
#include "../rng.h"
#include "../state.h"
#include "../state_registry.h"
#include "../successor_generator.h"
#include "../timer.h"
#include "../utilities.h"

#include "../ff_heuristic.h"

#include <algorithm>
#include <cassert>
#include <cmath>
#include <cstdlib>
#include <string>
#include <vector>

using namespace std;

PatternGenerationHaslum::PatternGenerationHaslum(const Options &opts)
: pdb_max_size(opts.get<int>("pdb_max_size")),
collection_max_size(opts.get<int>("collection_max_size")),
num_samples(opts.get<int>("num_samples")),
min_improvement(opts.get<int>("min_improvement")),
cost_type(OperatorCost(opts.get<int>("cost_type"))),
dominance_pruning(opts.get<bool>("dominance_pruning")),
time_limit(opts.get<int>("time_limit")),
iteration_limit(opts.get<int>("iteration_limit")),
auto_limit(opts.get<float>("auto_limit")),
candidate_var_diff(opts.get<int>("candidate_var_diff")),
min_low(opts.get<int>("min_low")),
prune_old_patterns(opts.get<bool>("prune_old_patterns")),
bad_time_limit(opts.get<bool>("bad_time_limit")),
remove_bad_looking_patterns(opts.get<bool>("remove_bad_looking_patterns")),
ffsampling(opts.get<bool>("ffsampling")),
opt(opts) {
    Timer timer;
    initialize();
    cout << "Pattern generation (Haslum et al.) time: " << timer << endl;
}

PatternGenerationHaslum::~PatternGenerationHaslum() {
}

size_t est_pattern_mem(const vector<int> &pattern) {
    size_t num_states = 1;
    for (size_t i = 0; i < pattern.size(); ++i) {
        num_states *= g_variable_domain[pattern[i]];
    }
    size_t size = 0;
    size += pattern.capacity() * sizeof (int); // pattern
    size += g_operators.capacity() * sizeof (int); // operator_cost
    size += g_operators.capacity() * sizeof (bool); // relevant_operators
    size += g_variable_name.capacity() * sizeof (int); // variable_to_index
    size += num_states * sizeof (int); // distances
    size += pattern.capacity() * sizeof (size_t); // hash_multipliers
    return size;
}

void PatternGenerationHaslum::generate_candidate_patterns(const vector<int> &pattern,
        set<vector<int>, PatternOrder > &candidate_patterns, int recurse, int last_var) {
    int current_size = current_heuristic->get_size();
    if (last_var == -1) {
        for (size_t i = 0; i < pattern.size(); ++i) {
            //cout << "Var: " << pattern[i] << endl;

            // causally relevant variables for current variable from pattern
            vector<int> rel_vars = g_legacy_causal_graph->get_predecessors(pattern[i]);
            //for(int ii=0;ii<rel_vars.size();++ii)cout << rel_vars[ii] << ", ";cout << endl;
            sort(rel_vars.begin(), rel_vars.end());
            vector<int> relevant_vars;
            assert_sorted_unique(rel_vars);
            assert_sorted_unique(pattern);
            // make sure we only use relevant variables which are not already included in pattern
            set_difference(rel_vars.begin(), rel_vars.end(), pattern.begin(), pattern.end(), back_inserter(relevant_vars));

            // add goal variables that occur as common effects with some variable in the pattern
            const vector<int> &common_goals = goal_common_action_effects[pattern[i]];
            //for(int ii=0;ii<common_goals.size();++ii)cout << common_goals[ii] << ", ";cout << endl;
            vector<int> relevant_goals;
            assert_sorted_unique(common_goals);
            assert_sorted_unique(pattern);
            set_difference(common_goals.begin(), common_goals.end(), pattern.begin(), pattern.end(), back_inserter(relevant_goals));

            vector<int> interesting_vars;
            assert_sorted_unique(relevant_vars);
            assert_sorted_unique(relevant_goals);
            set_union(relevant_vars.begin(), relevant_vars.end(), relevant_goals.begin(), relevant_goals.end(), back_inserter(interesting_vars));

            // 
            //vector<int> nvar = g_causal_graph->get_eff_to_pre(pattern[i]);
            //for(int ii=0;ii<nvar.size();++ii)cout << nvar[ii] << ", ";cout << endl;


            for (size_t j = 0; j < interesting_vars.size(); ++j) {
                // test against overflow and pdb_max_size
                if (current_size <= pdb_max_size / g_variable_domain[interesting_vars[j]]) {
                    // current_size * g_variable_domain[relevant_vars[j]] <= pdb_max_size
                    vector<int> new_pattern(pattern);
                    new_pattern.push_back(interesting_vars[j]);
                    sort(new_pattern.begin(), new_pattern.end());
                    candidate_patterns.insert(new_pattern);
                    if (recurse > 0) {
                        generate_candidate_patterns(new_pattern, candidate_patterns, recurse - 1, interesting_vars[j]);
                    }
                } else {
                    // [commented out the message because it might be too verbose]
                    // cout << "ignoring new pattern as candidate because it is too large" << endl;
                    num_rejected += 1;
                }
            }
        }
    } else {
        //cout << "Var: " << pattern[i] << endl;

        // causally relevant variables for current variable from pattern
        vector<int> rel_vars = g_legacy_causal_graph->get_predecessors(last_var);
        //for(int ii=0;ii<rel_vars.size();++ii)cout << rel_vars[ii] << ", ";cout << endl;
        sort(rel_vars.begin(), rel_vars.end());
        vector<int> relevant_vars;
        assert_sorted_unique(rel_vars);
        assert_sorted_unique(pattern);
        // make sure we only use relevant variables which are not already included in pattern
        set_difference(rel_vars.begin(), rel_vars.end(), pattern.begin(), pattern.end(), back_inserter(relevant_vars));

        // add goal variables that occur as common effects with some variable in the pattern
        const vector<int> &common_goals = goal_common_action_effects[last_var];
        //for(int ii=0;ii<common_goals.size();++ii)cout << common_goals[ii] << ", ";cout << endl;
        vector<int> relevant_goals;
        assert_sorted_unique(common_goals);
        assert_sorted_unique(pattern);
        set_difference(common_goals.begin(), common_goals.end(), pattern.begin(), pattern.end(), back_inserter(relevant_goals));

        vector<int> interesting_vars;
        assert_sorted_unique(relevant_vars);
        assert_sorted_unique(relevant_goals);
        set_union(relevant_vars.begin(), relevant_vars.end(), relevant_goals.begin(), relevant_goals.end(), back_inserter(interesting_vars));

        // 
        //vector<int> nvar = g_causal_graph->get_eff_to_pre(pattern[i]);
        //for(int ii=0;ii<nvar.size();++ii)cout << nvar[ii] << ", ";cout << endl;


        for (size_t j = 0; j < interesting_vars.size(); ++j) {
            // test against overflow and pdb_max_size
            if (current_size <= pdb_max_size / g_variable_domain[interesting_vars[j]]) {
                // current_size * g_variable_domain[relevant_vars[j]] <= pdb_max_size
                vector<int> new_pattern(pattern);
                new_pattern.push_back(interesting_vars[j]);
                sort(new_pattern.begin(), new_pattern.end());
                candidate_patterns.insert(new_pattern);
                if (recurse > 0) {
                    generate_candidate_patterns(new_pattern, candidate_patterns, recurse - 1, interesting_vars[j]);
                }
            } else {
                // [commented out the message because it might be too verbose]
                // cout << "ignoring new pattern as candidate because it is too large" << endl;
                num_rejected += 1;
            }
        }
    }
}

void PatternGenerationHaslum::sample_states(vector<State> &samples, double average_operator_cost) {
    const State &initial_state = g_initial_state();
    current_heuristic->evaluate(initial_state);
    assert(!current_heuristic->is_dead_end());

    int h = current_heuristic->get_heuristic();
    int n;
    if (h == 0) {
        n = 10;
    } else {
        // Convert heuristic value into an approximate number of actions
        // (does nothing on unit-cost problems).
        // average_operator_cost cannot equal 0, as in this case, all operators
        // must have costs of 0 and in this case the if-clause triggers.
        int solution_steps_estimate = int((h / average_operator_cost) + 0.5);
        n = 4 * solution_steps_estimate;
    }
    double p = 0.5;
    // The expected walk length is np = 2 * estimated number of solution steps.
    // (We multiply by 2 because the heuristic is underestimating.)

    int count_samples = 0;
    samples.reserve(num_samples);

    State current_state(initial_state);

    if (ffsampling) {
        FFHeuristic ffh(opt);
        ffh.evaluate(current_state);
        //cout << "FFH-init: " << ffh.get_heuristic() << endl;
        ++count_samples;
        samples.push_back(current_state);
        while (count_samples < num_samples) {
            State best_state(initial_state), test_state(initial_state);
            int best_h = -1;
            vector<const Operator *> applicable_ops;
            g_successor_generator->generate_applicable_ops(current_state, applicable_ops);
            if (applicable_ops.empty()) {
                //cout << "No more applicable operators" << endl;
                break;
            }
            for (int o = 0; o < applicable_ops.size(); ++o) {
                assert(applicable_ops[o]->is_applicable(current_state));
                // TODO for now, we only generate registered successors. This is a temporary state that
                // should should not necessarily be registered in the global registry: see issue386.
                test_state = g_state_registry->get_successor_state(current_state, *applicable_ops[o]);
                // if current state is a dead end, then restart with initial state

                ffh.evaluate(test_state);
                //cout << "FF eval: " << ffh.get_heuristic() << endl;
                if (!ffh.is_dead_end()) {
                    ++count_samples;
                    samples.push_back(test_state);
                    if (ffh.get_heuristic() < best_h || best_h == -1) {
                        best_h = ffh.get_heuristic();
                        best_state = test_state;
                    }
                }
            }
            if (best_h != -1) {
                current_state = best_state;
            } else {
                //cout << "Deadend" << endl;
                break;
            }
        }
        cout << "Samples generated from best first search: " << count_samples << endl;
    }

    for (; count_samples < num_samples; ++count_samples) {
        // calculate length of random walk accoring to a binomial distribution
        int length = 0;
        for (int j = 0; j < n; ++j) {
            double random = g_rng(); // [0..1)
            if (random < p)
                ++length;
        }

        // random walk of length length
        State current_state(initial_state);
        for (int j = 0; j < length; ++j) {
            vector<const Operator *> applicable_ops;
            g_successor_generator->generate_applicable_ops(current_state, applicable_ops);
            // if there are no applicable operators --> do not walk further
            if (applicable_ops.empty()) {
                break;
            } else {
                int random = g_rng.next(applicable_ops.size()); // [0..applicable_os.size())
                assert(applicable_ops[random]->is_applicable(current_state));
                // TODO for now, we only generate registered successors. This is a temporary state that
                // should should not necessarily be registered in the global registry: see issue386.
                current_state = g_state_registry->get_successor_state(current_state, *applicable_ops[random]);
                // if current state is a dead end, then restart with initial state
                current_heuristic->evaluate_dead_end(current_state);
                if (current_heuristic->is_dead_end())
                    current_state = initial_state;
            }
        }
        // last state of the random walk is used as sample
        samples.push_back(current_state);
    }
}

bool PatternGenerationHaslum::is_heuristic_improved(PDBHeuristic *pdb_heuristic,
        const State &sample,
        const vector<vector<PDBHeuristic *> > &max_additive_subsets) {
    pdb_heuristic->evaluate(sample);
    if (pdb_heuristic->is_dead_end()) {
        return true;
    }
    int h_pattern = pdb_heuristic->get_heuristic(); // h-value of the new pattern

    current_heuristic->evaluate(sample);
    int h_collection = current_heuristic->get_heuristic(); // h-value of the current collection heuristic
    for (size_t k = 0; k < max_additive_subsets.size(); ++k) { // for each max additive subset...
        int h_subset = 0;
        for (size_t l = 0; l < max_additive_subsets[k].size(); ++l) { // ...calculate its h-value
            max_additive_subsets[k][l]->evaluate(sample);
            assert(!max_additive_subsets[k][l]->is_dead_end());
            h_subset += max_additive_subsets[k][l]->get_heuristic();
        }
        if (h_pattern + h_subset > h_collection) {
            // return true if one max additive subest is found for which the condition is met
            return true;
        }
    }
    return false;
}

bool PatternGenerationHaslum::is_time_up(Timer &timer) {
    return time_limit > 0 && timer() > time_limit;
}

void PatternGenerationHaslum::hill_climbing(double average_operator_cost,
        set<vector<int>, PatternOrder > &initial_candidate_patterns) {
    Timer timer;
    Timer measureTimer;
    // stores all candidate patterns generated so far in order to avoid duplicates
    set<vector<int> > generated_patterns;
    // new_candidates is the set of new pattern candidates from the last call to generate_candidate_patterns
    set<vector<int>, PatternOrder > &new_candidates = initial_candidate_patterns;
    // all candidate patterns are converted into pdbs once and stored
    vector<PDBHeuristic *> candidate_pdbs;
    int num_iterations = 0;
    size_t max_pdb_size = 0;
    num_rejected = 0;
    int summed_improvement = 1;
    //const float ratio = 0.0625;
    int under_threshold = 0, req_under_threshold = 0;


    // Time estimation stuff
    Timer iteration_timer;
    double previous_iteration_time = 0, estimated_time = 0;
    size_t total_pdb_size = 0, pattern_size=0;
    
    cout << "Current peak memory: " << get_peak_memory_in_kb() << endl;
    const size_t k = 1024;
    const size_t total_pdb_size_limit = k * k * (k * 16 / 10); // 1.6GB
    cout << "Memory limit: " << total_pdb_size_limit << endl;
    int was_below_min_improvement = 0;
    while (true) {
        num_iterations += 1;
        if (iteration_limit > 0 && num_iterations > iteration_limit) {
            break;
        }
        cout << "current collection size is " << current_heuristic->get_size() << endl;
        current_heuristic->evaluate(g_initial_state());
        cout << "current initial h value: ";
        if (current_heuristic->is_dead_end()) {
            cout << "infinite => stopping hill-climbing" << endl;
            break;
        } else {
            cout << current_heuristic->get_heuristic() << endl;
        }

        // Reset measure timer
        measureTimer.reset();
        vector<State> samples;
        sample_states(samples, average_operator_cost);
        cout << "[measure] Sample time: " << measureTimer.reset() << endl;

        // For the new candidate patterns check whether they already have been candidates before and
        // thus already a PDB has been created and inserted into candidate_pdbs.
        bool mem_limit = false;
        for (set<vector<int> >::iterator it = new_candidates.begin(); it != new_candidates.end(); ++it) {
            const vector<int> &new_candidate = *it;
            size_t est_mem = est_pattern_mem(new_candidate);
            size_t total_mem = 0;
            total_mem += total_pdb_size + est_mem;
            total_mem += candidate_pdbs.size() * sizeof(PDBHeuristic*);
            total_mem += pattern_size * sizeof(int);
            if (total_mem >= total_pdb_size_limit*2/3)
                cout << "mem_est: " << total_mem << " (real:" << get_peak_memory_in_kb() << ")" << endl;
            if (total_mem >= total_pdb_size_limit) {
                if (!mem_limit) {
                    cout << "Memory limit in iteration: " << num_iterations << endl;
                }
                mem_limit = true;
            }
            if (generated_patterns.count(new_candidate) == 0 && est_mem + total_pdb_size < total_pdb_size_limit) {
                Options opts;
                opts.set<int>("cost_type", cost_type);
                opts.set<vector<int> >("pattern", new_candidate);
                candidate_pdbs.push_back(new PDBHeuristic(opts, false));
                max_pdb_size = max(max_pdb_size,
                        candidate_pdbs.back()->get_size());
                total_pdb_size += candidate_pdbs.back()->get_memory_usage();
                pattern_size += new_candidate.size();
                generated_patterns.insert(new_candidate);
            }
        }
        cout << "[measure] Check/generate time: " << measureTimer.reset() << endl;
        cout << "[measure] Current total pdb size: " << total_pdb_size << endl;

        if (mem_limit && prune_old_patterns) {
            current_heuristic->dominance_pruning();
            cout << "Pruning finished." << endl;
            const vector<PDBHeuristic*> &pdbs = current_heuristic->get_pattern_databases();
            cout << "Retrieved pdbs" << endl;
            size_t num_candidate_pdbs = candidate_pdbs.size();
            cout << "Memory limit reached, pruning old patterns(" << num_candidate_pdbs << ")" << endl;

            int count = 0;
            vector<int> set_diff;
            for (size_t i = 0; i < num_candidate_pdbs; ++i) {
                if (i % 1000 == 0) {
                    cout << ".";
                }
                if (candidate_pdbs[i]) {
                    bool still_valid = false;
                    const vector<int> &cpattern = candidate_pdbs[i]->get_pattern();
                    for (size_t ii = 0; ii < pdbs.size() && !still_valid; ++ii) {
                        const vector<int> &pattern = pdbs[ii]->get_pattern();
                        set_difference(cpattern.begin(), cpattern.end(), pattern.begin(), pattern.end(), back_inserter(set_diff));
                        if (set_diff.size() <= candidate_var_diff){
                            bool anchored = false;
                            for(size_t a=0;a<pattern.size();++a){ // anchor points in pattern
                                vector<int> rel_vars = g_legacy_causal_graph->get_predecessors(pattern[a]);
                                for(size_t f=0;f<rel_vars.size();++f){ // possible vars for anchor point
                                    for(size_t d=0;d<cpattern.size();++d){ // check if anchored
                                        if(rel_vars[f]==cpattern[d])
                                            anchored=true;
                                    }
                                }
                            }
                            
                            
                            still_valid = true;
                        }
                        set_diff.clear();
                    }
                    if (!still_valid) {
                        // Remove, no longer valid
                        total_pdb_size -= candidate_pdbs[i]->get_memory_usage();
                        delete candidate_pdbs[i];
                        candidate_pdbs[i] = NULL;
                        ++count;
                    }
                }
            }
            cout << " " << count << " patterns removed." << endl;
            cout << "[measure] Pruning time: " << measureTimer.reset() << endl;
        }



        // TODO: The original implementation by Haslum et al. uses astar to compute h values for
        // the sample states only instead of generating all PDBs.
        int improvement = -1; // best improvement (= hightest count) for a pattern so far
        int best_pdb_index = -1;
       
        // Iterate over all candidates and search for the best improving pattern/pdb
        for (size_t i = 0; i < candidate_pdbs.size() && (!is_time_up(timer) || bad_time_limit); ++i) {
            PDBHeuristic *pdb_heuristic = candidate_pdbs[i];
            if (pdb_heuristic == 0) { // candidate pattern is too large
                continue;
            }
            // If a candidate's size added to the current collection's size exceeds the maximum
            // collection size, then delete the PDB and let the PDB's entry point to a null reference
            if (current_heuristic->get_size() + pdb_heuristic->get_size() > collection_max_size) {
                total_pdb_size -= pdb_heuristic->get_memory_usage();
                delete pdb_heuristic;
                candidate_pdbs[i] = NULL;
                continue;
            }

            // Calculate the "counting approximation" for all sample states: count the number of
            // samples for which the current pattern collection heuristic would be improved
            // if the new pattern was included into it.
            // TODO: The original implementation by Haslum et al. uses m/t as a statistical
            // confidence interval to stop the astar-search (which they use, see above) earlier.
            int count = 0;
            vector<vector<PDBHeuristic *> > max_additive_subsets;
            current_heuristic->get_max_additive_subsets(pdb_heuristic->get_pattern(), max_additive_subsets);
            for (size_t j = 0; j < samples.size() && samples.size() - j + count >= improvement; ++j) {
                if (is_heuristic_improved(pdb_heuristic, samples[j], max_additive_subsets))
                    ++count;
            }
            if (count > improvement) {
                improvement = count;
                best_pdb_index = i;

                cout << "pattern: " << candidate_pdbs[i]->get_pattern()
                        << " - improvement: " << count << endl;
            }else if(remove_bad_looking_patterns && count < min_improvement){
                total_pdb_size -= pdb_heuristic->get_memory_usage();
                delete pdb_heuristic;
                candidate_pdbs[i] = NULL;
            }
        }
        cout << "[measure] Evaluate time: " << measureTimer.reset() << endl;
        summed_improvement += improvement;

        if (improvement >= min_improvement){
            was_below_min_improvement = 0;
        }
        
        bool stop_hill_climbing = true;
        if (best_pdb_index == -1) {
            cout << "No further possible patterns. Aborting hill climbing." << endl;
        } else if (improvement < min_improvement) {
            if(++was_below_min_improvement >= min_low){
                cout << "Improvement(" << improvement << ") below threshold. Aborting hill climbing." << endl;
            }else{
                cout << "Improvement(" << improvement << ") below threshold." << endl;
                stop_hill_climbing = false;
            }
        } else if (is_time_up(timer)) {
            cout << "Time limit reached. Aborting hill climbing. Time used: " << timer() << endl;
            stop_hill_climbing = true; // force stop even if minimum number of iterations not reached
        } else if (auto_limit > 0 && (float) improvement / summed_improvement < auto_limit) {
            cout << "Improvement(" << improvement << ") below ratio-threshold(" << (summed_improvement * auto_limit) << "). Aborting hill climbing." << endl;
            cout << "Iteration: " << num_iterations << endl;
            if (++under_threshold < req_under_threshold) {
                stop_hill_climbing = false;
            }
        } else {
            stop_hill_climbing = false;
        }

        if (stop_hill_climbing) { // end hill climbing algorithm
            // Note that using dominance pruning during hill-climbing could lead to
            // fewer discovered patterns and pattern collections.
            // A dominated pattern (collection) might no longer be dominated
            // after more patterns are added.
            if (dominance_pruning) {
                current_heuristic->dominance_pruning();
            }
            cout << "iPDB: iterations = " << num_iterations << endl;
            cout << "iPDB: num_patterns = "
                    << current_heuristic->get_pattern_databases().size() << endl;
            cout << "iPDB: size = " << current_heuristic->get_size() << endl;
            cout << "iPDB: improvement = " << improvement << endl;
            cout << "iPDB: generated = " << generated_patterns.size() << endl;
            cout << "iPDB: rejected = " << num_rejected << endl;
            cout << "iPDB: max_pdb_size = " << max_pdb_size << endl;
            /*
            // Output selected patterns
            cout << "Selected patterns: [";
            for (size_t i = 0; i < current_heuristic->get_pattern_databases().size(); ++i) {
                const vector<int> &pattern = current_heuristic->get_pattern_databases()[i]->get_pattern();
                cout << "[";
                for (size_t ii = 0; ii < pattern.size(); ++ii) {
                    cout << pattern[ii];
                    if (ii < pattern.size() - 1)
                        cout << ",";
                }
                cout << "]";
                if (i < current_heuristic->get_pattern_databases().size() - 1) {
                    cout << ",";
                }
            }
            cout << "]" << endl;
            */
            break;
        }

        // add the best pattern to the CanonicalPDBsHeuristic
        const vector<int> &best_pattern = candidate_pdbs[best_pdb_index]->get_pattern();
        cout << "found a better pattern with improvement " << improvement << endl;
        cout << "pattern: " << best_pattern << endl;
        current_heuristic->add_pattern(best_pattern);

        // clear current new_candidates and get successors for next iteration
        new_candidates.clear();
        generate_candidate_patterns(best_pattern, new_candidates, candidate_var_diff - 1);

        // remove the added PDB from candidate_pdbs
        total_pdb_size -= candidate_pdbs[best_pdb_index]->get_memory_usage();
        delete candidate_pdbs[best_pdb_index];
        candidate_pdbs[best_pdb_index] = NULL;
        cout << "[measure] Cleanup time: " << measureTimer.reset() << endl;
        cout << "Hill-climbing time so far: " << timer << endl;

        // Time estimation
        double iteration_time = iteration_timer.reset();
        cout << "[time_est]Estimated/real iteration runtime: " << estimated_time << "/" << iteration_time << endl;
        estimated_time = iteration_time * 2 - previous_iteration_time;
        previous_iteration_time = iteration_time;
        cout << "[time_est]Estimated runtime of next iteration: " << estimated_time << endl;
        if (time_limit > 0 && estimated_time + timer() > time_limit) {
            cout << "[time_est]Estimated runtime exceeds time limit" << endl;
            // Estimation not exact enough for use as stopping criterion
            //break;
        }
    }
    cout << "IPDB finished: 1" << endl;


    // delete all created PDB-pointer
    for (size_t i = 0; i < candidate_pdbs.size(); ++i) {
        delete candidate_pdbs[i];
    }
}

void PatternGenerationHaslum::initialize() {
    cout << "Number of operators " << g_operators.size() << endl;
    int num_vars = g_variable_name.size();
    long num_arcs = 0;
    for (int var = 0; var < num_vars; ++var) {
        num_arcs += g_causal_graph->get_successors(var).size();
    }
    cout << "Number of nodes in causal graph " << num_vars << endl;
    cout << "Number of edges in causal graph " << num_arcs << endl;

    // calculate goal variables that occur as common effects with other variables
    goal_common_action_effects = vector<vector<int> >(g_variable_name.size());
    vector<bool> is_goal(g_variable_name.size(), false);
    for (size_t i = 0; i < g_goal.size(); ++i) {
        is_goal[g_goal[i].first] = true;
    }
    for (size_t i = 0; i < g_operators.size(); ++i) {
        const vector<PrePost> &pre_posts = g_operators[i].get_pre_post();
        for (size_t j = 0; j < pre_posts.size(); ++j) {
            int var1 = pre_posts[j].var;
            if (!is_goal[var1]) {
                continue;
            }
            for (size_t k = 0; k < pre_posts.size(); ++k) {
                int var2 = pre_posts[k].var;
                if (var1 == var2) {
                    continue;
                }
                vector<int> &common_eff = goal_common_action_effects[var2];
                // TODO avoid O(n) find
                if (find(common_eff.begin(), common_eff.end(), var1) == common_eff.end()) {
                    common_eff.push_back(var1);
                }
            }
        }
    }
    for (size_t i = 0; i < g_variable_name.size(); ++i) {
        vector<int> &common_eff = goal_common_action_effects[i];
        sort(common_eff.begin(), common_eff.end());
    }

    // calculate average operator costs
    double average_operator_cost = 0;
    for (size_t i = 0; i < g_operators.size(); ++i) {
        average_operator_cost += get_adjusted_action_cost(g_operators[i], cost_type);
    }
    average_operator_cost /= g_operators.size();
    cout << "Average operator cost: " << average_operator_cost << endl;

    // initial collection: a pdb for each goal variable
    vector<vector<int> > initial_pattern_collection;
    for (size_t i = 0; i < g_goal.size(); ++i) {
        initial_pattern_collection.push_back(vector<int>(1, g_goal[i].first));
    }
    Options opts;
    opts.set<int>("cost_type", cost_type);
    opts.set<vector<vector<int> > >("patterns", initial_pattern_collection);
    current_heuristic = new CanonicalPDBsHeuristic(opts);
    current_heuristic->evaluate(g_initial_state());
    if (current_heuristic->is_dead_end())
        return;

    // initial candidate patterns, computed separately for each pattern from the initial collection
    set<vector<int>, PatternOrder> initial_candidate_patterns;
    for (size_t i = 0; i < current_heuristic->get_pattern_databases().size(); ++i) {
        const vector<int> &current_pattern = current_heuristic->get_pattern_databases()[i]->get_pattern();
        generate_candidate_patterns(current_pattern, initial_candidate_patterns, candidate_var_diff - 1);
    }
    cout << "done calculating initial pattern collection and candidate patterns for the search" << endl;

    // call to this method modifies initial_candidate_patterns (contains the new_candidates
    // after each call to generate_candidate_patterns)
    hill_climbing(average_operator_cost, initial_candidate_patterns);
}

void PatternGenerationHaslum::create_options(OptionParser &parser) {
    parser.add_option<int>("pdb_max_size",
            "max number of states per pdb", "2000000");
    parser.add_option<int>("collection_max_size",
            "max number of states for collection", "20000000");
    parser.add_option<int>("num_samples", "number of samples", "1000");
    parser.add_option<int>("min_improvement",
            "minimum improvement while hill climbing", "10");
    parser.add_option<bool>("dominance_pruning",
            "use dominance pruning to reduce number of patterns after hill climbing.",
            "true");
    parser.add_option<int>("time_limit",
            "Time in seconds that should be spent on hill climbing. Use 0 (default) for no limit.",
            "0");
    parser.add_option<bool>("bad_time_limit",
            "Use inferior time limit",
            "false");
    parser.add_option<int>("iteration_limit",
            "Number of iteration spent on hill climbing. Use 0 (default) for no limit.",
            "0");
    parser.add_option<float>("auto_limit",
            "Automatic improvement limit",
            "-1");
    parser.add_option<int>("candidate_var_diff",
            "Difference in number of variables for each candidate pattern",
            "1");
    parser.add_option<bool>("prune_old_patterns",
            "Prune old patterns when memory limit is reached",
            "false");
    parser.add_option<bool>("remove_bad_looking_patterns",
            "Remove patterns with little improvement",
            "false");
    parser.add_option<bool>("ffsampling",
            "Use FF heuristic for sampling",
            "false");
    parser.add_option<int>("min_low",
            "Number of times the automatic limit has to be violated before iPDB gets stopped",
            "1");
}

void PatternGenerationHaslum::sanity_check_options(OptionParser &parser, Options &opts) {
    if (opts.get<int>("pdb_max_size") < 1)
        parser.error("size per pdb must be at least 1");
    if (opts.get<int>("collection_max_size") < 1)
        parser.error("total pdb collection size must be at least 1");
    if (opts.get<int>("min_improvement") < 0)
        parser.error("minimum improvement must be at least 0");
    if (opts.get<int>("min_improvement") > opts.get<int>("num_samples"))
        parser.error("minimum improvement must not be higher than number of samples");
}

static Heuristic *_parse(OptionParser &parser) {
    parser.document_synopsis(
            "iPDB",
            "the pattern selection procedure by Haslum et al. (AAAI 2007); "
            "see also Sievers et al. (SoCS 2012) for implementation notes");
    PatternGenerationHaslum::create_options(parser);
    Heuristic::add_options_to_parser(parser);
    Options opts = parser.parse();
    if (parser.help_mode())
        return 0;
    PatternGenerationHaslum::sanity_check_options(parser, opts);

    if (parser.dry_run())
        return 0;

    PatternGenerationHaslum pgh(opts);
    return pgh.get_pattern_collection_heuristic();
}


static Plugin<Heuristic> _plugin("ipdb", _parse);
