#include "pattern_collection_generator_cegar.h"

#include "../option_parser.h"
#include "../plugin.h"
#include "pattern_database.h"
#include "../tasks/root_task.h"
#include "../globals.h"
#include "../task_utils/successor_generator.h"
#include "abstract_solution_data.h"
#include <utility>
#include "../utils/rng_options.h"
#include "../task_utils/task_properties.h"
#include "../utils/math.h"

using namespace std;

namespace pdbs{
PatternCollectionGeneratorCegar::PatternCollectionGeneratorCegar(
        const options::Options& opts)
        : rng(utils::parse_rng_from_options(opts)),
          concrete_task_proxy(*g_root_task()),
          max_refinements(opts.get<int>("max_refinements")),
          max_pdb_size(opts.get<int>("max_pdb_size")),
          interleaving(opts.get<bool>("interleaving")),
          blacklisting(false), // we set this further down in constructor
          max_blacklist_size(opts.get<int>("max_blacklist_size")),
          additivity(static_cast<Additivity>(opts.get_enum("additivity"))),
          initial(static_cast<InitialCollectionType>(opts.get_enum("initial"))),
          concrete_solution_index(-1) {
    cout << token << opts.get_unparsed_config() << endl;


    // save all goals in random order for refinement later
    for (auto goal : concrete_task_proxy.get_goals()) {
        goals.push_back(goal.get_variable().get_id());
    }
    rng->shuffle(goals);

    // initialize correlation matrix
    size_t nvars = concrete_task_proxy.get_variables().size();
    correlation_matrix.reserve(nvars);
    for (size_t i = 0; i < nvars; ++i) {
        correlation_matrix.emplace_back(nvars,false);
    }

    // used for preemptive blacklisting
    std::vector<int> correlation_counters(nvars,0);
    // find correlations
    for (auto op : concrete_task_proxy.get_operators()) {
        for (auto eff1 : op.get_effects()) {
            int var1 = eff1.get_fact().get_variable().get_id();
            // mark all variables that the operator writes to as correlated
            for (auto eff2 : op.get_effects()) {
                int var2 = eff2.get_fact().get_variable().get_id();
                if (var1 == var2) continue;
                if (!correlation_matrix[var1][var2]) {
                    correlation_counters[var1]++;
                    correlation_counters[var2]++;
                }
                correlation_matrix[var1][var2] = true;
                correlation_matrix[var2][var1] = true;
            }

            // With only partial additivity enabled, we only consider
            // variables that are written to by the same operator as
            // correlated. This means that read-write relationships
            // between two variables are not looked at, when building
            // the correlation matrix.
            if(additivity == Additivity::PARTIAL) {
                continue;
            }

            // mark all variables the operator reads from as correlated
            // to the ones the operator writes to
            for (auto pre : op.get_preconditions()) {
                int var2 = pre.get_variable().get_id();
                if (var2 == var1) continue;
                if (!correlation_matrix[var1][var2]) {
                    correlation_counters[var1]++;
                    correlation_counters[var2]++;
                }
                correlation_matrix[var1][var2] = true;
                correlation_matrix[var2][var1] = true;
            }
        }
    }

    // number of edges in causal graph
    int ml_edge_count = 0;
    for (int count : correlation_counters) {
        // each correlation describes an edge in correlation/causal graph
        ml_edge_count += count;
    }
    int ml_nops = concrete_task_proxy.get_operators().size();
    int ml_nvars = nvars;
    float ml_graph_density
            = static_cast<float>(ml_edge_count)/static_cast<float>(ml_nvars*(ml_nvars-1));
    cout << "ML Features:" << endl;
    cout << "ML: Number of variables: " << ml_nvars << endl;
    cout << "ML: Number of operators: " << ml_nops << endl;
    cout << "ML: Causal Graph Density: " << ml_graph_density << endl;

    switch (static_cast<Blacklisting>(opts.get_enum("blacklisting"))) {
        case NO_BLACKLISTING:
            blacklisting = false;
            break;
        case NAIVE:
            blacklisting = true;
            break;
        case ADAPTIVE:
            // hard-coded for now
            float ml_nvars_coeff = -1610.0f;
            float ml_nops_coeff = -2470.0f;
            float ml_graph_density_coeff = -3577.3914659f;
            float ml_intercept = 97.0f;
            float decision = ml_nvars*ml_nvars_coeff +
                             ml_nops*ml_nops_coeff +
                             ml_graph_density*ml_graph_density_coeff +
                             ml_intercept;
            blacklisting = decision >= 0;
            if(blacklisting) {
                cout << "Decision function decided to enable blacklisting: "
                     << decision << endl;
            } else {
                cout << "Decision function decided not to enable blacklisting: "
                     << decision << endl;
            }
            break;
    }

    if(!blacklisting) {
        return;
    } else {
        double blacklist_percent = opts.get<int>("max_blacklist_percentage");
        double frac = blacklist_percent != 0 ? 1.0/blacklist_percent : 0;

        if (frac >= 0) {
            max_blacklist_size = static_cast<int>(nvars * frac);
        }
    }

    for (int i = 0; i < max_blacklist_size; ++i) {
        if (nvars - goals.size() <= 0) {
            break;
        }
        int max_var = 0;
        for (size_t var = 0; var < nvars+blacklist.size(); ++var) {
            // make sure we don't blacklist goal variables
            if (std::find(goals.begin(), goals.end(), var) != goals.end()) {
                continue;
            }
            if (blacklist.count(max_var) > 0 ||
                (correlation_counters[max_var] < correlation_counters[var] &&
                blacklist.count(var) == 0)) {
                max_var = var;
            }
        }
        cout << token << "blacklisting var" << max_var
             << "(" << correlation_counters[max_var] << ")" << endl;
        blacklist.insert(max_var);
        nvars--;
    }
}

bool PatternCollectionGeneratorCegar::can_merge(
        const std::vector<unsigned int> &merge_list, int var) const {

    unsigned int num_states = static_cast<unsigned int>( var >= 0 ?
            concrete_task_proxy.get_variables()[var].get_domain_size() : 1);

    for (const unsigned int i : merge_list) {
        const std::unique_ptr<AbstractSolutionData> &sol = solutions[i];
        if (sol == nullptr) { continue; }

        for (auto vid : sol->get_pattern()) {
            VariableProxy v = concrete_task_proxy.get_variables()[vid];

            if (utils::is_product_within_limit(num_states, v.get_domain_size(),
                                               numeric_limits<int>::max())) {
                num_states *= v.get_domain_size();
            } else {
                return false;
            }
        }
    }

    cout << token << "Resulting pdb will have " << num_states << " states" << endl;
    return true;
}

bool PatternCollectionGeneratorCegar::termination_conditions_met() const {

    if(terminate) {
        return true;
    }

    // For now we only terminate after a certain number of refinements.
    // We can do this because abstraction refinement can be interrupted
    // at any point and still lead to an admissible search heuristic.
    if (max_refinements != -1) {
        if(refinement_counter >= max_refinements) {
            cout << token << "maximum allowed number of refinements reached." << endl;
            return true;
        }
    }

    if (max_pdb_size != -1) {
        for (const std::unique_ptr<AbstractSolutionData> &s : solutions) {
            if (s == nullptr) { continue; }
            if (s->get_pdb()->get_size() >= max_pdb_size) {
                cout << token << "a pdb has reached maximum allowed size" << endl;
                return true;
            }
        }
    }

    return false;
}

void PatternCollectionGeneratorCegar::generate_trivial_solution_collection(
        TaskProxy& task_proxy) {
    cout << token << "generating trivial solution collection" << endl;
    assert(!goals.empty());

    if (initial == InitialCollectionType::RANDOM_GOAL) {
        // for now we generate just one pattern containing one goal variable
        Pattern pattern;
        pattern.push_back(goals.back());
        goals.pop_back();
        solutions.emplace_back(new AbstractSolutionData(g_root_task(), pattern, rng));
        // first variable is stored in the first solution
        solution_lookup[pattern.back()] = 0;
    } else { // if the initial collection must contain all goal variables
        while (!goals.empty()) {
            int var = goals.back();
            goals.pop_back();

            if (additivity != Additivity::NONE) {
                refine_with_additivity(var);
            } else {
                Pattern pat{var};
                solutions.emplace_back(
                        new AbstractSolutionData(g_root_task(), pat, rng));
                solution_lookup[var] = solutions.size()-1;
            }
        }
    }

    // print trivial collection
    cout << token << "finished initializing collection" << endl;
    VariablesProxy vp = task_proxy.get_variables();
    cout << token << "initial collection: [";
    for(const auto &sol : solutions) {
        if (sol == nullptr) { continue; }
        cout << "[";
        for(const auto &var : sol->get_pattern()) {
            cout << vp[var].get_name() << ",";
        }
        cout << "]";
    }
    cout << "]" << endl;
}

PatternCollectionInformation PatternCollectionGeneratorCegar::generate(
        const std::shared_ptr<AbstractTask> &task) {
    cout << token << "generating pattern collection using CEGAR" << endl;
    concrete_task_proxy = TaskProxy(*task);

    // Start with a solution of the trivial abstraction
    generate_trivial_solution_collection(concrete_task_proxy);

    cout << endl;
    // main loop of the algorithm
    while (not termination_conditions_met()) {
        cout << "iteration #" << refinement_counter+1 << endl;
        unfixable_flaws = 0;

        // vector of solution indices and flaws associated with said solutions
        FlawList flaws = get_flaws();

        if (flaws.empty()) {
            if (unfixable_flaws == 0) {
                if (!interleaving && concrete_solution_index < 0) {
                    // Pattern collection has stabilized.
                    // This means we don't have any flaws we could fix,
                    // but none of the plans we have could solve the
                    // concrete task. If interleaving is disabled, then
                    // this situation is a dead-end for the algorithm.
                    cout << token << "pattern collection stable" << endl;
                    break;
                }

                if (blacklisting && !blacklist.empty()) {
                    cout << token
                         << "No flaws left, but because the algorithm has"
                         << " a non-empty blacklist, the task solution"
                         << endl
                         << " is not guaranteed to be given"
                         << endl;
                    break;
                }

                cout << token << "task solved during refinement" << endl;
                extract_plan();
                //utils::exit_with(utils::ExitCode::PLAN_FOUND);
                break;
            } else {
                // we only have unfixable flaws, so abort and return
                // the pattern collection we were able to generate so far
                assert(additivity == Additivity::PARTIAL);
                cout << token << "Only unfixable flaws left. Aborting." << endl;
                break;
            }
        }

        // if there was a flaw, then refine the abstraction
        // such that said flaw does not occur again
        refine(flaws);

        refinement_counter++;
        std::cout << endl;
    }

    // build and print final collection
    std::shared_ptr<PatternCollection> patterns = make_shared<PatternCollection>();
    std::shared_ptr<PDBCollection> pdbs = make_shared<PDBCollection>();
    VariablesProxy vp = concrete_task_proxy.get_variables();
    cout << token << "final collection: [";
    for (const auto &sol : solutions) {
        if(sol == nullptr) { continue; }
        Pattern pat;
        std::shared_ptr<PatternDatabase> pdb = sol->get_pdb();
        pdbs->push_back(pdb);
        cout << "[";
        for (const auto &var : sol->get_pattern()) {
            pat.push_back(var);
            cout << vp[var].get_name() << ",";
        }
        patterns->push_back(pat);
        cout << "],";
    }
    cout << "]" << endl;

    //
    PatternCollectionInformation pattern_collection_information(
            concrete_task_proxy, patterns);
    pattern_collection_information.set_pdbs(pdbs);
    cout << token << "cegar_pdbs algorithm finished" << endl;
    cout << token << "cegar runtime: " << runtime_timer() << endl;
    return pattern_collection_information;
}

FlawList PatternCollectionGeneratorCegar::get_flaws() {
    // for now interleaving is only allowed for (partially) additive collections
    if (interleaving) {
        auto &&res = additivity == FULL? get_flaws_with_interleaving()
                                       : get_flaws_with_greedy_interleaving();
        // we reset pointers to the current operator for each plan
        // before returning the result
        for (auto &solptr : solutions) {
            if (solptr != nullptr) solptr->reset();
        }
        return res;
    } else {
        auto &&res = get_flaws_without_interleaving();
        for (auto &solptr : solutions) {
            if (solptr != nullptr) solptr->reset();
        }
        return res;
    }
}

FlawList PatternCollectionGeneratorCegar::get_flaws_with_interleaving() {
    // make sure this function is only called when interleaving flag is true
    // and additivity is enabled
    assert(interleaving && additivity == Additivity::FULL);

    FlawList flaws;
    State current_state = concrete_task_proxy.get_initial_state();

    for (size_t i = 0; i < solutions.size(); ++i) {
        if(solutions[i] == nullptr) {
            continue;
        }

        AbstractSolutionData& solution = *solutions[i];

        if (!solution.solution_exists()) {
            std::cerr << token << "Problem unsolvable" << std::endl;
            utils::exit_with(utils::ExitCode::UNSOLVABLE);
        }

        auto pair = apply_solution(solution, current_state);
        current_state = std::move(pair.first);
        auto new_flaws = pair.second;

        for (FlawVariable v : new_flaws) {
            flaws.emplace_back(i,v);
        }
    }


    if (flaws.empty() &&
            !task_properties::is_goal_state(concrete_task_proxy,current_state)) {
        if(!goals.empty()) {
            // We just set the solution index to -1, it will not be used at any
            // point with this kind of flaw.
            flaws.emplace_back(-1,RANDOM_GOAL_VARIABLE);
        } else {
            cerr << token
                 << "some plan requested a random goal variable, but none are available."
                 << " this is a critical error if interleaving is enabled" << endl;
            utils::exit_with(utils::ExitCode::CRITICAL_ERROR);
        }
        cout << "CEGAR_PDBs: "
             << "interleaving finished, but did not lead to a goal state." << endl;
    }

    return flaws;
}

FlawList PatternCollectionGeneratorCegar::get_flaws_with_greedy_interleaving() {
    assert(additivity == Additivity::PARTIAL && interleaving);

    State current_state = concrete_task_proxy.get_initial_state();

    FlawList total_flaws;

    // We apply as many operators of a single plan as we can and once we
    // encounter a flaw (or once the plan is completely executed, if there
    // was no flaw), we move on to the next plan and so on...
    // We collect all flaws we encounter as we work our way through the list
    // of plans. Once we have reached the end of the list, we start over at
    // the beginning (because hopefully some of the later plans changed the
    // concrete state in such a way that the earlier plans can be advanced
    // again). The algorithm stops once it is unable to advance a single
    // plan during an entire iteration.
    bool plan_advanced; // could we advance at least one plan this iteration?
    do {
        plan_advanced = false;
        for (size_t i = 0; i < solutions.size(); ++i) {
            auto &solptr = solutions[i];
            // no need to try to advance a plan that is already finished
            // (or one that does not even exist)
            if (solptr == nullptr) continue;
            if (solptr->plan_finished()) continue;

            auto state_and_flaws = apply_solution(*solptr, current_state);
            State &new_state = state_and_flaws.first;

            if (new_state != current_state) {
                // the new state is different from our current state
                // this means that the plan in solptr could be advanced
                plan_advanced = true;
                current_state = std::move(new_state);
            }


            // now we must decide whether we want to collect the flaws
            // we found on this run of the for-loop
            std::vector<FlawVariable> &new_flaws = state_and_flaws.second;
            if (solptr->plan_finished()) {
                // if the plan could be successfully executed, then
                // this means that there were no flaws
                // (but we still check just to make sure)
                if (!new_flaws.empty()) {
                    cerr << token << "plan finished, yet somehow there are still flaws."
                                  << endl;
                    utils::exit_with(utils::ExitCode::CRITICAL_ERROR);
                }
                continue; // move on to the next plan
            } else {
                // Right now it is possible for a plan to get stuck on
                // a single operator for e.g. three iterations of the
                // for loop and then the apply_solution would return
                // the same flaws three times, and they would be added
                // to the total_flaws, thus making them more likely
                // to be selected during the refinement process.
                // This creates a bias. Is this ok?
                // TODO: some sort of duplicate avoidance
                for (auto flvar : new_flaws) {
                    // if the variable already exists in some pattern
                    // then this is an unfixable flaw, so no point
                    // in requesting it again
                    if (solution_lookup.count(flvar)) {
                        unfixable_flaws++;
                        continue;
                    }
                    total_flaws.emplace_back(i, flvar);
                }
            }
        }
    } while(plan_advanced);

    if (total_flaws.empty() &&
        !task_properties::is_goal_state(concrete_task_proxy,current_state)) {
        if(!goals.empty()) {
            // We just set the solution index to -1, it will not be used at any
            // point with this kind of flaw.
            total_flaws.emplace_back(-1,RANDOM_GOAL_VARIABLE);
        } else {
            cerr << token
                 << "some plan requested a random goal variable, but none are available."
                 << " this can happen when partial additivity is enabled" << endl;
            cerr << token << "Algorithm will not continue past this point." << endl;
        }
        cout << "CEGAR_PDBs: "
             << "interleaving finished, but did not lead to a goal state." << endl;
    }

    return total_flaws;
}

FlawList PatternCollectionGeneratorCegar::get_flaws_without_interleaving() {
    assert(!interleaving);

    FlawList flaws;
    State&& concrete_init = concrete_task_proxy.get_initial_state();

    for (size_t i = 0; i < solutions.size(); ++i) {
        if (solutions[i] == nullptr) {
            continue;
        }

        AbstractSolutionData& solution = *solutions[i];

        // abort here if no abstract solution could be found
        if (!solution.solution_exists()) {
            cerr << token << "Problem unsolvable" << endl;
            utils::exit_with(utils::ExitCode::UNSOLVABLE);
        }

        // find out if and why the abstract solution
        // would not work for the concrete task
        // when we don't interleave, we always start with the initial state
        auto pair = apply_solution(solution, concrete_init);
        auto new_flaws = pair.second;

        // no flaws means we have an abstract
        // solution which solves the concrete task
        if (new_flaws.empty()) {
            // drop everything as there is no need to continue
            concrete_solution_index = i;
            flaws.clear();
            return flaws;
        } else {
            for(FlawVariable v : new_flaws) {
                if (v == RANDOM_GOAL_VARIABLE && goals.empty()) continue;
                // if additivity is partial and the flaw variable is already
                // in some pattern, then this kind of flaw is not fixable
                // so no point in adding it.
                if (additivity == Additivity::PARTIAL && solution_lookup.count(v)) {
                    unfixable_flaws++;
                    continue;
                }
                flaws.emplace_back(i,v);
            }
        }
    }

    return flaws;
}

bool PatternCollectionGeneratorCegar::are_additive(
        const Pattern &pattern, int v) const {
    for (int var : pattern) {
        // not additive if at least one variable is correlated with v
        if (correlation_matrix[var][v]) {
            return false;
        }
    }

    return true;
}

std::pair<State,std::vector<FlawVariable>>
PatternCollectionGeneratorCegar::apply_solution(
        AbstractSolutionData &solution,
        const State &current) {
    std::vector<FlawVariable> flaws;
    bool plan_failed = false;
    State concrete_state(current);

    while (!solution.plan_finished()) {
        OperatorID abs_opid = solution.get_current_operator();
        OperatorProxy abs_op = solution.get_proxy().get_operators()[abs_opid];

        // retrieve the concrete operator that corresponds to the abstracted one
        OperatorID opid = abs_op.get_global_operator_id();
        OperatorProxy op = concrete_task_proxy.get_operators()[opid];

        // we do not use task_properties::is_applicable here because
        // checking for applicability manually allows us to directly
        // access the precondition that precludes the operator from
        // being applicable
        for (FactProxy precondition : op.get_preconditions()) {
            // we ignore blacklisted variables if blacklisting is enabled
            if (blacklisting
                && blacklist.count(precondition.get_variable().get_id())) {
                continue;
            }

            if (concrete_state[precondition.get_variable()] != precondition) {
                plan_failed = true;
                FlawVariable f = precondition.get_variable().get_id();
                flaws.push_back(f);
            }
        }

        if (plan_failed) {
            break;
        } else {
            concrete_state = concrete_state.get_successor(op);
            solution.advance();
        }
    }


    // If no flaw has been discovered so far, then the plan was
    // executed successfully. The only flaw that can still come up
    // is when the final state is not actually a goal state of the
    // concrete state-space. This check only needs to be performed
    // when interleaving is disabled, because if it is enabled, then
    // this check needs to only be performed once, for all plans
    // (see get_flaws_with_interleaving)
    if (!interleaving &&
        !task_properties::is_goal_state(concrete_task_proxy, concrete_state) &&
        !plan_failed) {
        flaws.push_back(RANDOM_GOAL_VARIABLE);
        cout << token << "plan of pattern "; solution.print_pattern(); cout << endl;
        cout << token << "finished, but did not lead to a goal state." << endl;
    }

    std::pair<State,std::vector<FlawVariable>> res(concrete_state,flaws);
    return res;
}

void PatternCollectionGeneratorCegar::extract_plan() {
    // if no interleaving was employed then the optimal plan
    // is contained in just one solution instance
    if(!interleaving) {
        solutions[concrete_solution_index]->print_plan();
        cout << "Plan length: "
             << solutions[concrete_solution_index]->get_plan_length()
             << " step(s)." << endl;
        cout << "Plan cost: "
             << solutions[concrete_solution_index]->get_plan_cost() << endl;
    } else {
        // ...otherwise the optimal plan is a combination
        // of all plans in the solution collection
        int sum_cost = 0;
        size_t sum_length = 0;
        for(const auto& solution : solutions) {
            if(solution == nullptr) continue;
            // because the order of plans does not matter if additivity holds
            // between all patters in the collection (which it does, in this case)
            // then the order of plans does not matter, so we just output them
            // sequentially in the order in which they are in the collection
            solution->print_plan();
            sum_cost += solution->get_plan_cost();
            sum_length += solution->get_plan_length();
        }
        cout << "Plan length: "
             << sum_length
             << " step(s)." << endl;
        cout << "Plan cost: "
             << sum_cost << endl;
    }
}

std::pair<int,int> PatternCollectionGeneratorCegar::pick_flaw(
        const FlawList &flaws) {
    if (flaw_selection_strategy == FlawSelectionStrategy::RANDOM_FLAW) {
        return flaws.random_flaw(rng);
    } else {
        return flaws.get_least_common_flaw();
    }
}

void PatternCollectionGeneratorCegar::refine(
        const FlawList &flaws) {
    assert(!flaws.empty());

    // pick a random flaw
    std::pair<int,int> flaw = pick_flaw(flaws);
    unsigned int sol_index = static_cast<unsigned int>(flaw.first);
    int var = flaw.second;
    bool is_goal_var = false;

    if (var == RANDOM_GOAL_VARIABLE) {
        assert(!goals.empty());
        // randomly select a goal variable that is not yet
        // in the pattern collection
        is_goal_var = true;
        do {
            var = goals.back();
            goals.pop_back();
        } while (solution_lookup.count(var) != 0);
        cout << token << "introducing goal variable " << var << endl;
    } else {
        cout << token << "plan of pattern ";
        solutions[sol_index]->print_pattern();
        cout << endl;
        cout << token << "failed because of var" << var << endl;
    }

    // we should have a real variable by now
    assert(var != RANDOM_GOAL_VARIABLE);

    if(additivity != Additivity::NONE) {
        refine_with_additivity(var);
    } else {
        refine_without_additivity(var, is_goal_var, sol_index);
    }
}

void PatternCollectionGeneratorCegar::refine_with_additivity(int var) {
    // keep track of the patterns we will have to merge
    std::vector<unsigned int> merge_list;
    size_t final_pattern_size = 1; // 1 and not 0 because we need to account for var
    for (size_t i = 0; i < solutions.size(); ++i) {
        if (solutions[i] == nullptr) continue;
        if (!are_additive(solutions[i]->get_pattern(), var)) {
            merge_list.push_back(i);
            final_pattern_size += solutions[i]->get_pattern().size();

            std::cout << token << "[" << var << "] is not additive with ";
            solutions[i]->print_pattern(); std::cout << std::endl;
        }
    }

    // the simple case: everything is additive, so we just introduce
    // pattern {var} to the collection
    if (merge_list.empty()) {
        Pattern pat{var};
        solutions.emplace_back(new AbstractSolutionData(g_root_task(), pat, rng));
        solution_lookup[var] = solutions.size()-1;

        std::cout << token
                  << "[" << var << "] is additive with all other patterns"
                  << std::endl;
        return;
    }

    // check if resulting pattern is not too large
    if (!can_merge(merge_list, var)) {
        cout << token << "Merging would result in an overflow. Aborting";
        terminate = true;
        return;
    }

    // merge selected patterns
    std::cout << token
              << "merging preselected patterns"
              << std::endl;
    Pattern pat;
    pat.reserve(final_pattern_size);
    pat.push_back(var);
    for (unsigned int i : merge_list) {
        const Pattern &old_pattern = solutions[i]->get_pattern();
        pat.insert(pat.end(), old_pattern.begin(), old_pattern.end());
    }
    std::sort(pat.begin(), pat.end());

    try {
        std::unique_ptr<AbstractSolutionData> merged(
                new AbstractSolutionData(g_root_task(), pat,rng));
        solutions.push_back(std::move(merged));
    } catch (std::length_error& err) {
        std::cout << token
                  << "unable to reserve memory for pdb. merging failed."
                  << endl;
        terminate = true;
        return;
    }

    // we unset previous patterns only now, that the
    // merged solution data could be successfully constructed.
    for (unsigned int i : merge_list) {
        solutions[i] = nullptr;
    }

    // update variable lookup table
    for( auto v : pat) {
        solution_lookup[v] = solutions.size()-1;
    }
}

void PatternCollectionGeneratorCegar::refine_without_additivity(
        int var, bool is_goal_var, unsigned int sol_index) {

    AbstractSolutionData& solution = *solutions[sol_index];

    // check if the variable is already part of some pattern
    auto it = solution_lookup.find(var);
    if (it != solution_lookup.end()) {
        unsigned int other_index = static_cast<unsigned>(it->second);
        assert(other_index != sol_index);
        assert(solutions[other_index] != nullptr);
        AbstractSolutionData& other = *solutions[other_index];
        const Pattern& pat = other.get_pattern();

        cout << token << "pattern "; solution.print_pattern(); cout << endl;
        cout << token << "wants var" << var << ", but it is already in" << endl;
        cout << token << "pattern "; other.print_pattern(); cout << endl;
        cout << token << "so the two will be merged" << endl;

        // check if merging is possible
        std::vector<unsigned int> merge_list{other_index,sol_index};
        if(!can_merge(merge_list)) {
            cout << token << "Merging would result in an overflow. Aborting";
            terminate = true;
            return;
        }

        // update lookup table before merging
        for (auto v : pat) {
            solution_lookup[v] = sol_index;
        }
        Pattern new_pattern = solution.get_pattern();
        // merge patterns
        new_pattern.insert(new_pattern.end(), pat.begin(), pat.end());
        std::sort(new_pattern.begin(), new_pattern.end());
        try {
            std::unique_ptr<AbstractSolutionData> merged(
                    new AbstractSolutionData(g_root_task(), new_pattern,rng));
            solutions.at(sol_index) = std::move(merged);
        } catch (std::length_error& err) {
            std::cout << token
                      << "unable to reserve memory for pdb. merging failed."
                      << endl;
            terminate = true;
            return;
        }
        // forget the other solution, as it is redundant now
        solutions.at(other_index) = nullptr;
    } else { // variable is not in the collection yet
        // if variable is a goal variable
        if (is_goal_var) {
            cout << token << "var" << var << " will receive it's own pattern" << endl;
            // create new pattern with the variable
            Pattern pat{var};
            solutions.emplace_back(new AbstractSolutionData(
                    g_root_task(), pat, rng));
            // variable is now associated with the newest solution
            solution_lookup[var] = solutions.size()-1;
        } else {
            cout << token << "var" << var << " will be added to pattern" << endl;
            cout << token; solution.print_pattern(); cout << endl;
            // add variable to current pattern
            Pattern pat(solution.get_pattern());
            pat.push_back(var);
            std::sort(pat.begin(), pat.end());
            std::unique_ptr<AbstractSolutionData> extended(
                    new AbstractSolutionData(g_root_task(), pat, rng));
            solutions.at(sol_index) = std::move(extended);
            solution_lookup[var] = sol_index;
        }
    }
}

static shared_ptr<PatternCollectionGenerator> _parse(
        options::OptionParser& parser) {
    parser.add_option<int>(
            "max_refinements",
            "maximum allowed number of refinements",
            "-1");
    std::vector<std::string> additivity_options;
    additivity_options.emplace_back("FULL");
    additivity_options.emplace_back("PARTIAL");
    additivity_options.emplace_back("NONE");
    parser.add_enum_option(
            "additivity",
            additivity_options,
            "type of additivity that should hold between all patters",
            "FULL");
    std::vector<std::string> blacklisting_options;
    blacklisting_options.emplace_back("NO_BLACKLISTING");
    blacklisting_options.emplace_back("NAIVE");
    blacklisting_options.emplace_back("ADAPTIVE");
    parser.add_enum_option(
            "blacklisting",
            blacklisting_options,
            "type of blacklisting that should be employed",
            "NO_BLACKLISTING");
    std::vector<std::string> flaw_options;
    flaw_options.emplace_back("RANDOM_FLAW");
    flaw_options.emplace_back("LEAST_COMMON_FIRST");
    parser.add_enum_option(
            "flaw_selection",
            flaw_options,
            "strategy for flaw selection",
            "LEAST_COMMON_FIRST");
    parser.add_option<bool>(
            "interleaving",
            "Allow the algorithm to interchange abstract solutions."
            "Works only with additivity enabled.",
            "true");
    parser.add_option<int>(
            "max_blacklist_size",
            "Maximum number of variables that can be blacklisted by the algorithm",
            "1"
    );
    parser.add_option<int>(
            "max_blacklist_percentage",
            "Percentage of variables that may be blacklisted. This setting overrides"
            "max_blacklist_size.",
            "-1"
    );
    parser.add_option<int>(
            "max_pdb_size",
            "maximum allowed number of states in a pdb",
            "1000000"
    );
    std::vector<std::string> initial_collection_options;
    initial_collection_options.emplace_back("RANDOM_GOAL");
    initial_collection_options.emplace_back("ALL_GOALS");
    parser.add_enum_option(
            "initial",
            initial_collection_options,
            "initial collection for refinement",
            "ALL_GOALS");
    utils::add_rng_options(parser);

    Options opts = parser.parse();
    if(parser.dry_run())
        return nullptr;

    return make_shared<PatternCollectionGeneratorCegar>(opts);
}

static PluginShared<PatternCollectionGenerator> _plugin("cegar_pdbs", _parse);
}
