#ifndef PDBS_PATTERN_COLLECTION_GENERATOR_CEGAR_H
#define PDBS_PATTERN_COLLECTION_GENERATOR_CEGAR_H

#include "pattern_generator.h"

#include "types.h"
#include "../tasks/pdb_abstracted_task.h"
#include "abstract_solution_data.h"
#include "../utils/rng.h"
#include "../utils/timer.h"

#include <map>

namespace pdbs {

// Wrapper around std::vector which keeps track of flaw-multiplicities
// (i.e. how often a flaw variable was added to the list)
class FlawList {
    std::vector<std::pair<int,int>> flaws;
    std::unordered_map<int,int> variable_count;
public:
    std::pair<int,int> get_least_common_flaw() const {
        auto least_common = flaws[0];
        for (auto flaw : flaws) {
            int flvar = flaw.second;
            if (variable_count.at(flvar) < variable_count.at(least_common.second)) {
                least_common = flaw;
            }
        }
        return least_common;
    }
    std::pair<int,int> operator[](size_t i) const {
        return flaws[i];
    }
    std::pair<int,int> random_flaw(
            std::shared_ptr<utils::RandomNumberGenerator> &rng) const {
        size_t flaw_index = static_cast<size_t>((*rng)(flaws.size()));
        return flaws[flaw_index];
    };
    void emplace_back(size_t sol_index, int flaw_var) {
        // if the variable is already in our flaw list
        if (variable_count.count(flaw_var) == 1) {
            variable_count[flaw_var]++;
        } else {
            variable_count[flaw_var] = 1;
        }
        flaws.emplace_back(sol_index, flaw_var);
    }
    bool empty() const {
        return flaws.empty();
    }
    void clear() {
        flaws.clear();
        variable_count.clear();
    }
};

class PatternCollectionGeneratorCegar : public PatternCollectionGenerator {
    std::string token = "CEGAR_PDBs: ";
    // This flag is used by various methods to signal the main loop
    // of the algorithm, that it should terminate
    bool terminate = false;
    // The value of this constant tells the algorithm that the abstract
    // solution has failed in the concrete state-space because some
    // unknown goal variable was not satisfied.
    // If this value shows up in the list of flaw variables, then it is
    // a signal for the algorithm to randomly (or intelligently) choose
    // a goal variable that is not yet in the pattern (collection) and
    // use it for refinement.
    const int RANDOM_GOAL_VARIABLE = -1;
    std::shared_ptr<utils::RandomNumberGenerator> rng;

    std::vector<std::unique_ptr<AbstractSolutionData>> solutions;
    TaskProxy concrete_task_proxy;
    std::vector<int> goals;

    // Takes a variable as key and returns the index of the solutions-entry
    // whose pattern contains said variable. Used for checking if a variable
    // is already included in some pattern as well as for quickly finding
    // the other partner for merging.
    std::unordered_map<int,int> solution_lookup;

    // Matrix of booleans which indicate whether there is a correlation
    // between two variables. Two variables x and y are considered correlated
    // if at least one of the following two conditions is met:
    // 1. there exists an operator which has a precondition on x and an effect on y.
    // 2. there exists an operator which has an effect on both x and y.
    // This matrix is used for quickly performing additivity checks, since
    // two patterns are only additive if all their variables are not correlated.
    std::vector<std::vector<bool>> correlation_matrix;

    // behavior defining parameters
    const int max_refinements;
    const int max_pdb_size;
    const bool interleaving;
    enum Additivity {
        FULL,
        PARTIAL,
        NONE
    };
    enum Blacklisting {
        NO_BLACKLISTING,
        NAIVE,
        ADAPTIVE
    };
    bool blacklisting;
    int max_blacklist_size;
    std::unordered_set<int> blacklist;
    const Additivity additivity;
    enum InitialCollectionType {
        RANDOM_GOAL,
        ALL_GOALS
    };
    const InitialCollectionType initial;

    enum FlawSelectionStrategy {
        RANDOM_FLAW,
        LEAST_COMMON_FIRST,
    };
    FlawSelectionStrategy flaw_selection_strategy;

    int refinement_counter = 0;
    // If the algorithm finds a single solution instance that solves
    // the concrete problem, then it will store its index here.
    // This enables simpler plan extraction later on.
    int concrete_solution_index;
    // When employing partial additivity, some flaws may become not
    // fixable. For example when a pattern fails because of a variable
    // and said variable is already in its own pattern, but the two are
    // never merged, because they are partially additive
    int unfixable_flaws = 0;

    utils::Timer runtime_timer;

    // Takes a list of indices to AbstractSolutionData instances in
    // the solutions-vector and checks whether pattern [var] and the
    // patterns referenced in the merge_list can be merged without
    // exceeding the maximal allowed PDB size. If var=-1 then, the
    // function checks only if entries from merge_list can be merged.
    bool can_merge(const std::vector<unsigned int> &merge_list, int var=-1) const;
    bool termination_conditions_met() const;
    FlawList get_flaws();
    FlawList get_flaws_with_interleaving();
    FlawList get_flaws_with_greedy_interleaving();
    FlawList get_flaws_without_interleaving();
    // checks if pattern {v} would be additive with the given pattern
    bool are_additive(const Pattern& pattern, int v) const;
    void extract_plan();
    void generate_trivial_solution_collection(TaskProxy& task_proxy);
    std::pair<int,int> pick_flaw(const FlawList &flaws);
    void refine(const FlawList& flaws);
    void refine_without_additivity(
            int var, bool is_goal_var, unsigned int sol_index);
    void refine_with_additivity(int var);
    /*
      Try to apply the specified abstract solution
      in concrete space by starting with the specified state.
      Return the last state that could be reached before the
      solution failed (if the solution did not fail, then the
      returned state is a goal state of the concrete task).
      The second element of the returned pair is a list of variables
      that caused the solution to fail.
     */
    std::pair<State,std::vector<FlawVariable>> apply_solution(
            AbstractSolutionData &solution,
            const State &concrete_state);

public:
    explicit PatternCollectionGeneratorCegar(const options::Options &opts);
    virtual ~PatternCollectionGeneratorCegar() = default;

    PatternCollectionInformation generate(
            const std::shared_ptr<AbstractTask> &task) override;
};
}


#endif //PDBS_PATTERN_COLLECTION_GENERATOR_CEGAR_H
