#ifndef PDBS_ABSTRACT_SOLUTION_DATA_H
#define PDBS_ABSTRACT_SOLUTION_DATA_H

#include <utility>
#include <deque>

#include "pattern_database.h"
#include "../tasks/pdb_abstracted_task.h"
#include "../task_utils/successor_generator.h"
#include "../utils/rng.h"

namespace pdbs {

using FlawVariable = int;

class AbstractSolutionData {
    TaskProxy concrete_task_proxy;
    std::shared_ptr<PatternDatabase> pdb;
    tasks::PDBAbstractedTask abstracted_task;
    TaskProxy abs_task_proxy;
    std::deque<OperatorID> plan;
    bool is_solvable;
    size_t op_index;

    class AstarSearchNode {
        OperatorID op;
        State state;
        unsigned int g;
        unsigned int h;
        std::shared_ptr<AstarSearchNode> prev;
    public:
        explicit AstarSearchNode(
                OperatorID oper,
                State st,
                unsigned int gval,
                unsigned int hval,
                std::shared_ptr<AstarSearchNode> previous)
                : op(oper), state(std::move(st)), g(gval), h(hval), prev(std::move(previous)) {}
        bool operator<(const AstarSearchNode &other) {
            return get_f() > other.get_f();
        }
        unsigned int get_f() const { return g+h; }
        unsigned int get_g() const { return g; }
        unsigned int get_h() const { return h; }
        const State &get_state() const { return state; }
        const std::shared_ptr<AstarSearchNode> &get_prev() const { return prev; }
        OperatorID get_operator_id() const { return op; }
    };

    void find_plan_astar(
            State &init,
            const successor_generator::SuccessorGenerator& succ_gen,
            const std::shared_ptr<utils::RandomNumberGenerator> &rng);
    void find_plan_greedy(
            State &init,
            const successor_generator::SuccessorGenerator& succ_gen,
            const std::shared_ptr<utils::RandomNumberGenerator> &rng);
public:
    AbstractSolutionData(
        const std::shared_ptr<AbstractTask> &parent,
        Pattern& pattern,
        const std::shared_ptr<utils::RandomNumberGenerator> &rng);

    inline const Pattern &get_pattern() const {
        return pdb->get_pattern();
    }

    inline const std::shared_ptr<PatternDatabase> &get_pdb() {
        return pdb;
    }

    inline bool solution_exists() const {
        return is_solvable;
    }

    inline void print_pattern() const {
        std::cout << "[";
        for(auto var : pdb->get_pattern()) {
            std::cout << var << ",";
        }
        std::cout << "]";
    }

    inline const TaskProxy &get_proxy() const {
        return abs_task_proxy;
    }

    inline const std::deque<OperatorID> &get_plan() const {
        return plan;
    }

    inline int get_plan_cost() const {
        int sum_cost = 0;
        for(auto opid : plan) {
            sum_cost += concrete_task_proxy.get_operators()[opid].get_cost();
        }
        return sum_cost;
    }

    inline size_t get_plan_length() const {
        return plan.size();
    }

    inline bool plan_finished() const {
        return op_index >= get_plan_length();
    }

    inline void advance() {
        assert(!plan_finished());
        op_index++;
    }

    inline void reset() {
        op_index = 0;
    }

    inline OperatorID get_current_operator() const {
        assert(!plan_finished());
        return plan[op_index];
    }

    void print_plan() const;


};
}


#endif
