/**
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.matthatem.ai.msa.heuristics;

import com.matthatem.ai.msa.MSAHeuristic;
import com.matthatem.ai.msa.SubMatrix;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

import com.matthatem.ai.msa.MSA.MSAState;
import com.matthatem.ai.msa.heuristics.HeuristicAFDivConq.TYPE;

import ilog.concert.*;
import ilog.cplex.*;

/**
 * A heuristic that uses a combination of 2D and 3D heuristics.
 * 
 * @author Matthew Hatem
 */
public class HeuristicPHO implements MSAHeuristic {

  private MSAHeuristic[] h3dTable; 
  private MSAHeuristic[] h2dTable; 
  private int[][] h3dIndex;
  private int[][] h2dIndex;
  private int[][] abstraction;
  private int[] abstotable;
  private int h2dSize, h3dSize;
  private HashMap<Integer,ArrayList<Integer>> conflicts;
  
  private IloCplex cplex;
  
  public static enum TYPE {DOUBLE, INT};
  
  public HeuristicPHO(char[][] seqs, SubMatrix sm, double weight, int[][] abstraction) {
    this(seqs, sm, TYPE.DOUBLE, weight, abstraction);  
  }
  
  public HeuristicPHO(char[][] seqs, SubMatrix sm, TYPE type, double weight, int[][] abstraction) {
      //fill buckets for strictly conflicting pattern collections
      conflicts = new HashMap();
      this.abstraction = abstraction;
      for(int a = 0; a < this.abstraction.length; a++) {
    	  int[] abs = this.abstraction[a];
    	  for(int i = 0; i < abs.length-1; i++) {
    		  for(int j = i+1; j < abs.length;j++) {
    			  int key = abs[i]*10+abs[j];
    			  ArrayList<Integer> conf = new ArrayList<Integer>();
    			  if(conflicts.get(key) != null){
    				  conf.addAll(conflicts.get(key));
    			  }
    			  conf.add(a);
    			  conflicts.put(key,conf);
    		  }
    	  }
      }
      System.out.println("conflict buckets:\n"+conflicts.toString());
	  
      //map abstraction index to h2dtable and h3dtable
      abstotable = new int[this.abstraction.length];
	  //get sizes
	  h2dSize = 0;
	  h3dSize = 0;
	  for(int i = 0;i<this.abstraction.length;i++) {
		  if(this.abstraction[i].length == 2){
			  abstotable[i]=h2dSize;
			  h2dSize++;
		  }
		  else if(this.abstraction[i].length == 3){
			  abstotable[i]=h3dSize;
			  h3dSize++;
		  }
	  }
	  
	  //init tables and indices
	  if(h3dSize>0) {
		  h3dTable = new Heuristic3D[h3dSize];
		  h3dIndex = new int[h3dSize][3];
	  }
	  if(h2dSize>0) {
		  h2dTable = new Heuristic2D[h2dSize]; 
		  h2dIndex = new int[h2dSize][2];
	  }
      
      //fill tables
      int h2di = 0, h3di = 0;
      for(int i = 0; i<this.abstraction.length;i++) {
    	  if(this.abstraction[i].length == 2) {
              char[][] seqSub = new char[2][];
              h2dIndex[h2di] = this.abstraction[i];
              seqSub[0] = new String(seqs[h2dIndex[h2di][0]]).trim().toCharArray();
              seqSub[1] = new String(seqs[h2dIndex[h2di][1]]).trim().toCharArray();
              h2dTable[h2di] = new Heuristic2D(seqSub, sm, weight);
			  h2di++;
		  }
		  else if(abstraction[i].length == 3) {
		      char[][] seqSub = new char[3][];
		      h3dIndex[h3di] = this.abstraction[i];
		      seqSub[0] = new String(seqs[h3dIndex[h3di][0]]).trim().toCharArray();
		      seqSub[1] = new String(seqs[h3dIndex[h3di][1]]).trim().toCharArray();
		      seqSub[2] = new String(seqs[h3dIndex[h3di][2]]).trim().toCharArray();
		      h3dTable[h3di] = new Heuristic3D(seqSub, sm, weight);
		      h3di++;
		  }
      }
      
    try {
		cplex = new IloCplex();
		cplex.setOut(null);
	} catch (IloException e) {
		// TODO Auto-generated catch block
		e.printStackTrace();
	}
  }
  
  public double getInitH() {
	  try {
		  cplex.clearModel();
		  IloNumVar[] weights = cplex.numVarArray(h3dSize+h2dSize,0.0,1.0);
		  IloLinearNumExpr expr = cplex.linearNumExpr();
		  for(int i = 0; i < this.abstraction.length;i++) {
			  if(this.abstraction[i].length == 2)
				  expr.addTerm(h2dTable[abstotable[i]].getInitH(), weights[i]);
			  else if(this.abstraction[i].length == 3)
				  expr.addTerm(h3dTable[abstotable[i]].getInitH(), weights[i]);
		  }
		  
		  for(HashMap.Entry<Integer, ArrayList<Integer>> entry : conflicts.entrySet()){
			  ArrayList<Integer> indizes = entry.getValue();
			  IloLinearNumExpr sum = cplex.linearNumExpr();
			  for(int i = 0; i < indizes.size();i++)
			  {
				  sum.addTerm(1.0, weights[indizes.get(i)]);
			  }
			  cplex.addLe(sum, 1.0);
		  }
		  cplex.addMaximize(expr);
		  cplex.solve();
		  double[] ws = cplex.getValues(weights);
		  for(int i = 0; i < ws.length;i++){
			  System.out.print(ws[i]+" ");
		  }
		  System.out.print("\n");
		  cplex.exportModel("cplex_model.lp");
		  return (int)cplex.getObjValue();
	  } catch(IloException e){
		  e.printStackTrace();
	  }
	  return -1;
  }

  public double getH(MSAState state, int[] delta, int[] index) {
    throw new IllegalArgumentException();
  }
  
  public double getH(MSAState state, int[] delta) {
	  try {
		  cplex.clearModel();
		  IloNumVar[] weights = cplex.numVarArray(h3dSize+h2dSize,0.0,1.0);
		  IloLinearNumExpr expr = cplex.linearNumExpr();
		  for(int i = 0; i < this.abstraction.length;i++) {
			  if(this.abstraction[i].length == 2)
				  expr.addTerm(h2dTable[abstotable[i]].getH(state, delta, h2dIndex[abstotable[i]]), weights[i]);
			  else if(this.abstraction[i].length == 3)
				  expr.addTerm(h3dTable[abstotable[i]].getH(state, delta, h3dIndex[abstotable[i]]), weights[i]);
		  }
		  
		  for(HashMap.Entry<Integer, ArrayList<Integer>> entry : conflicts.entrySet()){
			  ArrayList<Integer> indizes = entry.getValue();
			  IloLinearNumExpr sum = cplex.linearNumExpr();
			  for(int i = 0; i < indizes.size();i++)
			  {
				  sum.addTerm(1.0, weights[indizes.get(i)]);
			  }
			  cplex.addLe(sum, 1.0);
		  }
		  cplex.addMaximize(expr);
		  cplex.solve();
		  return (int)cplex.getObjValue();
	  } catch(IloException e){
		  e.printStackTrace();
	  }
	  return -1;
  }

}
