/**
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.matthatem.ai.msa.heuristics;

import com.matthatem.ai.msa.MSAHeuristic;
import com.matthatem.ai.msa.SubMatrix;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;

import com.matthatem.ai.msa.MSA.MSAState;
import com.matthatem.ai.msa.heuristics.HeuristicAFDivConq.TYPE;

/**
 * A heuristic that uses a combination of 2D and 3D heuristics.
 * 
 * @author Matthew Hatem
 */
public class HeuristicGZOCP implements MSAHeuristic {
	
    
  private static final int DG = 0;
  private static final int HZ = 1;
  private static final int VT = 2;

  private int lrGapCost;
  private MSAHeuristic[] h3dTable; 
  private MSAHeuristic[] h2dTable; 
  private int[][] h3dIndex;
  private int[][] h2dIndex;
  private int[][] abstraction;
  private int[] abstotable;
  private int h2dSize, h3dSize;
  private HashMap<Integer,ArrayList<Integer>> conflicts;
  
  private boolean[] OpsUsed;
  private int[][][][] OpsTables;
  private double D[][];
  
  public static enum TYPE {DOUBLE, INT};
  
  public HeuristicGZOCP(char[][] seqs, SubMatrix sm, double weight, int[][] abstraction) {
    this(seqs, sm, TYPE.DOUBLE, weight, abstraction);  
  }
  
  public HeuristicGZOCP(char[][] seqs, SubMatrix sm, TYPE type, double weight, int[][] abstraction) {
	  D = sm.getTable();
	  lrGapCost = (int) sm.getLinearGapCost();
      //fill buckets for strictly conflicting pattern collections
      conflicts = new HashMap();
      this.abstraction = abstraction;
      for(int a = 0; a < this.abstraction.length; a++) {
    	  int[] abs = this.abstraction[a];
    	  for(int i = 0; i < abs.length-1; i++) {
    		  for(int j = i+1; j < abs.length;j++) {
    			  int key = abs[i]*10+abs[j];
    			  ArrayList<Integer> conf = new ArrayList<Integer>();
    			  if(conflicts.get(key) != null){
    				  conf.addAll(conflicts.get(key));
    			  }
    			  conf.add(a);
    			  conflicts.put(key,conf);
    		  }
    	  }
      }
      System.out.println("conflict buckets:\n"+conflicts.toString());
      
      //map abstraction index to h2dtable and h3dtable
      abstotable = new int[this.abstraction.length];
	  //get sizes
	  h2dSize = 0;
	  h3dSize = 0;
	  for(int i = 0;i<this.abstraction.length;i++) {
		  if(this.abstraction[i].length == 2){
			  abstotable[i]=h2dSize;
			  h2dSize++;
		  }
		  else if(this.abstraction[i].length == 3){
			  abstotable[i]=h3dSize;
			  h3dSize++;
		  }
	  }
	  
	  //init tables and indices
	  if(h3dSize>0) {
		  h3dTable = new MSAHeuristic[h3dSize];
		  h3dIndex = new int[h3dSize][3];
	  }
	  if(h2dSize>0) {
		  h2dTable = new MSAHeuristic[h2dSize]; 
		  h2dIndex = new int[h2dSize][2];
	  }
	  
      //create operator tables
      OpsTables = new int[(seqs.length-1)*10+seqs.length][][][];
      for(int i = 0; i < seqs.length-1;i++){
    	  for(int j = i+1; j < seqs.length;j++){
    		  char[] a = new String(seqs[i]).trim().toCharArray();
    	      char[] b = new String(seqs[j]).trim().toCharArray();
    	      OpsTables[i*10+j] = initOps(a,b);
    	  }
      }
      
      OpsUsed = new boolean[(seqs.length-1)*10+seqs.length];
      Arrays.fill(OpsUsed, Boolean.FALSE);
      
      //fill tables
      int h2di = 0, h3di = 0;
      for(int i = 0; i<this.abstraction.length;i++) {
    	  if(this.abstraction[i].length == 2) {
    		  System.out.println("abslength 2");
              char[][] seqSub = new char[2][];
              h2dIndex[h2di] = this.abstraction[i];
              seqSub[0] = new String(seqs[h2dIndex[h2di][0]]).trim().toCharArray();
              seqSub[1] = new String(seqs[h2dIndex[h2di][1]]).trim().toCharArray();
              if(!OpsUsed[h2dIndex[h2di][0]*10+h2dIndex[h2di][1]]){
            	  h2dTable[h2di] = new Heuristic2D(seqSub, sm, weight);
            	  OpsUsed[h2dIndex[h2di][0]*10+h2dIndex[h2di][1]] = true;
            	  OpsTables[h2dIndex[h2di][0]*10+h2dIndex[h2di][1]] = new int[seqSub[0].length+1][seqSub[1].length+1][3];
              }
              else{
            	  h2dTable[h2di] = new HeuristicZero();
              }
              //System.out.println(h2dTable[h2di].toString());
			  h2di++;
		  }
		  else if(abstraction[i].length == 3) {
			  System.out.println("abslength 3");
		      char[][] seqSub = new char[3][];
		      h3dIndex[h3di] = this.abstraction[i];
		      seqSub[0] = new String(seqs[h3dIndex[h3di][0]]).trim().toCharArray();
		      seqSub[1] = new String(seqs[h3dIndex[h3di][1]]).trim().toCharArray();
		      seqSub[2] = new String(seqs[h3dIndex[h3di][2]]).trim().toCharArray();
		      
		      int OpsTablesSub[][][][] = new int[3][][][];
		      OpsTablesSub[0] = OpsTables[h3dIndex[h3di][0]*10+h3dIndex[h3di][1]];
		      OpsTablesSub[1] = OpsTables[h3dIndex[h3di][0]*10+h3dIndex[h3di][2]];
		      OpsTablesSub[2] = OpsTables[h3dIndex[h3di][1]*10+h3dIndex[h3di][2]];
		      //System.out.println(this.printOps(OpsTablesSub[2]));
		      if(!(OpsUsed[h3dIndex[h3di][0]*10+h3dIndex[h3di][1]] 
		    		  && OpsUsed[h3dIndex[h3di][0]*10+h3dIndex[h3di][2]]
		    				  && OpsUsed[h3dIndex[h3di][1]*10+h3dIndex[h3di][2]])){
		      h3dTable[h3di] = new Heuristic3DFactoredOps(seqSub, sm, weight, OpsTablesSub);
		      } else
		    	  h3dTable[h3di] = new HeuristicZero();
		      OpsUsed[h3dIndex[h3di][0]*10+h3dIndex[h3di][1]] = true;
        	  OpsTables[h3dIndex[h3di][0]*10+h3dIndex[h3di][1]] = new int[seqSub[0].length+1][seqSub[1].length+1][3];
        	  OpsUsed[h3dIndex[h3di][0]*10+h3dIndex[h3di][2]] = true;
        	  OpsTables[h3dIndex[h3di][0]*10+h3dIndex[h3di][2]] = new int[seqSub[0].length+1][seqSub[2].length+1][3];
        	  OpsUsed[h3dIndex[h3di][1]*10+h3dIndex[h3di][2]] = true;
        	  OpsTables[h3dIndex[h3di][1]*10+h3dIndex[h3di][2]] = new int[seqSub[1].length+1][seqSub[2].length+1][3];
		      //System.out.println(test.toString());
		      //System.out.println(h3dTable[h3di].toString());
		      h3di++;
		  }
      }
  }
  
  private int[][][] initOps(char[] A, char[] B) {
	  final int x = A.length;
	  final int y = B.length;
	  int Ops[][][] = new int[x+1][y+1][3];
	  Ops[0][0][DG] = Ops[0][0][HZ] = Ops[0][0][VT] = 0;
	  for(int xpos = x-1; xpos >= 0; --xpos)
	  {
		  Ops[xpos][y][VT] = lrGapCost;
	  }
	  for(int ypos = y-1; ypos >= 0; --ypos)
	  {
		  Ops[x][ypos][HZ] = lrGapCost;
	  }
	  for(int xpos = x-1; xpos >= 0; --xpos)
	  {
		  for(int ypos = y-1; ypos >= 0; --ypos)
		  {
		  Ops[xpos][ypos][VT] = lrGapCost;
		  Ops[xpos][ypos][HZ] = lrGapCost;
		  Ops[xpos][ypos][DG] = (int) D[A[xpos]][B[ypos]];
		  }
	  } 
	  return Ops;
}
  
  public double getInitH() {
	  int h = 0;
	  for(int i = 0; i < this.abstraction.length;i++) {
		  if(this.abstraction[i].length == 2){
			  int dh = (int) h2dTable[abstotable[i]].getInitH();
			  //System.out.println(dh);
			  h+=dh;
		  }
		  else if(this.abstraction[i].length == 3){
			  int dh=(int)h3dTable[abstotable[i]].getInitH();
			  //System.out.println(dh);
			  h+=dh;
		  }
	  }
	  return h;
  }

  public double getH(MSAState state, int[] delta, int[] index) {
    throw new IllegalArgumentException();
  }
  
  public double getH(MSAState state, int[] delta) {
	  int h = 0;
		  for(int i = 0; i < this.abstraction.length;i++) {
			  if(this.abstraction[i].length == 2)
				  h+=h2dTable[abstotable[i]].getH(state, delta, h2dIndex[abstotable[i]]);
			  else if(this.abstraction[i].length == 3)
				  h+=h3dTable[abstotable[i]].getH(state, delta, h3dIndex[abstotable[i]]);
		  }
	return h;
  }
  
  public String printOps(int[][][] ops) {
	    StringBuffer sb = new StringBuffer();
	      for (int x=0; x<ops.length; x++) {
	        for (int y=0; y<ops[0].length; y++) {
	            sb.append("AB: "+x+" "+y+"   ");
	            sb.append("["+(int)ops[x][y][DG]+" "+(int)ops[x][y][HZ]+" "+(int)ops[x][y][VT]+"]  ");
	          }
	          sb.append("\n");
	      }
	    sb.append("\n\n\n");
	    return sb.toString();
	  }

}
