/**
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.matthatem.ai.msa.heuristics;

import com.matthatem.ai.msa.MSAHeuristic;
import com.matthatem.ai.msa.SubMatrix;
import com.matthatem.ai.msa.MSA.MSAState;

/**
 * A 3D heuristic for MSA with affine gap costs and integer values.
 * 
 * @author Matthew Hatem
 */
public class Heuristic3DFactoredOps implements MSAHeuristic {
  
	  private static final int DG = 0;
	  private static final int HZ = 1;
	  private static final int VT = 2;
	  
	  int x;
	  int y;
	  int z;
  private int afGapCost;
  private int tmGapCost;
  private int lrGapCost;
  private double weight;
  
  private char[][] seqs;
  private double D[][];
  private int[][][][][][] scoreTable;
  
  private int[][][][] Ops;
  
  private int n;
  private int[][] ni = new int[3][3];
  
  
  public Heuristic3DFactoredOps(char[][] seqs, SubMatrix sm, double weight, int[][][][] Ops) {
    D = sm.getTable();
    this.Ops = Ops;
    this.seqs = seqs;
    lrGapCost = (int) sm.getLinearGapCost();
    this.weight = weight;
    compute();
  }

  private void compute() {
    scoreTable = new int[seqs.length][seqs.length][seqs.length][][][];
    n = seqs.length/3;
    
    for (int i=0; i<n; i++) {
      ni[i][0] = i*3; ni[i][1] = (i*3)+1; ni[i][2] = (i*3)+2;
      char[] A = new String(seqs[ni[i][0]]).trim().toCharArray();
      char[] B = new String(seqs[ni[i][1]]).trim().toCharArray();
      char[] C = new String(seqs[ni[i][2]]).trim().toCharArray();
      scoreTable[ni[i][0]][ni[i][1]][ni[i][2]] = DP2(A, B, C);
    }
  }

private int[][][] DP2(final char[] A, final char[] B, final char[] C) {    
    x = A.length;
    y = B.length;
    z = C.length;
    int P[][][] = new int [x+1][y+1][z+1];
    
    // initialize the starting cell
    P[x][y][z] = 0;
    
    // initialize the x edge
    for (int xpos = x-1; xpos >= 0; --xpos) {
      P[xpos][y][z] = P[xpos+1][y][z] + Ops[0][xpos][y][VT] + Ops[1][xpos][z][VT];
    }
       
    // initialize the y edge
    for (int ypos = y-1; ypos >= 0; --ypos) {
      P[x][ypos][z] = P[x][ypos+1][z] + Ops[0][x][ypos][HZ] + Ops[2][ypos][z][VT];
    }
    
    // initialize the z edge
    for (int zpos = z-1; zpos >= 0; --zpos) {
      P[x][y][zpos] = P[x][y][zpos+1] + Ops[1][x][zpos][HZ] + Ops[2][y][zpos][HZ];
    }
    
    //xy
    for (int xpos = x-1; xpos >= 0; --xpos) {
        for (int ypos = y-1; ypos >= 0; --ypos) {
        	P[xpos][ypos][z] = 
        			min(P[xpos+1][ypos][z]+ Ops[0][xpos][ypos][VT] + Ops[1][xpos][z][VT],
        				P[xpos][ypos+1][z]+ Ops[0][xpos][ypos][HZ] + Ops[2][ypos][z][VT],
        				P[xpos+1][ypos+1][z]+ Ops[1][xpos][z][VT] + Ops[2][ypos][z][VT]
        						+Ops[0][xpos][ypos][DG]);
        }
    }
    //xz
    for (int xpos = x-1; xpos >= 0; --xpos) {
        for (int zpos = z-1; zpos >= 0; --zpos) {
        	P[xpos][y][zpos] = 
        			min(P[xpos+1][y][zpos]+ Ops[0][xpos][y][VT] + Ops[1][xpos][zpos][VT],
            				P[xpos][y][zpos+1]+ Ops[1][xpos][zpos][VT] + Ops[2][y][zpos][HZ],
            				P[xpos+1][y][zpos+1]+ Ops[0][xpos][y][VT] + Ops[2][y][zpos][HZ]
            						+Ops[1][xpos][zpos][DG]);
        }
    }
    
    //yz
    for (int ypos = y-1; ypos >= 0; --ypos) {
        for (int zpos = z-1; zpos >= 0; --zpos) {
        	P[x][ypos][zpos] = 
        			min(P[x][ypos+1][zpos] + Ops[0][x][ypos][HZ] + Ops[2][ypos][zpos][VT],
        				P[x][ypos][zpos+1]+ Ops[1][x][zpos][HZ] + Ops[2][ypos][zpos][VT],
        				P[x][ypos+1][zpos+1]+Ops[0][x][ypos][HZ] + Ops[1][x][zpos][HZ]
        						+Ops[2][ypos][zpos][DG]);
        }
    }
    
    for (int xpos = x-1; xpos >= 0; --xpos) {
      for (int ypos = y-1; ypos >= 0; --ypos) {
        for(int zpos = z-1;zpos >=0; --zpos){
    	  P[xpos][ypos][zpos] = 
    	  			min(P[xpos+1][ypos][zpos] + Ops[0][xpos][ypos][VT] + Ops[1][xpos][zpos][VT], 
    	  				P[xpos][ypos+1][zpos] + Ops[0][xpos][ypos][HZ] + Ops[2][ypos][zpos][VT], 
    	  				P[xpos][ypos][zpos+1] + Ops[1][xpos][zpos][VT] + Ops[2][ypos][zpos][HZ], 
    	  				P[xpos+1][ypos+1][zpos] + Ops[1][xpos][zpos][VT] + Ops[2][ypos][zpos][VT] +Ops[0][xpos][ypos][DG], 
    	  				P[xpos+1][ypos][zpos+1] + Ops[0][xpos][ypos][VT] + Ops[2][ypos][zpos][HZ] +Ops[1][xpos][zpos][DG], 
    	  				P[xpos][ypos+1][zpos+1] + Ops[0][xpos][ypos][HZ] + Ops[1][xpos][zpos][HZ] +Ops[2][ypos][zpos][DG], 
    	  				P[xpos+1][ypos+1][zpos+1] + Ops[0][xpos][ypos][DG] + Ops[1][xpos][zpos][DG]+ Ops[2][ypos][zpos][DG]);
        }
      }
    }
    return P;
  }

public String printOps(int[][][] ops) {
    StringBuffer sb = new StringBuffer();
      for (int x=0; x<ops.length; x++) {
        for (int y=0; y<ops[0].length; y++) {
            sb.append("AB: "+x+" "+y+"   ");
            sb.append("["+(int)ops[x][y][DG]+" "+(int)ops[x][y][HZ]+" "+(int)ops[x][y][VT]+"]  ");
          }
          sb.append("\n");
      }
    sb.append("\n\n\n");
    return sb.toString();
  }
/*
public int[][][][] saturate(int[][][][] Ops){
	System.out.println(this.toString());
	//DG
	for (int xpos = x-1; xpos >= 0; --xpos) {
	      for (int ypos = y-1; ypos >= 0; --ypos) {
	        for(int zpos = z-1; zpos >=0; --zpos){
	        	System.out.println(scoreTable[0][1][2][xpos][ypos][zpos]+" - "+scoreTable[0][1][2][xpos+1][ypos+1][zpos+1]+" = "+(scoreTable[0][1][2][xpos][ypos][zpos]-scoreTable[0][1][2][xpos+1][ypos+1][zpos+1]));
	        	System.out.println(scoreTable[0][1][2][xpos][ypos][zpos+1]);//ops 0 dg
	        	System.out.println(scoreTable[0][1][2][xpos][ypos+1][zpos]);//ops 1 dg
	        	System.out.println(scoreTable[0][1][2][xpos+1][ypos][zpos]);//ops 2 dg
	        	System.out.println( Ops[0][xpos][ypos][DG] +" "+ Ops[1][xpos][zpos][DG]+" "+ Ops[2][ypos][zpos][DG]);
	        	System.out.println("________");
	        }
	      }
	}
	return Ops;
}*/
  
  public double getH(MSAState state, int[] delta, int index[]) {
    int[] pos = state.pos;
    int cost = 0;
    for (int p=0; p<n; p++) {
      int h = ni[p][0]; int i = ni[p][1]; int j = ni[p][2];
      int x = pos[index[0]]; int y = pos[index[1]]; int z = pos[index[2]];
          cost += scoreTable[h][i][j][x][y][z];
    }
    return weight*cost;
  }
  
  public double getH(MSAState state, int[] delta) {
    int[] pos = state.pos;
    int cost = 0;
    for (int p=0; p<n; p++) {
      int h = ni[p][0]; int i = ni[p][1]; int j = ni[p][2];
      int x = pos[h]; int y = pos[i]; int z = pos[j];
      cost += scoreTable[h][i][j][x][y][z];
    }
    return weight*cost;
  }
  
  public double getInitH() {
    double cost = 0.0f;
    for (int p=0; p<n; p++) {
      int h = ni[p][0]; int i = ni[p][1]; int j = ni[p][2];
      cost += scoreTable[h][i][j][0][0][0];
    }
    return weight*cost;
  }
  
  private static final int min(int x, int y, int z) {
    return Math.min(Math.min(x, y), z);
  }
  
  private static final int min(int x, int y, int z, int xy, 
      int yz, int zx, int xyz) {
    // TODO maybe something more efficient here!!
    return Math.min(Math.min(Math.min(Math.min(Math.min(Math.min(x, y), z), xy), yz), zx), xyz);
  }
  
  public void printTable(double t[][][], char[] A, char[] B, char[] C) {
      final int x = A.length;
      final int y = B.length;
      final int z = C.length;
      
      // XY face
      StringBuffer sb = new StringBuffer();
      for (int ypos=y; ypos>=0; ypos--) {
        for (int xpos=x; xpos>=0; xpos--) {
          sb.append("XYZ: "+xpos+" "+ypos+" "+z+"   ");
          sb.append("["+(int)t[xpos][ypos][z]+"]  ");
        }
        sb.append("\n");
      }
      System.out.println(sb.toString());
      
      // YZ face
      sb = new StringBuffer();
      for (int zpos=z; zpos>=0; zpos--) {
        for (int ypos=y; ypos>=0; ypos--) {
          sb.append("XYZ: "+ypos+" "+zpos+" "+z+"   ");
          sb.append("["+(int)t[x][ypos][zpos]+"]  ");
        }
        sb.append("\n");
      }
      System.out.println(sb.toString());
      
      // ZX face
      sb = new StringBuffer();
      for (int xpos=x; xpos>=0; xpos--) {
        for (int zpos=z; zpos>=0; zpos--) {
          sb.append("XYZ: "+xpos+" "+zpos+" "+z+"   ");
          sb.append("["+(int)t[xpos][y][zpos]+"]  ");
        }
        sb.append("\n");
      }
      System.out.println(sb.toString());
  }
  public String toString() {
    StringBuffer sb = new StringBuffer();
    for (int p=0; p<n; p++) {
      int h = ni[p][0]; int i = ni[p][1]; int j = ni[p][2];
      int t[][][] = scoreTable[h][i][j];
      
      for (int x=0; x<t.length; x++) {
        for (int y=0; y<t[x].length; y++) {
          for (int z=0; z<t[x][y].length; z++) {
            sb.append("XYZ: "+x+" "+y+" "+z+"   ");
            sb.append("["+(int)t[x][y][z]+"]  ");
          }
          sb.append("\n");
        }
        sb.append("\n\n");
      }
      
    }
    sb.append("\n\n\n");
    return sb.toString();
  }
}
