/**
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.matthatem.ai.msa.heuristics;

import com.matthatem.ai.msa.MSAHeuristic;
import com.matthatem.ai.msa.SubMatrix;
import com.matthatem.ai.msa.MSA.MSAState;

/**
 * A 2D heuristic for MSA with affine gap costs.
 * 
 * @author Matthew Hatem
 */
public class Heuristic2D implements MSAHeuristic {
  
  private double lrGapCost;
  private double weight;
  
  private char[][] seqs;
  private double D[][];
  private double[][][][] scoreTable;
    
  public Heuristic2D(char[][] seqs, SubMatrix sm, double weight) {
    D = sm.getTable();

    this.seqs = seqs;
    lrGapCost = sm.getLinearGapCost();
    this.weight = weight;
    compute();
    //System.out.println(scoreTable[1][0][0][0]);
  }

  private void compute() {
    scoreTable = new double[seqs.length][seqs.length][][];
    for (int i=1; i<seqs.length; i++) {
      for (int j=0; j<i; j++) {
        char[] A = new String(seqs[i]).trim().toCharArray();
        char[] B = new String(seqs[j]).trim().toCharArray();
        scoreTable[i][j] = DP2(A, B);
      }
    }
  }
  
  private double[][] DP2(char[] A, char[] B) {
    int i, j;
    int pos, end;
    int n = A.length;
    int m = B.length;
    double P[][] = new double [n+1][m+1];

    end = m;
    P[n][end] = 0;
    pos = end-1;
    for (j = m - 1; j >= 0; --j, pos -= 1) {
      P[n][pos] = P[n][pos+1] + lrGapCost;
    }
    for (i = n - 1; i >= 0; --i) {
      P[i][end] = P[i + 1][end] + lrGapCost;

      for (j = m - 1; j >= 0; --j, pos -= 1) {
        P[i][j] = min(P[i + 1][j]+lrGapCost,P[i + 1][j+1]+D[A[i]][B[j]],P[i][j+1]+lrGapCost) ;

      }
    } 
    return P;
  }
  
  public double getH(MSAState state, int[] delta, int[] index) {
    int[] pos = state.pos;
    double cost = 0.0f;
    for (int i=1; i<seqs.length; i++) {
      for (int j = 0; j < i; j++) {
        int col = pos[index[i]]; int row = pos[index[j]];
        cost += scoreTable[i][j][col][row];
      }
    }
    return weight*cost;
  }
  
  public double getH(MSAState state, int[] delta) {
    int[] pos = state.pos;
    double cost = 0.0f;
    for (int i=1; i<seqs.length; i++) {
      for (int j = 0; j < i; j++) {
        int col = pos[i]; int row = pos[j];
        cost += scoreTable[i][j][col][row];
      }
    }
    return weight*cost;
  }
  
  public double getInitH() {
    double cost = 0.0f;
    for (int i=1; i<seqs.length; i++) {
      for (int j = 0; j < i; j++) {
    	  cost += scoreTable[i][j][0][0];
      }
    }
    return weight*cost;
  }
  
  private static final double min(double x, double y, double z) {
    return Math.min(Math.min(x, y), z);
  }
  
  public String toString() {
    StringBuffer sb = new StringBuffer();    
    for (int i=1; i<seqs.length; i++) {
      for (int j=0; j<i; j++) {
        double t[][] = scoreTable[i][j];
        for (int x=0; x<t.length; x++) {
          for (int y=0; y<t[x].length; y++) {
            sb.append("["+(int)t[x][y]+"]");
          }
          sb.append("\n");
        }
        sb.append("\n\n\n");
      }
    }    
    return sb.toString();
  }

}
