package wordle.java;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.LinkedList;
import java.util.List;
import java.util.Random;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;

 /**
  * Provides different algorithms to find the next guess in Wordle.
  * Entropy calculation supports both String and int representation of words with
  * int representation beeing arround 3-4 times faster.
  * Legacy String representation is not maintained.
  */

public class WordleGuessFinder {
    public Wordle wordle;

    long tt0;
    long feedbackTime = 0;
    long entropyTime = 0;
    int simulatedGames;


    public WordleGuessFinder() {
        this.wordle = new Wordle(-1);
    }

    public WordleGuessFinder(int solutionWord) {
        this.wordle = new Wordle(solutionWord);
    }

    public WordleGuessFinder(String solutionWord) {
        this.wordle = new Wordle(solutionWord);
    }
    
    /**
     * Provides the average entropy for a guess word given a list of possible solution words
     * by counting individual feedbacks received across all possible solution words.
     * Uses String representation of words.
     * @param guessWord Word for which the entropy reduction is calculated
     * @param solutionList List of possible solution words
     */
    public double getEntropyForWordString(String guessWord, List<String> solutionList, int numSolutions) {
        int[] feedbackCountArray = new int[243];
        for (String solution: solutionList) {
            feedbackCountArray[wordle.getFeedbackInt(guessWord, solution)] += 1;
        }
        return computeEntropyFromProbabilityDistributiion(feedbackCountArray, numSolutions);
    }

    /**
     * Provides the average entropy for a guess word given a list of possible solution words
     * by counting individual feedbacks received across all possible solution words.
     * Uses int representation of guess words.
     * @param guessWord Word for which the entropy reduction is calculated
     * @param solutionList List of possible solution words
     */
    public double getEntropyForWord(int guessWord, int[] solutionsWords) {
        int[] feedbackCountArray = new int[243];
        for (int solution = 0; solution < solutionsWords.length; solution++) {
            feedbackCountArray[wordle.getFeedbackInt(guessWord, solutionsWords[solution])] += 1;
        }
        return computeEntropyFromProbabilityDistributiion(feedbackCountArray, solutionsWords.length);
    }

    /**
     * Provides an average entropy value for a given feedback distribution.
     * @param counts Amount of occurences of distinct feedbacks
     * @param numSolutions Amount of total feedbacks
     */
    public double computeEntropyFromProbabilityDistributiion(int[] counts, int numSolutions) {
        double entropy = 0.0;
        double inverseNumSolutions = 1.0 / numSolutions;
        double inverseLog2 = 1.0 / Math.log(2);

        for (int i = 0; i < counts.length; i++) {
            if (counts[i] == 0) continue;

            double probability = counts[i] * inverseNumSolutions;
            entropy -= probability * Math.log(probability) * inverseLog2;

        }
        return entropy;
    }

    /**
     * Provides the guess word with the highest average entropy reduction for a list of possible solution words
     * by using String representation.
     * @param guessList List of guess words to compare
     * @param solutionList List of possible solution words
     */
    public String getHighestEntropyString(List<String>  guessList, List<String> solutionList) {
        int numSolutions = solutionList.size();
        if (numSolutions == 1) return solutionList.get(0);

        double maxEntropy = 0.0;
        String bestGuess = "";

        for (String guess : guessList) {
            double entropy = getEntropyForWordString(guess, solutionList, numSolutions);

            if (entropy > maxEntropy) {
                maxEntropy = entropy;
                bestGuess = guess;
            }
        }
        return bestGuess;
    }

    /**
     * Provides the guess word with the highest average entropy reduction for a list of possible solution words
     * by using int representation.
     * @param guessAmount Guess words to compare
     * @param solutionWords Possible solution words
     */
    public int getHighestEntropy(int guessAmount, int[] solutionWords) {
        if (solutionWords.length == 1) return wordle.solutionToGuessWord[solutionWords[0]];

        double maxEntropy = 0.0;
        int bestGuess = -1;

        for (int guess = 0; guess < guessAmount; guess++) {
            double entropy = getEntropyForWord(guess, solutionWords);

            if (entropy > maxEntropy) {
                maxEntropy = entropy;
                bestGuess = guess;
            }
        }
        return bestGuess;
    }

    /**
     * Provides the guess word with the highest average entropy reduction for a list of possible solution words
     * by using int representation. Uses tie braking based on if guess words are solution candiates.
     * @param guessAmount Guess words to compare
     * @param solutionWords Possible solution words
     */
    public int getHighestEntropyTieBraking(int guessAmount, int[] solutionWords) {
        if (solutionWords.length == 1) return wordle.solutionToGuessWord[solutionWords[0]];

        double maxEntropy = 0.0;
        int bestGuess = -1;

        for (int guess = 0; guess < guessAmount; guess++) {
            double entropy = getEntropyForWord(guess, solutionWords);

            if (entropy > maxEntropy) {
                maxEntropy = entropy;
                bestGuess = guess;
            }
            
            if (entropy == maxEntropy && bestGuess != -1) {
                boolean oldBestInSolutions = contains(wordle.getSolutionIntFromGuessInt(bestGuess), solutionWords);
                boolean newBestInSolutions = contains(wordle.getSolutionIntFromGuessInt(guess), solutionWords);

                if (newBestInSolutions && !oldBestInSolutions) {
                    bestGuess = guess;
                }
            }
        }
        return bestGuess;
    }

    /**
     * Provides a sorted list of guess words with the highest average entropy reduction
     * by using String representaion.
     * @param guessList List of guess words to compare
     * @param solutionList List of possible solution words
     * @param nBest Specifies how many guess words should be provided
     */
    public List<String> getHighestEntropyListString(List<String> guessList, List<String> solutionList, int nBest) {
        int numSolutions = solutionList.size();
    
        if (numSolutions == 1) return solutionList.subList(0, Math.min(nBest, solutionList.size()));
    
        List<GuessEntropyString> guessEntropies = new ArrayList<>(nBest);
    
        for (String guess : guessList) {
            double entropy = getEntropyForWordString(guess, solutionList, numSolutions);
            guessEntropies.add(new GuessEntropyString(guess, entropy));
        }
    
        guessEntropies.sort(GuessEntropyString.descending());
    
        List<String> bestGuesses = new ArrayList<>();
        for (int i = 0; i < Math.min(nBest, guessEntropies.size()); i++) {
            bestGuesses.add(guessEntropies.get(i).guess);
        }
    
        return bestGuesses;
    }

    /**
     * Provides a sorted list of guess words with the highest average entropy reduction
     * by using int representation.
     * @param guessAmount Guess words to compare
     * @param solutionWords Possible solution words
     * @param nBest Specifies how many guess words should be provided
     */
    public int[] getHighestEntropyList(int guessAmount, int[] solutionWords, int nBest) {

        List<GuessEntropy> guessEntropies = new ArrayList<>(nBest);
        GuessEntropy ge = new GuessEntropy(1, 1);

        for (int guess = 0; guess < guessAmount; guess++) {
            double entropy = getEntropyForWord(guess, solutionWords);
            guessEntropies.add(new GuessEntropy(guess, entropy));
        }

        guessEntropies.sort(ge.descending(solutionWords));

        int[] bestGuesses = new int[nBest];

        for (int i = 0; i < nBest; i++) {
            bestGuesses[i] = guessEntropies.get(i).guess;
        }

        return bestGuesses;
    }


    /**
     * Provides the guess word with the highest average entropy reduction for a list of possible solution words
     * by using String representation.
     * Utilizes parallelization to maximize speed.
     * @param guessList List of guess words to compare
     * @param solutionList List of possible solution words
     * @param numThreads Number of threads for parallel execution
     */
    public String getHighestEntropyParallelString(List<String> guessList, List<String> solutionList, int numThreads) {
        int numSolutions = solutionList.size();
        if (numSolutions == 1) return solutionList.get(0);

        ExecutorService executor = Executors.newFixedThreadPool(numThreads);
        List<Future<GuessEntropyString>> futures = new ArrayList<>();

        int chunkSize = guessList.size() / numThreads;  // split list in chunks for parallel execution
        for (int i = 0; i < numThreads; i++) {
            int start = i * chunkSize;
            int end = Math.min(start + chunkSize, guessList.size());
            List<String> guessChunk = guessList.subList(start, end);

            Future<GuessEntropyString> future = executor.submit(() -> {   // start execution of chunk
                double maxEntropy = 0.0;
                String bestGuess = "";
                for (String guess : guessChunk) {
                    double entropy = getEntropyForWordString(guess, solutionList, numSolutions);
                    if (entropy > maxEntropy) {
                        maxEntropy = entropy;
                        bestGuess = guess;
                    }
                }
                return new GuessEntropyString(bestGuess, maxEntropy);
            });
            futures.add(future);
        }

        // wait for threads to finish
        double maxEntropy = 0.0;
        String bestGuess = "";
        for (Future<GuessEntropyString> future : futures) {
            try {
                GuessEntropyString result = future.get();
                if (result.entropy > maxEntropy) {
                    maxEntropy = result.entropy;
                    bestGuess = result.guess;
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        }

        executor.shutdown();
        return bestGuess;
    }

    /**
     * Stores a guess word with its associated entropy value.
     * Helper class for parallel execution of entropy calculation.
     */
    private static class GuessEntropyString {
        String guess;
        double entropy;

        public GuessEntropyString(String guess, double entropy) {
            this.guess = guess;
            this.entropy = entropy;
        }

        public static Comparator<GuessEntropyString> descending() {
            return new Comparator<GuessEntropyString>() {
                @Override
                public int compare(GuessEntropyString guessEntropy1, GuessEntropyString guessEntropy2) {
                    // Compare in descending order: higher entropy comes first
                    return Double.compare(guessEntropy2.entropy, guessEntropy1.entropy);
                }
            };
        }
    }


    /**
     * Provides the guess word with the highest average entropy reduction for a list of possible solution words
     * by using int representation.
     * Utilizes parallelization to maximize speed.
     * @param guessAmount Guess words to compare
     * @param solutionWords Possible solution words
     * @param numThreads Number of threads for parallel execution
     */
    public int getHighestEntropyParallel(int guessAmount, int[] solutionWords, int numThreads) {
        if (solutionWords.length == 1) return wordle.solutionToGuessWord[solutionWords[0]];
    
        ExecutorService executor = Executors.newFixedThreadPool(numThreads);
        List<Future<GuessEntropy>> futures = new ArrayList<>();
    
        int chunkSize = guessAmount / numThreads; // Split the range of guesses
        for (int i = 0; i < numThreads; i++) {
            int start = i * chunkSize;
            int end;
            if (i == numThreads - 1) {
                end = guessAmount;
            } else {
                end = start + chunkSize;
            }
    
            Future<GuessEntropy> future = executor.submit(() -> {
                double maxEntropy = 0.0;
                int bestGuess = -1;
                for (int guess = start; guess < end; guess++) {
                    double entropy = getEntropyForWord(guess, solutionWords);
                    if (entropy > maxEntropy) {
                        maxEntropy = entropy;
                        bestGuess = guess;
                    }
                }
                return new GuessEntropy(bestGuess, maxEntropy);
            });
            futures.add(future);
        }
    
        double maxEntropy = 0.0;
        int bestGuess = -1;
        for (Future<GuessEntropy> future : futures) {
            try {
                GuessEntropy result = future.get();
                if (result.entropy > maxEntropy) {
                    maxEntropy = result.entropy;
                    bestGuess = result.guess;
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
    
        executor.shutdown();
        return bestGuess;
    }

    /**
     * Provides the guess word with the highest average entropy reduction for a list of possible solution words
     * by using int representation. Uses tie braking based on if guess words are solution canditates.
     * Utilizes parallelization to maximize speed.
     * @param guessAmount Guess words to compare
     * @param solutionWords Possible solution words
     * @param numThreads Number of threads for parallel execution
     */
    public int getHighestEntropyParallelTieBraking(int guessAmount, int[] solutionWords, int numThreads) {
        if (solutionWords.length == 1) return wordle.solutionToGuessWord[solutionWords[0]];
    
        ExecutorService executor = Executors.newFixedThreadPool(numThreads);
        List<Future<GuessEntropy>> futures = new ArrayList<>();
    
        int chunkSize = guessAmount / numThreads; // Split the range of guesses
        for (int i = 0; i < numThreads; i++) {
            int start = i * chunkSize;
            int end;
            if (i == numThreads - 1) {
                end = guessAmount;
            } else {
                end = start + chunkSize;
            }
    
            Future<GuessEntropy> future = executor.submit(() -> {
                double maxEntropy = 0.0;
                int bestGuess = -1;
                for (int guess = start; guess < end; guess++) {
                    double entropy = getEntropyForWord(guess, solutionWords);
                    if (entropy > maxEntropy) {
                        maxEntropy = entropy;
                        bestGuess = guess;
                    }
                    if (entropy == maxEntropy && bestGuess != -1) {
                        boolean oldBestInSolutions = contains(wordle.getSolutionIntFromGuessInt(bestGuess), solutionWords);
                        boolean newBestInSolutions = contains(wordle.getSolutionIntFromGuessInt(guess), solutionWords);
        
                        if (newBestInSolutions && !oldBestInSolutions) {
                            bestGuess = guess;
                        }
                    }
                }
                return new GuessEntropy(bestGuess, maxEntropy);
            });
            futures.add(future);
        }
    
        double maxEntropy = 0.0;
        int bestGuess = -1;
        for (Future<GuessEntropy> future : futures) {
            try {
                GuessEntropy result = future.get();
                if (result.entropy > maxEntropy) {
                    maxEntropy = result.entropy;
                    bestGuess = result.guess;
                }
                if (result.entropy == maxEntropy && bestGuess != -1) {
                    boolean oldBestInSolutions = contains(wordle.getSolutionIntFromGuessInt(bestGuess), solutionWords);
                    boolean newBestInSolutions = contains(wordle.getSolutionIntFromGuessInt(result.guess), solutionWords);
    
                    if (newBestInSolutions && !oldBestInSolutions) {
                        bestGuess = result.guess;
                    }
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
    
        executor.shutdown();
        return bestGuess;
    }
    
    /**
     * Stores a guess word with its associated entropy value.
     * Helper class for parallel execution of entropy calculation.
     */
    private class GuessEntropy {
        int guess;
        double entropy;
    
        GuessEntropy(int guess, double entropy) {
            this.guess = guess;
            this.entropy = entropy;
        }

        public Comparator<GuessEntropy> descending(int[] solutionWords) {
            return new Comparator<GuessEntropy>() {
                @Override
                public int compare(GuessEntropy guessEntropy1, GuessEntropy guessEntropy2) {
                    // Compare in descending order: higher entropy comes first
                    double entropy1 = guessEntropy1.entropy;
                    double entropy2 = guessEntropy2.entropy;
                    if (entropy1 > entropy2) return -1;
                    if (entropy2 > entropy1) return 1;
                    
                    boolean oldBestInSolutions = contains(wordle.getSolutionIntFromGuessInt(guessEntropy1.guess), solutionWords);
                    boolean newBestInSolutions = contains(wordle.getSolutionIntFromGuessInt(guessEntropy2.guess), solutionWords);
    
                    if (newBestInSolutions && !oldBestInSolutions) { // advanced tie braking logic. Use commented line below for alphabetical tie braking
                        return -1;
                    } else if (!newBestInSolutions && oldBestInSolutions) {
                        return 1;
                    } else {
                        return 0;
                    }

                    // Double.compare(guessEntropy2.entropy, guessEntropy1.entropy);
                }
            };
        }
    }



    // Rollout


    /**
     * Provides the best next guess according to the rollout algorithm.
     * Uses int representation.
     * @param guessAmount Guess words to compare
     * @param solutionWords Possible solution words
     * @param currentGuessAmount Amount of guesses already done in the game, used for simulation
     * @param nBest Only take into account the best guesses according to entropy heursitic
     */
    public int findNextGuessRollout(int guessAmount, int[] solutionWords, int currentGuessAmount, int nBest, boolean useSolutionWordsAtEndGame) {
        if (solutionWords.length == 1) return wordle.solutionToGuessWord[solutionWords[0]];

        int bestGuess = -1;
        double bestScore = Double.MAX_VALUE;

        int[] newGuessList = getHighestEntropyList(guessAmount, solutionWords, nBest);

        // Append solution words to guess word list at endgame
        if (useSolutionWordsAtEndGame && solutionWords.length <= 10) {
            addSolutionWordsToGuessWordArray(newGuessList, solutionWords);
        }

        for (int guessIndex = 0; guessIndex < newGuessList.length; guessIndex++) {
            int guess = newGuessList[guessIndex];
            int score = 0;
            for (int i = 0; i < solutionWords.length; i++) {
                score += simulateGame(guess, solutionWords[i], currentGuessAmount, guessAmount, solutionWords);
            }
            double avgScore = (double)score / solutionWords.length;

            if (avgScore < bestScore) {
                bestScore = avgScore;
                bestGuess = guess;
            }
            
            if (avgScore == bestScore) {
                int nextGuess = tieBraking(bestGuess, guess, solutionWords);
                if (nextGuess == guess) {
                    bestGuess = guess;
                }
            }
        }
        return bestGuess;
    }

    /**
     * Adds solution words to the guess words array for the rollout algorithm.
     * Only adds words not contained in the guess word array.
     * @return
     */
    public int[] addSolutionWordsToGuessWordArray(int[]currentGuessWords, int[] solutionWords) {
        int[] tempNewGuessList = new int[currentGuessWords.length + solutionWords.length];
            int tempIndex = 0;
            for (int value : currentGuessWords) {
                tempNewGuessList[tempIndex] = value;
                tempIndex++;
            }
            for (int value : solutionWords) {
                boolean exists = false;
                for (int i = 0; i < tempIndex; i++) {
                    if (tempNewGuessList[i] == value) {
                        exists = true;
                        break;
                    }
                }
                if (!exists) {
                    tempNewGuessList[tempIndex] = value;
                    tempIndex++;
                }
            }
            currentGuessWords = new int[tempIndex];
            for (int i = 0; i < tempIndex; i++) {
                currentGuessWords[i] = tempNewGuessList[i];
            }
            return currentGuessWords;
    }

    /**
     * Decides on which guess to use if policy yields equal scores.
     * @param solutionWords List of solution words
     */
    public int tieBraking(int oldBestGuess, int newBestGuess, int[] solutionWords) {
        // Entropy based tie braking
        double oldGuessEntropy = getEntropyForWord(oldBestGuess, solutionWords);
        double newGuessEntropy = getEntropyForWord(newBestGuess, solutionWords);

        
        if (newGuessEntropy > oldGuessEntropy) {
            return newBestGuess;
        } else if (newGuessEntropy < oldGuessEntropy) {
            return oldBestGuess;
        }

        boolean oldBestInSolutions = contains(wordle.getSolutionIntFromGuessInt(oldBestGuess), solutionWords);
        boolean newBestInSolutions = contains(wordle.getSolutionIntFromGuessInt(newBestGuess), solutionWords);

        if (newBestInSolutions && !oldBestInSolutions) {
            return newBestGuess;
        } else { // if only old guess, or no guess is a solution word, keep old best guess (alphabetical order)
            return oldBestGuess;
        }
    }

    public boolean contains(int value, int[] array) {
        for (int i = 0; i < array.length; i++) {
            if (array[i] == value) return true;
        }
        return false;
    }

    /**
     * Simulates a wordle game from any stage using the entropy heuristic.
     * Uses int representation.
     * @param guess The next word that should be guessed in the game
     * @param solution The assumed solution word to the game
     * @param currerntGuessAmount The amount of guesses already done
     * @param guessAmount Guess words to consider
     * @param solutionWords Possible solution words at current stage
     * @return Amount of guesses requierd to finish the game
     */
    public int simulateGame(int guess, int solution, int currerntGuessAmount, int guessAmount, int[] solutionWords) {
        simulatedGames++;
        int nextGuess = guess;

        int feedback = wordle.getFeedbackInt(nextGuess, solution);
        solutionWords = filterWordList(feedback, nextGuess, solutionWords);
        currerntGuessAmount++;

        while (feedback != 0) {
            currerntGuessAmount++;
            //if (currerntGuessAmount > 6) return 1000; // penalty for not finishing a game in at most 6 guesses

            // The next guess is selected according to heurstic. Use comment to decide on which version to use.

            //nextGuess = getHighestEntropyParallel(guessAmount, solutionWords, 7);
            nextGuess = getHighestEntropyParallelTieBraking(guessAmount, solutionWords, 7);
            feedback = wordle.getFeedbackInt(nextGuess, solution);
            solutionWords = filterWordList(feedback, nextGuess, solutionWords);
        }
        return currerntGuessAmount;
    }


    /**
     * Filters the list of possible solution words after a guess is made by only keeping solutions that provide the
     * same feedback for the guess as the game.
     * Uses String representation.
     * @param feedback Feedback recieved on the last guess
     * @param solutionList List of possible solutions to be filtered according to the last guess made
     */
    public List<String> filterWordList(String feedback, String guessWord, List<String> solutionList) {
        List<String> newWordList = new ArrayList<>();

        for (String solutionWord : solutionList) {
            if (wordle.getFeedbackInt(guessWord, solutionWord) == wordle.feedbackToInt(feedback)) newWordList.add(solutionWord);
        }

        return newWordList;
    }

    /**
    * Filters the list of possible solution words after a guess is made by only keeping solutions that provide the
     * same feedback for the guess as the game.
     * Uses int representation.
     * @param feedback Feedback received on the last guess
     * @param solutionWords List of possible solution words o be filtered according to the last guess made
     */
    public int[] filterWordList(int feedback, int guessWord, int[] solutionWords) {
        int[] newWords = new int[solutionWords.length];
        int solutionIndex = 0;

        for (int i = 0; i < solutionWords.length; i++) {
            if (wordle.getFeedbackInt(guessWord, solutionWords[i]) == feedback) {
                newWords[solutionIndex] = solutionWords[i];
                solutionIndex++;
            }
        }

        // inefficient rezise of arry, so it is not longer than it needs to be (its length is used later)
        //newWords = Arrays.copyOf(newWords, solutionIndex);
        int[] ret = new int[solutionIndex];
        for (int i = 0; i < ret.length; i++) {
            ret[i] = newWords[i];
        }
        
        return ret;
    }


    // UCT

    /**
     * A node in the Monte Carlo Search Tree.
     * Stores guess information and handles successor ordering for search.
     */
    public class TreeNode {
        int guess;
        int visitCounter;
        double avgCost;
        ArrayList<TreeNode> visitedSuccessors;
        LinkedList<TreeNode> unvisitedSuccessors;
        int nextGuessSuccessor;
        boolean root;

        public TreeNode(int guess) {
            this.guess = guess;
            this.root = false;
            visitCounter = 0;
            avgCost = 0.0;
            visitedSuccessors = new ArrayList<>();
            nextGuessSuccessor = wordle.getLegalWordAmount() - 1;
        }

        /**
         * Provides next successor based on reverse alphabetical ordering.
         */
        public TreeNode getUnvisitedSuccessor(int[] solutionWords) {
            if (nextGuessSuccessor < 0) return null;
            if (root) return getUnvisitedSuccessorRoot(solutionWords);

            while (nextGuessSuccessor >= 0 && getEntropyForWord(nextGuessSuccessor, solutionWords) == 0) {
                nextGuessSuccessor--;
            }

            if (nextGuessSuccessor < 0) return null;
            return new TreeNode(nextGuessSuccessor);
        }

        /**
         * Provides next unvisited successor, for the root node, only the n best successors according to the entropy heursitic are considerd.
         * Successors are ordered by entropy values to enable tie braking to pick best successor after rollouts are carried out.
         */
        public TreeNode getUnvisitedSuccessorRoot(int[] solutionWords) {
            TreeNode successor = unvisitedSuccessors.pollFirst();
            while (successor != null) {
                if (getEntropyForWord(successor.guess, solutionWords) != 0) return successor;
                successor = unvisitedSuccessors.pollFirst();
            }
            return successor;
        }

    }

    /**
     * Creates root node for monte carlo search by adding n unvisited successors according to entropy values.
     * @param lastGuess
     * @param solutionWords
     * @param nBest
     * @return
     */
    public TreeNode createRootNode(int lastGuess, int[] solutionWords, int nBest) {
        TreeNode ret = new TreeNode(lastGuess);
        ret.root = true;
        int[] nextGuesses = getHighestEntropyList(wordle.getLegalWordAmount(), solutionWords, nBest);
        ret.unvisitedSuccessors = new LinkedList<>();
        for (int guess = 0; guess < nBest; guess++) {
            ret.unvisitedSuccessors.add(new TreeNode(nextGuesses[guess]));
        }
        return ret;
    }

    /**
     * Provides next best guess according to monte carlo search with UCT formula.
     * Each rollout assumes a random solution word from the list of possible solutions.
     * @param lastGuess Previous guess made in the game
     * @param solutionWords Possible solution words
     * @param currentGuessAmount Amount of guesses already made in the game
     * @param nBestRootSuccessors Amount of successors to consider for the root of mcts
     * @param nRollouts Amount of rollouts to be done
     */
    public int UCT(int lastGuess, int[] solutionWords, int currentGuessAmount, int nBestRootSuccessors, int nRollouts, boolean addSolutionWordsAtEndGame) {
        if (solutionWords.length == 1) return wordle.solutionToGuessWord[solutionWords[0]];
        TreeNode root = new TreeNode(lastGuess);
        root = createRootNode(lastGuess, solutionWords, nBestRootSuccessors);

        // Optimization: add solution words to root successors at end game
        if (addSolutionWordsAtEndGame && solutionWords.length <= 10) {
            for (int i = 0; i < solutionWords.length; i++) {
                TreeNode newNode = new TreeNode(wordle.getGuessWordIntFromString(wordle.getSolutionWordFromInt(solutionWords[i])));
                if (!root.unvisitedSuccessors.contains(newNode)) {
                    root.unvisitedSuccessors.add(newNode);
                }
            }
        }

        Random random = new Random(10);
        double bias = 1.0;

        for (int i = 0; i < nRollouts; i++) {
            bias = root.avgCost / 5;
            int solution = solutionWords[random.nextInt(solutionWords.length)];
            visitTreeNode(root, solution, solutionWords, bias);
        }
        
        double bestCost = Double.MAX_VALUE;
        int bestGuess = -1;
        for (TreeNode successor : root.visitedSuccessors) {
            if (successor.avgCost < bestCost) {
                bestCost = successor.avgCost;
                bestGuess = successor.guess;
            }
            if (successor.avgCost == bestCost) {
                int nextGuess = tieBraking(bestGuess, successor.guess, solutionWords);
                if (nextGuess == successor.guess) {
                    bestGuess = successor.guess;
                }
            }
        }
        return bestGuess;
    }

    /**
     * Provides updated utility (cost) value for a tree node by eihter simulating a game or by visiting a successor acording to UCT formula.
     * @param n Node to visit
     * @param solution solution word for this rollout
     * @param solutionWords List of possible solutions
     */
    public double visitTreeNode(TreeNode n, int solution, int[] solutionWords, double bias) {
        solutionWords = filterWordList(wordle.getFeedbackInt(n.guess, solution), n.guess, solutionWords);

        if (solutionWords.length == 1) {    // terminal position
            return 1.0;
        }

        double utility = 0.0;

        TreeNode s = n.getUnvisitedSuccessor(solutionWords);

        if (s == null) {
            TreeNode next = applyTreePolicy(n, bias);
            utility = visitTreeNode(next, solution, solutionWords, bias);
        } else {
            utility = simulateGame(s.guess, solution, 0, wordle.getLegalWordAmount(), solutionWords);
            s.avgCost = utility;
            n.visitedSuccessors.add(s);
        }
        n.visitCounter++;
        n.avgCost = n.avgCost + ((utility - n.avgCost)/n.visitCounter);

        return utility;
    }

    /**
     * Provides the best successor according to UCT formula.
     */
    TreeNode applyTreePolicy(TreeNode n, double bias) {
        if (n.visitCounter == 0) return n;
        double maxUCT = Double.NEGATIVE_INFINITY;
        TreeNode bestNode = null;
        for (TreeNode successor : n.visitedSuccessors) {
            double sqrt = Math.sqrt((Math.log(n.visitCounter))/successor.visitCounter);
            double UCT = bias * sqrt - successor.avgCost;

            if (UCT > maxUCT) {
                maxUCT = UCT;
                bestNode = successor;
            }
        }

        return bestNode;
    }




    // UCT with assumed additional rollouts based on heursitic, should require less rollouts to get good policy, but takes much longer to compute.
    // Not tested or maintained.
    public int UCTOptimized(int lastGuess, int[] solutionWords, int currentGuessAmount, int nRollouts, int additionalRollouts) {
        if (solutionWords.length == 1) return wordle.solutionToGuessWord[solutionWords[0]];
        TreeNode root = new TreeNode(lastGuess);
        Random random = new Random(10);
        double bias = 1.0;

        for (int i = 0; i < nRollouts; i++) {
            int solution = solutionWords[random.nextInt(solutionWords.length)];
            System.out.println(i);
            visitTreeNodeOptimized(root, solution, solutionWords, currentGuessAmount, bias, additionalRollouts);
        }
        
        double bestCost = Double.MAX_VALUE;
        int bestGuess = -1;
        for (TreeNode successor : root.visitedSuccessors) {
            if (successor.avgCost < bestCost) {
                bestCost = successor.avgCost;
                bestGuess = successor.guess;
            }
        }

        return bestGuess;
    }

    public double visitTreeNodeOptimized(TreeNode n, int solution, int[] solutionWords, int currentGuessAmount, double bias, int additionalRollouts) {
        solutionWords = filterWordList(wordle.getFeedbackInt(n.guess, solution), n.guess, solutionWords);

        if (solutionWords.length == 1) {
            return 1.0;
        }

        double utility = 0.0;

        TreeNode s = n.getUnvisitedSuccessor(solutionWords);

        if (s == null) {
            TreeNode next = applyTreePolicyOptimized(n, bias, additionalRollouts, solution, solutionWords, currentGuessAmount);
            utility = visitTreeNodeOptimized(next, solution, solutionWords, currentGuessAmount + 1, bias, additionalRollouts);
        } else {
            utility = simulateGame(s.guess, solution, currentGuessAmount, wordle.getLegalWordAmount(), solutionWords);
            s.avgCost = utility;
            n.visitedSuccessors.add(s);
        }
        n.visitCounter++;
        n.avgCost = n.avgCost + ((utility - n.avgCost)/n.visitCounter);

        return utility;
    }

    TreeNode applyTreePolicyOptimized(TreeNode n, double bias, int additionalRollouts, int solution, int[] solutionWords, int currentGuessAmount) {
        if (n.visitCounter == 0) return n;

        double maxUCT = Double.NEGATIVE_INFINITY;
        TreeNode bestNode = null;
        for (TreeNode successor : n.visitedSuccessors) {
            int nVisits = n.visitCounter + additionalRollouts;
            int succVisits = successor.visitCounter + additionalRollouts;

            double succCost = successor.visitCounter * successor.avgCost + additionalRollouts * simulateGame(successor.guess, solution, 0, wordle.getLegalWordAmount(), solutionWords);
            succCost = succCost / (successor.visitCounter + additionalRollouts);

            double sqrt = Math.sqrt((Math.log(nVisits))/succVisits);
            double UCT = bias * sqrt - succCost;

            if (UCT > maxUCT) {
                maxUCT = UCT;
                bestNode = successor;
            }
        }

        return bestNode;
    }
}
