package wordle.java;

import java.io.*;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.*;


/**
 * Class to represent the Wordle game.
 * Handles word lists and feedback to guesses.
 * Allows for playing a game by counting guesses.
 */
public class Wordle{

    public List<String> legalList;
    public List<String> solutionList;
    private String solutionWord;
    private int solutionWordInt;
    private int guessCount = 0;
    private String[][] feedbackMatrix;
    private int[][] feedbackMatrixInt;
    private int[] guessToSolution;
    public Map<String, Integer> indexMapGuess;
    public Map<String, Integer> indexMapSolution;
    
    private int legalWordAmount;
    private int solutionWordAmount;
    public int[] solutionToGuessWord;
    //private int[] feedbackArrayKeys;
    //private String[] feedbackArray;
    //private static String legalListPath = "./src/wordle/data/legal_list.txt";
    //private static String solutionListPath = "./src/wordle/data/solution_list.txt";
    // Next two lists from: https://github.com/deedy/wordle-solver
    private static String legalListPath = "./wordle/data/official_wordle_all_sorted.txt";
    private static String solutionListPath = "./wordle/data/official_wordle_solution_sorted.txt";
    //private static String feedbackMatrixPath = "./src/wordle/data/feedbackMatrixInt.txt";
    //private String feedbackPathJson = "./src/wordle/data/guess_feedback.json";

    public long tt0 = 0;
    public long tt1 = 0;
    int feedbackLookupCount = 0;

    /**
     * String representation of words
     * Not maintained
     */
    public Wordle(String solutionWord) {
        this.legalList = readWordList(legalListPath);
        //Collections.sort(legalList);
        this.legalWordAmount = legalList.size();
        this.solutionList = readWordList(solutionListPath);
        this.solutionWordAmount = solutionList.size();
        //Collections.sort(solutionList);

        if (solutionWord == "1") {
            Random random = new Random();
            this.solutionWord = this.solutionList.get(random.nextInt(this.solutionList.size()));
        } else {
            this.solutionWord = solutionWord;
        }
        System.out.println("Compute Feedbacks");
        computeFeedbackMatrices();
        System.out.println("Feedback computation done");

    }

    /**
     * Handles integer version of words.
     * Precomputes guess feedback and initializes conversions from integer to String representation
     * aswell as from guess and solution integer encoding.
     * @param solutionWordInt Solution to the first Wordle game; -1 for random solution
     */
    public Wordle(int solutionWordInt) {
        this.legalList = readWordList(legalListPath);
        this.legalWordAmount = legalList.size();
        this.solutionList = readWordList(solutionListPath);
        this.solutionWordAmount = solutionList.size();

        if (solutionWordInt == -1) {
            Random random = new Random();
            this.solutionWordInt = random.nextInt(this.solutionWordAmount);
        } else {
            this.solutionWordInt = solutionWordInt;
        }

        System.out.println("Compute Feedbacks...");
        computeFeedbackMatrixInt();
        //this.feedbackMatrixInt = readFeedbackMatrix(feedbackMatrixPath); // used for local testing as reading from disc is faster than computing all feedbacks
        solutionToGuessWord = new int[solutionWordAmount];
        for (int solution = 0; solution < solutionWordAmount; solution++) {
            solutionToGuessWord[solution] = legalList.indexOf(solutionList.get(solution));
        }
        System.out.println("Feedbacks compute");

        this.guessToSolution = new int[legalWordAmount];
        for (int i = 0; i < guessToSolution.length; i++) {
            if (solutionList.contains(getGuessWordFromInt(i))) {
                guessToSolution[i] = solutionList.indexOf(getGuessWordFromInt(i));
            } else {
                guessToSolution[i] = -3;
            }
        }
    }

    /**
     * Handles integer version of words.
     * Initializes conversions from integer to String representation
     * Experiments require two instances of the Wordle class allowing for passing of the precomputed  guess feedback.
     * @param solutionWordInt Solution to the first Wordle game; -1 for random solution
     * @param feedbackMatrixInt Already computed guess feedback
     */
    public Wordle(int solutionWordInt, int[][] feedbackMatrixInt) {
        this.legalList = readWordList(legalListPath);
        this.legalWordAmount = legalList.size();
        this.solutionList = readWordList(solutionListPath);
        this.solutionWordAmount = solutionList.size();

        if (solutionWordInt == -1) {
            Random random = new Random();
            this.solutionWordInt = random.nextInt(this.solutionWordAmount);
        } else {
            this.solutionWordInt = solutionWordInt;
        }

        this.feedbackMatrixInt = feedbackMatrixInt;
        solutionToGuessWord = new int[solutionWordAmount];
        for (int solution = 0; solution < solutionWordAmount; solution++) {
            solutionToGuessWord[solution] = legalList.indexOf(solutionList.get(solution));
        }
        this.guessToSolution = new int[legalWordAmount];
        for (int i = 0; i < guessToSolution.length; i++) {
            if (solutionList.contains(getGuessWordFromInt(i))) {
                guessToSolution[i] = solutionList.indexOf(getGuessWordFromInt(i));
            } else {
                guessToSolution[i] = -3;
            }
        }
    }

    /**
     * Precomputes guess feedback.
     * Used for early String representation and testing.
     * Not maintained.
     */
    private void computeFeedbackMatrices() {
        int legalListSize = legalList.size();
        int solutionListSize = solutionList.size();

        feedbackMatrix = new String[legalListSize][solutionListSize];
        feedbackMatrixInt = new int[legalListSize][solutionListSize];
        indexMapGuess = new HashMap<>();
        indexMapSolution = new HashMap<>();

        for (int i = 0; i < legalListSize; i++) {
            indexMapGuess.put(legalList.get(i), i);
        }
        for (int i = 0; i < solutionListSize; i++) {
            indexMapSolution.put(solutionList.get(i), i);
        }

        for (String guess : legalList) {
            for (String solution : solutionList) {
                String feedback = computeFeedbackFromStringWords(guess, solution);
                feedbackMatrix[indexMapGuess.get(guess)][indexMapSolution.get(solution)] = feedback;
                feedbackMatrixInt[indexMapGuess.get(guess)][indexMapSolution.get(solution)] = feedbackToInt(feedback);
            }
        }

    }

    /**
     * Precomputes all guess feedback for the integer representation.
     */
    public void computeFeedbackMatrixInt() {
        feedbackMatrixInt = new int [legalWordAmount][solutionWordAmount];

        for (int guess = 0; guess < legalWordAmount; guess++) {
            for (int solution = 0; solution < solutionWordAmount; solution++) {
                String feedback = computeFeedbackFromStringWords(legalList.get(guess), solutionList.get(solution));
                feedbackMatrixInt[guess][solution] = feedbackToInt(feedback);
            }
        }
    }

    /**
     * Converts the feedback String to a integer of base3
     * @param feedback String encoded feedback
     * @return  base3 integer encoded feedback
     */
    public int feedbackToInt(String feedback) {
        int fb = 0;
        for (int i = 0; i < feedback.length(); i++) {
            char c = feedback.charAt(i);
            int value = 0;
            switch (c) {
                case 'g': value = 0; break;
                case 'y': value = 1; break;
                case 'b': value = 2; break;
                default: throw new IllegalArgumentException("Invalid character: " + c);
            }
            fb = fb * 3 + value; // Shift the base and add the new value
        }
        return fb;
    }


    /**
     * Creates a List for words from a file
     */
    private List<String> readWordList(String filePath) {
        try {
            return Files.readAllLines(Paths.get(filePath));
        } catch (IOException e) {
            System.out.println(e);
            return Collections.emptyList();
        }
    }

    /**
     * Reads precomputed feedback from a file.
     * Used for local testing.
     */
    
    private int[][] readFeedbackMatrix(String filePath) {
        List<int[]> arrayList = new ArrayList<>();

        try (BufferedReader reader = new BufferedReader(new FileReader(filePath))) {
            String line;
            while ((line = reader.readLine()) != null) {
                String[] numbers = line.split(" ");
                int[] row = new int[numbers.length];
                for (int i = 0; i < numbers.length; i++) {
                    row[i] = Integer.parseInt(numbers[i]);
                }
                arrayList.add(row);
            }
        } catch (Exception e) {

        }

        int[][] feedbackMatrixInt = new int[arrayList.size()][];
        for (int i = 0; i < arrayList.size(); i++) {
            feedbackMatrixInt[i] = arrayList.get(i);
        }
        return feedbackMatrixInt;
    } 


    /**
     * Starts a game of Wordle using String representation by setting the solution word and guess counter.
     * @param solutionWord new solution to the game
     */
    public void startGame(String solutionWord, int currenntGuessCount) {
        this.guessCount = currenntGuessCount;
        this.solutionWord = solutionWord;
    }

    /**
     * Starts a game of Wordle using integer representation by setting the solution word and guess counter.
     * @param solutionWord new solution to the game
     */
    public void startGame(int solutionWord, int currenntGuessCount) {
        this.guessCount = currenntGuessCount;
        this.solutionWordInt = solutionWord;
    }


    /**
     * Gives feedback for a guess to a game while couting guesses.
     * Allows for human game play
     */
    public String makeGuess(String guessWord) {
        if (guessCount >= 6) {
            return "-1"; //to many attemps
        }
        if (!legalList.contains(guessWord)) {
            return "-2"; //word not in legal list
        }

        guessCount++;
        return getFeedbackString(guessWord, solutionWord);
    }

    /**
     * Gives feedback for a guess to a game while couting guesses.
     * Allows for human game play
     */
    public int makeGuess(int guessWord) {
        if (guessCount >= 6) {
            return -1; //to many attemps
        }
        if (guessWord < 0 || guessWord >= legalWordAmount) {
            return -2; //word not in legal list
        }

        guessCount++;
        return getFeedbackInt(guessWord, solutionWordInt);
    }

    /**
     * Retrieves precomputed String guess feedback for String represenated words by using
     * index hashmaps.
     */
    public String getFeedbackString(String guessWord, String solutionWord) {
        int guessIndex = indexMapGuess.get(guessWord);
        int solutionIndex = indexMapSolution.get(solutionWord);
        String feedback = feedbackMatrix[guessIndex][solutionIndex];
        return feedback;
    }

    /**
     * Retrieves precomputed String guess feedback with integer represented words.
     */
    public String getFeedbackString(int guessWord, int solutionWord) {
        String feedback = feedbackMatrix[guessWord][solutionWord];
        return feedback;
    }

    /**
     * Retrieves precomputed integer feedback for String represented words by using
     * index hashmaps.
     */
    public int getFeedbackInt(String guessWord, String solutionWord) {
        int guessIndex = indexMapGuess.get(guessWord);
        int solutionIndex = indexMapSolution.get(solutionWord);
        return feedbackMatrixInt[guessIndex][solutionIndex];
    }

    /**
     * Retrieves precomputed integer feedback for integer represented words
     */
    public int getFeedbackInt(int guessWord, int solutionWord) {
        return feedbackMatrixInt[guessWord][solutionWord];
    }


    /**
     * Computes the feedback of a single guess word in String representation by checking each
     * letter for correctness.
     * @return feedback String: g for green, y for yellow, b for gey
     */
    public String computeFeedbackFromStringWords(String guessWord, String solutionWord) {
        char[] feedback = new char[guessWord.length()];
        Arrays.fill(feedback, 'b');

        char[] guessChars = guessWord.toCharArray();
        char[] solutionChars = solutionWord.toCharArray();
        // green pass
        for (int i = 0; i < guessChars.length; i++) {
            if (guessChars[i] == solutionChars[i]) {
                feedback[i] = 'g';
                solutionChars[i] = '0';
            }
        }

        // yellow pass
        for (int i = 0; i < guessChars.length; i++) {
            //if (feedback[i] == 'b' && contains(solutionChars, guessChars[i])) {
            if (feedback[i] == 'b' && indexOf(solutionChars, guessChars[i]) != -1) {
                feedback[i] = 'y';
                int index = indexOf(solutionChars, guessChars[i]);
                solutionChars[index] = '0';
            }
        }
        
        return new String(feedback);

    }

    private int indexOf(char[] arr, char ch) {
        for (int i = 0; i < arr.length; i++) {
            if (arr[i] == ch) return i;
        }
        return -1;
    }

    public List<String> getLegalList() {
        return legalList;
    }

    public List<String> getSolutionList() {
        return solutionList;
    }

    public int getLegalWordAmount() {
        return legalWordAmount;
    }

    public int getSolutionWordAmount() {
        return solutionWordAmount;
    }

    public String getGuessWordFromInt(int guess) {
        return legalList.get(guess);
    }

    public String getSolutionWordFromInt(int solution) {
        return solutionList.get(solution);
    }

    public int getGuessWordIntFromString(String guess) {
        return legalList.indexOf(guess);
    }

    public int getSolutionWordIntFromString(String solution) {
        return solutionList.indexOf(solution);
    }

    public int getSolutionIntFromGuessInt(int guess) {
        return guessToSolution[guess]; // precomputed as called frequently by tie breaking of heuristic
    }

    public int[][] getFeedbackMatrixInt() {
        return feedbackMatrixInt;
    }
}
