package wordle.java;

import java.io.BufferedWriter;
import java.io.FileWriter;
import java.io.IOException;
import java.util.List;

/**
 * Class to get the average score of a wordle algorithm.
 * Used for local testing.
 */
public class TestPerformance {
    private static Wordle wordle;
    private static WordleGuessFinder finder;

    static List<String> legalList;
    static List<String> solutionList;
    
    public static void main(String[] args) {
        testInt();
    }

    /**
     * Computes average score for integer versions of algorithms.
     * Algorithms are switched with comments in the while loop.
     */
    public static void testInt() {
        finder = new WordleGuessFinder(0);
        wordle = new Wordle(0, finder.wordle.getFeedbackMatrixInt());
        int firstGuess = wordle.getGuessWordIntFromString("salet");

        int guessWordAmount = wordle.getLegalWordAmount();
        int solutionWordAmount = wordle.getSolutionWordAmount();

        int totalGuesses = 0;
        int[] amountGuesses = new int[8];

        long t0 = System.currentTimeMillis();
        for (int solution = 0; solution < solutionWordAmount; solution++) {
            int[] solutionWords = new int[solutionWordAmount];
            for (int i = 0; i < solutionWords.length; i++) {
                solutionWords[i] = i;
            }

            wordle.startGame(solution, 0);
            int feedback = wordle.makeGuess(firstGuess);
            int guesses = 1;
            solutionWords = finder.filterWordList(feedback, firstGuess, solutionWords);
            int nextGuess = firstGuess;

            while (feedback != 0) {
                //nextGuess = finder.getHighestEntropy(guessWordAmount, solutionWords);
                //nextGuess = finder.getHighestEntropyTieBraking(guessWordAmount, solutionWords);
                //nextGuess = finder.getHighestEntropyParallel(guessWordAmount, solutionWords, 7);
                nextGuess = finder.getHighestEntropyParallelTieBraking(guessWordAmount, solutionWords, 7);
                //nextGuess = finder.findNextGuessRollout(guessWordAmount, solutionWords, guesses, 10, true);
                //nextGuess = finder.UCT(nextGuess, solutionWords, guesses, 10, 4000, true);


                //System.out.println("next guess: " + nextGuess);

                feedback = wordle.makeGuess(nextGuess);
                guesses++;
                if (feedback == -1) {
                    System.out.println("Failed on solution: " + solution);
                    break;
                }

                if (feedback == -2) {
                    System.out.println("Guess" + nextGuess + ", Feedback: " + feedback + ", solution: " + solution);
                }

                if (feedback == 0) break;

                solutionWords = finder.filterWordList(feedback, nextGuess, solutionWords);
                //System.out.println("New solutionWords size: " + solutionWords.length);
            }
            totalGuesses += guesses;
            amountGuesses[guesses] += 1;
            
            
            if (solution % 10 == 0) {
                double timeTakenSoFar = (System.currentTimeMillis() - t0) / 1000.0;
                double ittPerSec = solution / timeTakenSoFar;
                System.out.print("\rSolutionword " + solution + " done.   Iterations/s: " + ittPerSec);
            }
        }
        System.out.println("");
        long t1 = System.currentTimeMillis();
        //double timeTotalEntropy = (finder.feedbackTime / 1000000.0) + (finder.entropyTime / 1000000.0);
        System.out.println("\n\nFirst guess: " + wordle.getGuessWordFromInt(firstGuess));
        System.out.println("Average score: " + (double) totalGuesses/solutionWordAmount);
        System.out.println("Time taken: " + ((t1-t0) / 1000.0) + "s" + ", avg time: " + (double)(t1-t0)  / solutionWordAmount + "ms");
        //System.out.println("Time spend retriving feedback: " + finder.feedbackTime / 1000000.0 + "ms  " + (finder.feedbackTime / 1000000.0) / timeTotalEntropy * 100 
          //  + "%" +", Time spend computing entropy: " + finder.entropyTime / 1000000.0 + "ms " + (finder.entropyTime / 1000000.0) / timeTotalEntropy * 100 + "%");
        StringBuilder s = new StringBuilder();
        for (int i = 1; i < amountGuesses.length; i++) {
            s.append(i).append(": ").append(amountGuesses[i]).append("\n");
        }
        System.out.println(s.toString());
        /* 
        try (BufferedWriter writer = new BufferedWriter(new FileWriter("./src/wordle/java/Test.txt", true))) {
            writer.write("Results of using Integer representation\n");
            writer.write("First guess: " + firstGuess + " (" + wordle.getGuessWordFromInt(firstGuess) + ")" + '\n');
            writer.write("Average guesses: " + ((double)totalGuesses / solutionWordAmount) + "\n");
            writer.write("Frequencies of results: \n");
            writer.write(s.toString());
            writer.write("Time taken: " + ((t1-t0) / 1000.0) + "s" + ", Average time taken: " + (double)(t1-t0)  / solutionWordAmount + "ms" + "\n\n\n");
            //writer.write("Average time taken: " + ((t1 - t0) / solutionList.size()) + "ms" + "\n\n\n");
            //writer.write("Used precomputed feedback: " + usePrecomputedFeedback + '\n');
        } catch (IOException e) {

        } */
    }


    /**
     * Tests the string version of the heuristic.
     * Not maintained
     */
    public static void testString() {
        wordle = new Wordle("audio");
        finder = new WordleGuessFinder("audio");

        String firstGuess = "soare";

        legalList = wordle.getLegalList();
        solutionList = wordle.getSolutionList();
        //solutionList = List.of("force", "forge", "audio", "basic", "aback", "abase", "abate");

        List<String> currentSolutionList;
        for (int size = 0; size <= 0; size++) {

            int[] amountGuesses = new int[8];
            int totalGuesses = 0;
            //int count_itterations = 0;

            long t0 = System.currentTimeMillis();

            for (String solution : solutionList) {
                //count_itterations++;
                int guessCount = 0;
                int c = 1;
                String feedback = "";
                currentSolutionList = solutionList;
                wordle.startGame(solution, 0);

                while (!feedback.equals("ggggg")) {
                    String nextGuess = "";
                    if (feedback.equals("-1")) {
                        guessCount = 7;
                        System.out.println("Faield on: " + solution);
                        break;
                    }
        
                    if (c == 1) {
                        nextGuess = firstGuess;
                        c = 2;
                    } else {
                        //nextGuess = finder.findNextGuessRolloutString(legalList, currentSolutionList, guessCount, 4);
                        nextGuess = finder.getHighestEntropyParallelString(legalList, currentSolutionList, 8);
                    }
                    //System.out.println("next guess: " + nextGuess);
                    feedback = wordle.makeGuess(nextGuess);
                    guessCount++;

                    if (feedback.equals("ggggg")) {
                        break;
                    }
        
                    currentSolutionList = finder.filterWordList(feedback, nextGuess, currentSolutionList);
                    //System.out.println("Size solution list: " + currentSolutionList.size());
                }
                amountGuesses[guessCount]++;
                totalGuesses += guessCount;
                
                /*
                if (count_itterations % 10 == 0) {
                    long t2 = System.currentTimeMillis();
                    double currTime = (double) (t2 - t0) / 1000;
                    System.out.println("Solution word: " + solution + "   Guesses: " + guessCount + "   Itteration: "  + count_itterations + " done   " + "time: " + currTime + "s    "+ "itterations/s: " + count_itterations/currTime + "   " + ((double)count_itterations/(double)solutionList.size()) + "%");
                    //System.out.println("countItterations: " + count_itterations + "   solution list size: " + solutionList.size());
                } */
            }

            long t1 = System.currentTimeMillis();
            double time = (t1 - t0) / 1000.0;

            System.out.println("\n\nFinished!");
            System.out.println("First guess: " + firstGuess);
            System.out.println("Cutoff: " + size);
            System.out.println("Time taken: " + ((t1-t0) / 1000.0) + "s" + ", avg time: " + (double)(t1-t0)  / solutionList.size() + "ms");
            System.out.println("Avg number of guesses: " + ((double)totalGuesses / (double)solutionList.size()));

            StringBuilder s = new StringBuilder();
            for (int i = 1; i < amountGuesses.length; i++) {
                s.append(i).append(": ").append(amountGuesses[i]).append("\n");
            }
            System.out.println(s.toString());

            try (BufferedWriter writer = new BufferedWriter(new FileWriter("./src/wordle/java/Test.txt", true))) {
                writer.write("Results of using String representation\n");
                writer.write("Cutoff at solution list size: " + size + "\n");
                writer.write("First guess: " + firstGuess + '\n');
                writer.write("Average guesses: " + ((double)totalGuesses / (double)solutionList.size()) + "\n");
                writer.write("Frequencies of results: \n");
                writer.write(s.toString());
                writer.write("Time taken: " + time + "s" + ", Average time taken: " + ((double)(t1 - t0) / solutionList.size()) + "ms" +"\n\n\n");
                //writer.write("Average time taken: " + ((t1 - t0) / solutionList.size()) + "ms" + "\n\n\n");
                //writer.write("Used precomputed feedback: " + usePrecomputedFeedback + '\n');
            } catch (IOException e) {

            }
        }
    }
}
