package wordle.java;

/**
 * Class used to perfom experiments on the scicore cluster.
 * Experiments compute the averge score and runtime of a specified algorithm across all possible solution words with a fixed opening guss
 * Arguments control which algorithm is used:
 * run with: java Experiment int:wordIndex String:algorithm int:nBest int:nRollouts int:nTasksForUCT boolean:optimize
 * wordIndex: opening guess according to fistGuess list (slurm task id)
 * algorithm: "heurisitc", "rollout", "uct"
 * nBest: number of most promissing guesses to be considered for rollout and uct
 * nRollouts: number of rollouts for uct
 * nTasksForUCT: split up run on cluster because uct's long runtime
 * optimize: apply optimizations for rollout and uct, experiments were only run with optimized uct
 * 
 * Time constraints did not allow to add a parameter to switch the heursitic version. This was done by
 * adapting the simulateGame and getHighestEntropyList method in the WordleGuessFinder class as well as the
 * heurstic verison in the testEntropyHeuristic method in the Experiment class.
 * Current implementation uses advanced tie braking for the heuristic.
 * 
 * Time constraints further did not allow to use the optimization for the rollout algorithm.
 * 
 * Experiments with MIGA where carried out on implementarions from the git commit "dbb7f6c"
 * Experiments with MIGB where carried out on implementations from the git commit "c7614ef"
 */
public class Experiment {
    private static Wordle wordle;
    private static WordleGuessFinder finder;


    /**
     * Starts experiment according to command line arguments
     */
    public static void main (String[] args) {
        finder = new WordleGuessFinder(0);
        wordle = new Wordle(0, finder.wordle.getFeedbackMatrixInt());

        String[] firstGuesses = {"salet", "reast", "crate", "trace", "slate", "trape", "slane", "prate", "crane", "carle", "train", "raise", "clout", "soare", "salut", "lands"};

        int firstGuess = wordle.getGuessWordIntFromString(firstGuesses[Integer.valueOf(args[0])]);
        for (int i = 0; i < args.length; i++) {
            System.out.print(args[i] + " ");
        }
        System.out.println();

        switch (args[1]) {
            case "heuristic":
                testEntropyHeuristic(finder, wordle, firstGuess);
                break;
            case "rollout":
                testRollout(finder, wordle, firstGuess, Integer.valueOf(args[2]), Boolean.parseBoolean(args[3]));
                break;
            case "uct": // tests only for "salet" as first word. Split test over multiple array tasks to speed up test.
                int taskCount = Integer.valueOf(args[4]);
                int chunkSize = (int) Math.ceil((double) wordle.getSolutionWordAmount() / taskCount);
                int startIndex = Integer.valueOf(args[0]) * chunkSize;
                int endIndex = Math.min(startIndex + chunkSize, wordle.getSolutionWordAmount());

                testUCT(finder, wordle, wordle.getGuessWordIntFromString("salet"), Integer.parseInt(args[2]), Integer.parseInt(args[3]), startIndex, endIndex, Boolean.parseBoolean(args[5]));
                break;
        }
    }

    /**
     * Computes average score for the parallelized version of the heurstic for a fixed opening guess.
     */
    public static void testEntropyHeuristic(WordleGuessFinder finder, Wordle wordle, int firstGuess) {
        int guessWordAmount = wordle.getLegalWordAmount();
        int solutionWordAmount = wordle.getSolutionWordAmount();

        int nextGuess = firstGuess;
        int totalGuesses = 0;
        int[] amountGuesses = new int[8];

        long t0 = System.currentTimeMillis();
        for (int solution = 0; solution < solutionWordAmount; solution++) {
            nextGuess = firstGuess;
            int[] solutionWords = new int[solutionWordAmount];
            for (int i = 0; i < solutionWords.length; i++) {
                solutionWords[i] = i;
            }

            int feedback = wordle.getFeedbackInt(nextGuess, solution);
            int guesses = 1;
            solutionWords = finder.filterWordList(feedback, nextGuess, solutionWords);


            while (feedback != 0) {
                if (guesses > 6) break;
                nextGuess = finder.getHighestEntropyParallelTieBraking(guessWordAmount, solutionWords, 8);
                feedback = wordle.getFeedbackInt(nextGuess, solution);
                guesses++;
                if (feedback == 0) break;
                solutionWords = finder.filterWordList(feedback, nextGuess, solutionWords);
            }
            totalGuesses += guesses;
            amountGuesses[guesses] += 1;
        }
        long t1 = System.currentTimeMillis();
        printResult(firstGuess, totalGuesses, solutionWordAmount, amountGuesses, t0, t1, "Heursitic only");
    }

    /**
     * Computes average score for the rollout algorithm for a fixed opening guess and specified
     * amount of most promissing guesses to consider.
     */
    public static void testRollout(WordleGuessFinder finder, Wordle wordle, int firstGuess, int nBest, boolean useSolutionWordsAtEndGame) {
        int guessWordAmount = wordle.getLegalWordAmount();
        int solutionWordAmount = wordle.getSolutionWordAmount();

        int nextGuess = firstGuess;
        int totalGuesses = 0;
        int[] amountGuesses = new int[8];

        long t0 = System.currentTimeMillis();
        for (int solution = 0; solution < solutionWordAmount; solution++) {
            nextGuess = firstGuess;
            int[] solutionWords = new int[solutionWordAmount];
            for (int i = 0; i < solutionWords.length; i++) {
                solutionWords[i] = i;
            }

            int feedback = wordle.getFeedbackInt(nextGuess, solution);
            int guesses = 1;
            solutionWords = finder.filterWordList(feedback, nextGuess, solutionWords);


            while (feedback != 0) {
                if (guesses > 6) break;
                nextGuess = finder.findNextGuessRollout(guessWordAmount, solutionWords, guesses, nBest, useSolutionWordsAtEndGame);
                feedback = wordle.getFeedbackInt(nextGuess, solution);
                guesses++;
                if (feedback == 0) break;
                solutionWords = finder.filterWordList(feedback, nextGuess, solutionWords);
            }
            totalGuesses += guesses;
            amountGuesses[guesses] += 1;
        }
        long t1 = System.currentTimeMillis();
        printResult(firstGuess, totalGuesses, solutionWordAmount, amountGuesses, t0, t1, "Standart rollout with nBest = " + nBest);
    }

    /**
     * Computes average score for the uct algorithm for a fixed opening guess and specified
     * amount of most promising guesses and rollouts for parts of the solution list.
     */
    public static void testUCT(WordleGuessFinder finder, Wordle wordle, int firstGuess, int nBest, int nRollouts, int startIndex, int endIndex, boolean useSolutionWordsAtEndGame) {
        int solutionWordAmount = wordle.getSolutionWordAmount();

        int nextGuess = firstGuess;
        int totalGuesses = 0;
        int[] amountGuesses = new int[8];

        long t0 = System.currentTimeMillis();
        for (int solution = startIndex; solution < endIndex; solution++) {
            nextGuess = firstGuess;
            int[] solutionWords = new int[solutionWordAmount];
            for (int i = 0; i < solutionWords.length; i++) {
                solutionWords[i] = i;
            }

            int feedback = wordle.getFeedbackInt(nextGuess, solution);
            int guesses = 1;
            solutionWords = finder.filterWordList(feedback, nextGuess, solutionWords);


            while (feedback != 0) {
                if (guesses > 6) break;
                nextGuess = finder.UCT(nextGuess, solutionWords, guesses, nBest, nRollouts, useSolutionWordsAtEndGame);

                feedback = wordle.getFeedbackInt(nextGuess, solution);
                guesses++;
                if (feedback == 0) break;
                solutionWords = finder.filterWordList(feedback, nextGuess, solutionWords);
            }
            totalGuesses += guesses;
            amountGuesses[guesses] += 1;
        }
        long t1 = System.currentTimeMillis();
        printResult(firstGuess, totalGuesses, endIndex - startIndex, amountGuesses, t0, t1, "UCT with nBest = " + nBest + " and nRollouts = " + nRollouts + "; Solutions " + startIndex + " to " + (endIndex - 1));
    }


    /**
     * Formates the results of a experiment and prints them to the
     * command line. 
     */
    static void printResult(int firstGuess, int totalGuesses, int solutionWordAmount, int[] amountGuesses, long t0, long t1, String algorithm) {
        System.out.println("");
        System.out.println("\nAlgorithm: " + algorithm + "\nFirst guess: " + wordle.getGuessWordFromInt(firstGuess));
        System.out.println("Average score: " + (double) totalGuesses/solutionWordAmount);
        System.out.println("Time taken: " + ((t1-t0) / 1000.0) + "s" + ", avg time: " + (double)(t1-t0)  / solutionWordAmount + "ms");
        StringBuilder s = new StringBuilder();
        for (int i = 1; i < amountGuesses.length; i++) {
            s.append(i).append(": ").append(amountGuesses[i]).append("\n");
        }
        System.out.println(s.toString());
    }
}
