from ortools.linear_solver import pywraplp

import main


def solveDistribution(itemName, items, isLin, isCon):
    if isCon:
        if not checkItemHasCon(itemName, items): return None, None, None

    itemList = prepareItemList(itemName, items)
    if isLin:
        solver = pywraplp.Solver.CreateSolver('GLOP')
        variablesMap, variablesPerSecNameMap, solver = generateLinVariables(solver, itemList)
    else:
        solver = pywraplp.Solver.CreateSolver('SCIP')
        variablesMap, variablesPerSecNameMap, solver = generateIntVariables(solver, itemList)

    solver = solveEquation(solver, itemList, variablesPerSecNameMap, variablesMap, isCon, isLin)

    if solver.Solve() == pywraplp.Solver.OPTIMAL:
        return solver.Objective().Value(), extractValueFromVariable(variablesMap), solver.wall_time()
    else:
        print('The problem does not have an optimal solution. {' + itemName + '} \n')
        return None, None, None


def prepareItemList(itemName, items):
    itemList = eliminateDuplicates(getItemList(itemName, items))
    itemList = eliminateDuplicates(replaceFuel(itemList, items))
    return itemList


def generateLinVariables(solver, itemList):
    variablesMap = []
    variablesPerSecNameMap = []
    for i, item in enumerate(itemList):
        varName = " ".join(str(x) for x in item[0])
        variablesMap.append([item[0], solver.NumVar(0, solver.infinity(), varName)])
        variablesPerSecNameMap.append([item[0], solver.NumVar(0, solver.infinity(), varName + "PerSec")])
    return variablesMap, variablesPerSecNameMap, solver


def generateIntVariables(solver, itemList):
    variablesMap = []
    variablesPerSecNameMap = []
    for i, item in enumerate(itemList):
        varName = " ".join(str(x) for x in item[0])
        variablesMap.append([item[0], solver.IntVar(0, solver.infinity(), varName)])
        variablesPerSecNameMap.append([item[0], solver.NumVar(0, solver.infinity(), varName + "PerSec")])
    return variablesMap, variablesPerSecNameMap, solver


def solveEquation(solver, itemList, variablesPerSecNameMap, variablesMap, isCon, isLin):
    solver = createConstraints(solver, itemList, variablesPerSecNameMap, variablesMap, isCon, isLin)

    solver.Maximize(variablesPerSecNameMap[0][1])

    return solver


def createConstraints(solver, itemList, variablesPerSecNameMap, variablesMap, isCon, isLin):
    if isLin:
        solver = calculateTimeLinearConstraints(solver, itemList, variablesPerSecNameMap, variablesMap)
    else:
        solver = calculateTimeBoundedConstraints(solver, itemList, variablesPerSecNameMap, variablesMap)

    solver = calculateWorkerBoundConstraint(solver, variablesMap)

    for elm in itemList:
        if elm[1].recipe:
            if isCon:
                solver = calculateRecipeLinearConstraints(solver, elm[0], itemList, variablesPerSecNameMap)
            else:
                solver = calculateRecipeBoundConstraints(solver, elm[0], itemList, variablesPerSecNameMap)

    if isCon:
        solver.Add(
            getVar(itemList[0][0], variablesPerSecNameMap) <= main.nrOfBuildings / itemList[0][1].consumptionTime)

    return solver


def calculateTimeBoundedConstraints(solver, itemList, variablesPerSecNameMap, variablesMap):
    for i, item in enumerate(itemList):
        solver.Add(variablesPerSecNameMap[i][1] <= variablesMap[i][1] * item[1].unitsPerSecond)
    return solver


def calculateTimeLinearConstraints(solver, itemList, variablesPerSecNameMap, variablesMap):
    for i, item in enumerate(itemList):
        solver.Add(variablesPerSecNameMap[i][1] == variablesMap[i][1] * item[1].unitsPerSecond)
    return solver


def calculateWorkerBoundConstraint(solver, variablesMap):
    constraint = solver.Constraint(0, main.worker)
    for variable in variablesMap:
        if variable[0][-1] != "Water":
            constraint.SetCoefficient(variable[1], 1)
    return solver


def calculateRecipeBoundConstraints(solver, itemName, itemList, variablesPerSecNameMap):
    item = getItem(itemName[-1], [x[1] for x in itemList])
    itemVar = getVar(itemName, variablesPerSecNameMap)
    nrOfSameItems = distributionOfItems(item.recipe)

    for i, item in enumerate(eliminateDuplicates(item.recipe)):
        if item == "Fuel":
            key = itemName.copy()
            key.append(main.fuelType)
            solver.Add(
                itemVar <= insertFuelPropellantCoefficient() * getVar(key,
                                                                      variablesPerSecNameMap) /
                nrOfSameItems[
                    i])
        else:
            key = itemName.copy()
            key.append(item)
            solver.Add(itemVar <= getVar(key, variablesPerSecNameMap) / nrOfSameItems[i])

    return solver


def calculateRecipeLinearConstraints(solver, itemName, itemList, variablesPerSecNameMap):
    item = getItem(itemName[-1], [x[1] for x in itemList])
    itemVar = getVar(itemName, variablesPerSecNameMap)
    nrOfSameItems = distributionOfItems(item.recipe)

    for i, item in enumerate(eliminateDuplicates(item.recipe)):
        if item == "Fuel":
            key = itemName.copy()
            key.append(main.fuelType)
            solver.Add(
                itemVar == insertFuelPropellantCoefficient() * getVar(key,
                                                                      variablesPerSecNameMap) /
                nrOfSameItems[
                    i])
        else:
            key = itemName.copy()
            key.append(item)
            solver.Add(itemVar == getVar(key, variablesPerSecNameMap) / nrOfSameItems[i])

    return solver


def calculateCoinRanking(items, isLin, isCon):
    listOfLists = assigningToColorList(items, isLin, isCon)
    for index, list in enumerate(listOfLists):
        listOfLists[index] = sorted(list, key=getKey, reverse=True)
    return listOfLists


def assigningToColorList(items, isLin, isCon):
    rankingYellowList = []
    rankingRedList = []
    rankingBlueList = []
    rankingPurpleList = []

    for item in items:
        list = getColorList(item.coinType, rankingYellowList, rankingRedList, rankingBlueList, rankingPurpleList)
        if list is not None:
            objectValue, variableList, _ = solveDistribution(item.name, items, isLin, isCon)
            amountOfWorker = countWorker(variableList)

            if amountOfWorker > 0:
                coinPerWorker = objectValue * item.coinPerUnit / amountOfWorker
            else:
                coinPerWorker = 0

            list.append((item.name, objectValue, item.coinPerUnit, objectValue * item.coinPerUnit, amountOfWorker,
                         coinPerWorker))
    return [rankingYellowList, rankingRedList, rankingBlueList, rankingPurpleList]


def calculateMaxCoinDistribution(items, color, isLin):
    if isLin:
        solver = pywraplp.Solver.CreateSolver('GLOP')
    else:
        solver = pywraplp.Solver.CreateSolver('SCIP')

    itemList = prepareMaxCoinItemList(items, color, isLin)

    variableCoin, solver = generateMaxCoinVariable(itemList, solver, isLin)

    solver = createMaxCoinWorkerConstrains(solver, itemList, variableCoin)

    if solver.Solve() == pywraplp.Solver.OPTIMAL:
        return solver.Objective().Value(), sumOfUsedWorker(itemList, variableCoin), extractInterestingNumbers(itemList,
                                                                                                              variableCoin), solver.wall_time()
    else:
        print('The problem does not have an optimal solution. {' + color + '} \n')
        return None, None, None, None


def prepareMaxCoinItemList(items, color, isLin):
    itemList = []
    for item in items:
        if getColorTyp(item.coinType) == color:
            if isLin:
                objectValue, variableList, _ = solveDistribution(item.name, items, True, True)
            else:
                objectValue, variableList, _ = solveDistribution(item.name, items, False, True)
            amountOfWorker = countWorker(variableList)
            list = [item.name, objectValue * item.coinPerUnit, amountOfWorker]
            if list not in itemList:
                itemList.append(list)
    return itemList


def generateMaxCoinVariable(itemList, solver, isLin):
    objective = solver.Objective()
    variableCoin = []
    for i, item in enumerate(itemList):
        if isLin:
            variableCoin.append([item[0], solver.NumVar(0, 1, item[0])])
        else:
            variableCoin.append([item[0], solver.IntVar(0, 1, item[0])])
        objective.SetCoefficient(variableCoin[i][1], item[1])
    objective.SetMaximization()
    return variableCoin, solver


def createMaxCoinWorkerConstrains(solver, itemList, variableCoin):
    constraints = solver.Constraint(0, main.worker)
    for index, item in enumerate(itemList):
        constraints.SetCoefficient(variableCoin[index][1], item[2])
    return solver


def calculateBalanceCoins(items, balance, isLin):
    if isLin:
        solver = pywraplp.Solver.CreateSolver('GLOP')
    else:
        solver = pywraplp.Solver.CreateSolver('SCIP')

    solver, variableCoins, variableColorCoins = generateCoinColorVariablesList(items, balance, solver,
                                                                               isLin)
    solver = createBalanceConstraints(solver, variableCoins, variableColorCoins, balance)

    solver = createObjectiveFunction(solver, variableCoins)

    if solver.Solve() == pywraplp.Solver.OPTIMAL:
        return createInformationList(solver, variableCoins, variableColorCoins)
    else:
        print('The problem does not have an optimal solution. Balance: {' + balance + '} \n')
        return None


def generateCoinColorVariablesList(items, balance, solver, isLin):
    if isLin:
        listOfLists = assigningToColorList(items, True, True)
    else:
        listOfLists = assigningToColorList(items, False, True)

    variableColorCoins = []
    variableCoins = []
    for index, list in enumerate(listOfLists):
        if balance[index] > 0:
            variableCoin, solver = generateColorVariables(list, solver, isLin)
            variableColorCoins.append(variableCoin)
            if isLin:
                variableCoins.append(solver.NumVar(0, solver.infinity(),
                                                   getColorTyp(getItem(list[0][0], items).coinType)))
            else:
                variableCoins.append(solver.IntVar(0, solver.infinity(),
                                                   getColorTyp(getItem(list[0][0], items).coinType)))

    return solver, variableCoins, variableColorCoins


def generateColorVariables(list, solver, isLin):
    variableCoin = []
    for i, listEntry in enumerate(list):
        if isLin:
            var = [listEntry[0], listEntry[3], listEntry[4], solver.NumVar(0, 1, listEntry[0])]
        else:
            var = [listEntry[0], listEntry[3], listEntry[4], solver.IntVar(0, 1, listEntry[0])]

        if var is not variableCoin:
            variableCoin.append(var)
    return variableCoin, solver


def createBalanceConstraints(solver, variableCoins, variableColorCoins, balance):
    solver = createBalanceWorkerConstrains(solver, variableColorCoins)

    solver = createCoinBalanceConstrains(solver, variableCoins, variableColorCoins)

    solver = createBalanceConstrains(solver, variableCoins, balance)

    for var in variableCoins:
        solver.Add(var >= 1)

    return solver


def createObjectiveFunction(solver, variableCoins):
    objective = solver.Objective()
    for index, var in enumerate(variableCoins):
        objective.SetCoefficient(var, 1)
    objective.SetMaximization()
    return solver


def createBalanceWorkerConstrains(solver, variableColorCoins):
    constraints = solver.Constraint(0, main.worker)
    for list in variableColorCoins:
        for var in list:
            constraints.SetCoefficient(var[3], var[2])
    return solver


def createCoinBalanceConstrains(solver, variableCoins, variableColorCoins):
    for i, list in enumerate(variableColorCoins):
        constraints = solver.Constraint(0, solver.infinity())
        for var in list:
            constraints.SetCoefficient(var[3], var[1])
        constraints.SetCoefficient(variableCoins[i], -1)
    return solver


def createBalanceConstrains(solver, variableCoins, balance):
    balanceList = [value for value in balance if value != 0]
    varCoin = variableCoins.copy()

    while varCoin.__len__() > 0:
        indexMax = balanceList.index(max(balanceList))
        for index, var in enumerate(varCoin):
            if var is not varCoin[indexMax]:
                solver.Add(varCoin[indexMax] * balanceList[index] == var * balanceList[indexMax])
        del varCoin[indexMax]
        del balanceList[indexMax]

    return solver


def distributionOfItems(recipe):
    if len(recipe) == 0:
        return
    counter = 1
    nrOfSameItems = []
    for index, item in enumerate(recipe):
        if index < len(recipe) - 1:
            if item != recipe[index + 1]:
                nrOfSameItems.append(counter)
                counter = 1
            else:
                counter += 1
    nrOfSameItems.append(counter)
    return nrOfSameItems


def replaceFuel(itemList, items):
    for index, item in enumerate(itemList):
        if item[1].name == "Fuel":
            itemList[index][1] = getItem(main.fuelType, items)
            itemList[index][1].recipe = []
            itemList[index][0][-1] = main.fuelType
    return itemList


def insertFuelPropellantCoefficient():
    switcher = {
        "Fertilizer": 1,
        "Wood": 2,
        "Coal": 4,
        "Magma": 8
    }
    return switcher.get(main.fuelType)


def extractValueFromVariable(variablesMap):
    list = []
    for var in variablesMap:
        list.append([var[0], var[1].solution_value()])
    return list


def extractInterestingNumbers(itemList, variableCoin):
    list = []
    for var in variableCoin:
        if var[1].solution_value() > 0:
            for item in itemList:
                if item[0] == var[0]:
                    item[1] = item[1] * var[1].solution_value()
                    item[2] = item[2] * var[1].solution_value()
                    item.append(var[1].solution_value())
                    list.append(item)
    return list


def createInformationList(solver, variableCoins, variableColorCoins):
    listOfInfos = [solver.Objective().Value(), [var.solution_value() for var in variableCoins]]
    nrOfWorkerTotal = 0
    varList = []
    for varColorList in variableColorCoins:
        colorVarList = []
        nrOfWorker = 0
        for var in varColorList:
            if var[3].solution_value() > 0:
                nrOfWorker += var[2] * var[3].solution_value()
                var[1] = var[1] * var[3].solution_value()
                var[2] = var[2] * var[3].solution_value()
                var[3] = var[3].solution_value()
                if var not in colorVarList:
                    colorVarList.append(var)
        nrOfWorkerTotal += nrOfWorker
        varList.append([colorVarList, nrOfWorker])
    listOfInfos.append(varList)
    listOfInfos.append(nrOfWorkerTotal)
    listOfInfos.append(solver.wall_time())
    return listOfInfos


def sumOfUsedWorker(itemList, variableCoin):
    sumWorker = 0
    for var in variableCoin:
        if var[1].solution_value() > 0:
            for item in itemList:
                if item[0] == var[0]:
                    sumWorker += item[2] * var[1].solution_value()
                    break
    return sumWorker


def eliminateDuplicates(itemList):
    resultantList = []
    for element in itemList:
        if element not in resultantList:
            resultantList.append(element)
    return resultantList


def getItemList(itemName, items):
    itemList = []
    path = [itemName]
    item = getItem(itemName, items)
    itemList.append([path.copy(), item])
    for item in item.recipe:
        itemList += getItemListRec(item, items, path)
    return itemList


def getItemListRec(itemName, items, path):
    itemList = []
    currentPath = path.copy()
    currentPath.append(itemName)
    item = getItem(itemName, items)
    itemList.append([currentPath, item])
    for item in item.recipe:
        itemList += getItemListRec(item, items, currentPath)
    return itemList


def getItem(itemName, items):
    for item in items:
        if item.name == itemName:
            return item


def getKey(item):
    return item[5]


def getVar(itemName, variablesPerSecNameMap):
    for var in variablesPerSecNameMap:
        if var[0] == itemName:
            return var[1]


def getColorList(coinType, rankingYellowList, rankingRedList, rankingBlueList, rankingPurpleList):
    switcher = {
        "FFFFC000": rankingYellowList,
        "FFC00000": rankingRedList,
        "FF0070C0": rankingBlueList,
        "FF7030A0": rankingPurpleList
    }
    return switcher.get(coinType)


def getColorTyp(coinType):
    switcher = {
        "FFFFC000": "yellow",
        "FFC00000": "red",
        "FF0070C0": "blue",
        "FF7030A0": "purple"
    }
    return switcher.get(coinType)


def checkItemHasCon(itemName, items):
    item = getItem(itemName, items)
    if item.consumptionTime is None:
        return False
    return True


def countWorker(variableList):
    sumWorker = 0
    for item in variableList:
        if item[0][-1] != "Water":
            sumWorker += item[1]
    return sumWorker

# IntSolver:
# solver = pywraplp.Solver.CreateSolver('SCIP')
# solver = pywraplp.Solver.CreateSolver('CBC')
# solver = pywraplp.Solver.CreateSolver('BOP')
# solver = pywraplp.Solver.CreateSolver('SAT')

# LinSolver:
# solver = pywraplp.Solver.CreateSolver('CLP')
# solver = pywraplp.Solver.CreateSolver('GLOP')
# solver = pywraplp.Solver.CreateSolver('GLPK_LP')
