#! /usr/bin/env python

"""Solve some tasks with A* and the LM-Cut heuristic."""

from __future__ import division
import os
import os.path
import platform
import re
import suites
import matplotlib as mpl

from lab.environments import BaselSlurmEnvironment

from downward.experiment import FastDownwardExperiment
from downward.reports.absolute import AbsoluteReport
from downward.reports.scatter import ScatterPlotReport

#print mpl.rcParams.keys()


SUITE = suites.suite_optimal_strips()
ENV = BaselSlurmEnvironment()
REPO = "/infai/oldphi00/master_code"
BENCHMARKS_DIR = os.environ["DOWNWARD_BENCHMARKS"]

exp = FastDownwardExperiment(environment=ENV)
exp.add_suite(BENCHMARKS_DIR, SUITE)

time_limit = "30m"

# Without anything
exp.add_algorithm(
    '1-ianot', REPO, 'default', ['--search', 'astar(time(ia=true,tv=false,rf=false,ip=false,rd=0,transform=adapt_costs(cost_type=ONE)),cost_type=ONE)'], driver_options=["--search-time-limit", time_limit, '--build', 'release64'], build_options=["release64"])

# IP
exp.add_algorithm(
    '1-iaip', REPO, 'default', ['--search', 'astar(time(ia=true,tv=false,rf=false,ip=true,rd=0,transform=adapt_costs(cost_type=ONE)),cost_type=ONE)'], driver_options=["--search-time-limit", time_limit, '--build', 'release64'], build_options=["release64"])

# TV
exp.add_algorithm(
    '1-iatv', REPO, 'default', ['--search', 'astar(time(ia=true,tv=true,rf=false,ip=false,rd=0,transform=adapt_costs(cost_type=ONE)),cost_type=ONE)'], driver_options=["--search-time-limit", time_limit, '--build', 'release64'], build_options=["release64"])

# RD
exp.add_algorithm(
    '1-iard', REPO, 'default', ['--search', 'astar(time(ia=true,tv=false,rf=false,ip=false,rd=1,transform=adapt_costs(cost_type=ONE)),cost_type=ONE)'], driver_options=["--search-time-limit", time_limit, '--build', 'release64'], build_options=["release64"])

# RF
exp.add_algorithm(
    '1-iarf', REPO, 'default', ['--search', 'astar(time(ia=true,tv=false,rf=true,ip=false,rd=0,transform=adapt_costs(cost_type=ONE)),cost_type=ONE)'], driver_options=["--search-time-limit", time_limit, '--build', 'release64'], build_options=["release64"])
    
# WITH LI
# Without anything
exp.add_algorithm(
    'li-ianot', REPO, 'default', ['--search', 'astar(time(ia=true,tv=false,rf=false,ip=false,rd=0,li=true,transform=adapt_costs(cost_type=ONE)),cost_type=ONE)'], driver_options=["--search-time-limit", time_limit, '--build', 'release64'], build_options=["release64"])

# IP
exp.add_algorithm(
    'li-iaip', REPO, 'default', ['--search', 'astar(time(ia=true,tv=false,rf=false,ip=true,rd=0,li=true,transform=adapt_costs(cost_type=ONE)),cost_type=ONE)'], driver_options=["--search-time-limit", time_limit, '--build', 'release64'], build_options=["release64"])

# TV
exp.add_algorithm(
    'li-iatv', REPO, 'default', ['--search', 'astar(time(ia=true,tv=true,rf=false,ip=false,rd=0,li=true,transform=adapt_costs(cost_type=ONE)),cost_type=ONE)'], driver_options=["--search-time-limit", time_limit, '--build', 'release64'], build_options=["release64"])

# RD
exp.add_algorithm(
    'li-iard', REPO, 'default', ['--search', 'astar(time(ia=true,tv=false,rf=false,ip=false,rd=1,li=true,transform=adapt_costs(cost_type=ONE)),cost_type=ONE)'], driver_options=["--search-time-limit", time_limit, '--build', 'release64'], build_options=["release64"])

# RF
exp.add_algorithm(
    'li-iarf', REPO, 'default', ['--search', 'astar(time(ia=true,tv=false,rf=true,ip=false,rd=0,li=true,transform=adapt_costs(cost_type=ONE)),cost_type=ONE)'], driver_options=["--search-time-limit", time_limit, '--build', 'release64'], build_options=["release64"])

# LM-Cut
exp.add_algorithm(
    '2-lmcut', REPO, 'default', ['--search', 'astar(lmcut(transform=adapt_costs(cost_type=ONE)),cost_type=ONE)'], driver_options=["--search-time-limit", time_limit, '--build', 'release64'], build_options=["release64"])

# State equation heuristic
exp.add_algorithm(
    '2-stateequ', REPO, 'default', ['--search', 'astar(operatorcounting([state_equation_constraints()],transform=adapt_costs(cost_type=ONE)),cost_type=ONE)'], driver_options=["--search-time-limit", time_limit, '--build', 'release64'], build_options=["release64"])

# Some timesteps with repetition and IP
# for step in timesteps_list:
#    exp.add_algorithm(
#        'timerepip{}gl'.format(step), REPO, 'default', ['--search', 'astar(time(wr=true, ts={}, #gl=true, ip=true))'.format(step)], driver_options=["--search-time-limit", time_limit, '--build', #'release64'], build_options=["release64"])
#    exp.add_algorithm(
#        'timerepip{}rf'.format(step), REPO, 'default', ['--search', 'astar(time(wr=true, ts={}, #rf=true, ip=true))'.format(step)], driver_options=["--search-time-limit", time_limit, '--build', #'release64'], build_options=["release64"])


# Make a report (AbsoluteReport is the standard report).
exp.add_report(
    AbsoluteReport())

exp.add_report(
    AbsoluteReport(attributes=["Unexplained Errors", "Info", "Summary", "cost", "coverage",
        "evaluations_until_last_jump", "expansions_until_last_jump", "generated_until_last_jump", 
        "initial_h_value", "memory", "reopened_until_last_jump", "search_time", "error", "my_timespent",
        "my_lpVars", "my_percInit", "my_lpCons", "my_peakMem", "my_tsrequired"]), outfile='myreport.html')

lmcut_costs = dict()

def init_lmcut_costs(run):
    if run["algorithm"] == "2-lmcut":
        if "plan_length" in run:
            lmcut_costs[run["domain"] + run["id"][2]] = run["plan_length"]
    return run


exp.add_fetcher(name='parse-again', parsers=['./ownLogParser.py'])

def getPercentage(run):
    if(run["domain"] + run["id"][2] in lmcut_costs) and "initial_h_value" in run:
        run["percentage"] = run["initial_h_value"] / lmcut_costs[run["domain"] + run["id"][2]]
        if run["initial_h_value"] / lmcut_costs[run["domain"] + run["id"][2]] > 1:
            print "fail in", run["domain"], " ", run["problem"]
        return run
    return False
        

exp.add_report(
    ScatterPlotReport(
        attributes=["percentage"], filter_algorithm=["1-ianot", "2-stateequ"], xscale='linear', yscale='linear', filter=[init_lmcut_costs,getPercentage], format="tex"),
    outfile='final_scatter_MATU_stateequ.tex')

exp.add_report(
    ScatterPlotReport(
        attributes=["percentage"], filter_algorithm=["1-ianot", "2-lmcut"], xscale='linear', yscale='linear', filter=[init_lmcut_costs,getPercentage], format="tex"),
    outfile='final_scatter_MATU_lmcut.tex')

exp.add_report(
    ScatterPlotReport(
        attributes=["percentage"], filter_algorithm=["1-iaip", "1-ianot"], xscale='linear', yscale='linear', filter=[init_lmcut_costs,getPercentage], format="tex"),
    outfile='final_scatter_ip_not.tex')

# Parse the commandline and show or run experiment steps.
exp.run_steps()
