#! /usr/bin/env python

"""Solve some tasks with A* and the LM-Cut heuristic."""

import os
import os.path
import platform

from lab.environments import BaselSlurmEnvironment

from downward.experiment import FastDownwardExperiment
from downward.reports.absolute import AbsoluteReport
from downward.reports.scatter import ScatterPlotReport


SUITE = ['floortile-opt14-strips', 'freecell',
'ged-opt14-strips', 'grid', 'gripper', 'hiking-opt14-strips',
'logistics98', 'miconic', 'movie', 'mprime', 'mystery', 'openstacks-opt08-strips']
ENV = BaselSlurmEnvironment()
REPO = "/infai/oldphi00/master_code"
BENCHMARKS_DIR = os.environ["DOWNWARD_BENCHMARKS"]

exp = FastDownwardExperiment(environment=ENV)
exp.add_suite(BENCHMARKS_DIR, SUITE)

# add attributes for sanity checks
exp.add_resource('sanity_parser', 'sanitychecks.py', dest='sanitychecks.py')
exp.add_command('sanitychecks', ['{sanity_parser}'])

exp.add_algorithm(
    'timeit', REPO, 'default', ['--search', 'astar(time(ia=true))'], driver_options=["--search-time-limit", "10m", '--build', 'release64'], build_options=["release64"])
exp.add_algorithm(
    'timerep0', REPO, 'default', ['--search', 'astar(time(wr=true, ts=0))'], driver_options=["--search-time-limit", "10m", '--build', 'release64'], build_options=["release64"])
exp.add_algorithm(
    'timerep5', REPO, 'default', ['--search', 'astar(time(wr=true, ts=5))'], driver_options=["--search-time-limit", "10m", '--build', 'release64'], build_options=["release64"])
exp.add_algorithm(
    'timerep10', REPO, 'default', ['--search', 'astar(time(wr=true, ts=10))'], driver_options=["--search-time-limit", "10m", '--build', 'release64'], build_options=["release64"])
exp.add_algorithm(
    'lmcut', REPO, 'default', ['--search', 'astar(lmcut())'], driver_options=["--search-time-limit", "10m", '--build', 'release64'], build_options=["release64"])

# Make a report (AbsoluteReport is the standard report).
exp.add_report(
    AbsoluteReport())

exp.add_report(
    AbsoluteReport(attributes=["Unexplained Errors", "Info", "Summary", "cost", "coverage",
        "evaluations_until_last_jump", "expansions_until_last_jump", "generated_until_last_jump", 
        "initial_h_value", "memory", "reopened_until_last_jump", "search_time"]), outfile='myreport.html')

# Compare the number of expansions in a scatter plot.
exp.add_report(
    ScatterPlotReport(
        attributes=["expansions"], filter_algorithm=["timeit", "timerep0"]),
    outfile='scatterplot.png')

def add_initialoroptimalattribute(run):
    if run["algorithm"] == "lmcut":
        run["initialoroptimal"] = run.get("cost")
    else:
	run["initialoroptimal"] = run.get("initial_h_value")
    return run

def removelargehvalues(run):
    return run.get("initialoroptimal", 100) < 100

# Compare the number of expansions in a scatter plot.
exp.add_report(
    ScatterPlotReport(
        attributes=["expansions"], filter_algorithm=["timeit", "timerep0"]),
    outfile='scatterplot.png')
exp.add_report(
    ScatterPlotReport(
        attributes=["initialoroptimal"], filter_algorithm=["lmcut", "timerep5"], filter=[add_initialoroptimalattribute, removelargehvalues]),
    outfile='initialhvalues.png')

# Sanity checks
def test(run):
    print(run["algorithm"])
    for i in run:
        print(i, run[i])

# lambda run: run.algo.name == "lmcut", lambda run: run.algo.name
exp.add_report(
    ScatterPlotReport(
        attributes=["expansions"], filter=lambda run: run["algorithm"] == "lmcut", get_category=lambda run: run["algorithm"]),
    outfile='scatterplot_sanity_domination.png')

# Parse the commandline and show or run experiment steps.
exp.run_steps()
