#include "yahsp_ff_heuristic.h"

#include "../plugins/plugin.h"

#include "../task_utils/task_properties.h"
#include "../utils/logging.h"

#include <cassert>

using namespace std;

namespace yahsp_ff_heuristic {
// construction and destruction
YahspFFHeuristic::YahspFFHeuristic(
    tasks::AxiomHandlingType axioms,
    const shared_ptr<AbstractTask> &transform, bool cache_estimates,
    const string &description, utils::Verbosity verbosity)
    : YahspAdditiveHeuristic(
          axioms, transform, cache_estimates, description,
          verbosity),
      relaxed_plan(task_proxy.get_operators().size(), false) {
    if (log.is_at_least_normal()) {
        log << "Initializing yahsp FF heuristic..." << endl;
    }
}

void YahspFFHeuristic::mark_preferred_operators_and_relaxed_plan(
    const State &state, PropID goal_id) {
    Proposition *goal = get_proposition(goal_id);
    if (!goal->marked) { // Only consider each subgoal once.
        goal->marked = true;
        OpID op_id = goal->reached_by;
        if (op_id != NO_OP) { // We have not yet chained back to a start node.
            UnaryOperator *unary_op = get_operator(op_id);
            //BA: the following is the helpful action, not the goal-preferred action:
            bool is_preferred = true;
            //BA: we keep applying the marking rules backwards to tha end
            for (PropID precond : get_preconditions(op_id)) {
                mark_preferred_operators_and_relaxed_plan(
                    state, precond);
                //BA: initially true proposition. no op bc it wasn't reached by an operator.
                if (get_proposition(precond)->reached_by != NO_OP) {
                    //BA: no longer considered as helpful because it can't be applied on the first initial state of this current search (not whole search, for a given state we do each search)
                    is_preferred = false;
                }
            }
            int operator_no = unary_op->operator_no;
            if (operator_no != -1) {
                // This is not an axiom.
                relaxed_plan[operator_no] = true; //BA:just shows which operators are part of the relaxed plan, but in reverse order (PROBABLY)
                if (is_preferred) {
                    OperatorProxy op = task_proxy.get_operators()[operator_no];
                    assert(task_properties::is_applicable(op, state));
                    //BA: let's not use these helpful actions, let's use something else!
                    set_preferred(op);
                    //BA: since checked whether applicable and helpful, add to helpful actions
                    helpful_actions.insert(operator_no);
                }
            }
        }
    }
}


int YahspFFHeuristic::compute_heuristic(const State &ancestor_state) {
    return 2; //we return some number when it re-evaluates using this function
    /*State state = convert_ancestor_state(ancestor_state);
    int h_add = compute_add_and_ff(state);
    if (h_add == DEAD_END)
        return h_add;

    // Collecting the relaxed plan also sets the preferred operators.
    for (PropID goal_id : goal_propositions)
        mark_preferred_operators_and_relaxed_plan(state, goal_id); //BA: here those that are marked as preferred are the helpful actions. we want to output these

    int h_ff = 0;
    for (size_t op_no = 0; op_no < relaxed_plan.size(); ++op_no) {
        if (relaxed_plan[op_no]) {
            relaxed_plan[op_no] = false; // Clean up for next computation.
            h_ff += task_proxy.get_operators()[op_no].get_cost();
        }
    }
    return h_ff; */
}

//YAHSP version:
//compute_heuristic(S,A) -> returns <relaxed plan, H, R>
//Fast Downward version, current: h(S) -> heuristic value
//states r a bunch of propositions. it is denoted which of them r true

std::tuple<int, std::vector<int>, std::unordered_set<int>, std::unordered_set<int>> YahspFFHeuristic::compute_yahsp_heuristic(const State &ancestor_state, bool gp) {
    helpful_actions.clear();
    std::unordered_set<int> rescue_actions; 
    std::vector<int> rp; //BA: initialize copy of relaxed plan to later flip
    OperatorsProxy operators = task_proxy.get_operators();
    State state = convert_ancestor_state(ancestor_state);
    int h_add = compute_add_and_ff(state, gp); //BA: relaxed exploration happens here so pass the gp bool here
    if (h_add == DEAD_END)
        return make_tuple(h_add, rp, helpful_actions, rescue_actions); //BA: returning empty relaxed plan and helpful actions

    // Collecting the relaxed plan also sets the preferred operators.
    for (PropID goal_id : goal_propositions)
        mark_preferred_operators_and_relaxed_plan(state, goal_id); //BA: here those that are marked as preferred are the helpful actions. we want to output these

    for (OperatorProxy op : operators) {
        int operator_no = op.get_id();
        if (helpful_actions.find(operator_no) == helpful_actions.end() && !op.is_axiom()) {
            rescue_actions.insert(operator_no);
        }
    }
    //BA: before this, the relaxed plan is built inside mark_preferred_operators_... so it can't be captured any earlier
    int h_ff = 0;
    for (size_t op_no = 0; op_no < relaxed_plan.size(); ++op_no) {
        if (relaxed_plan[op_no]) {
            relaxed_plan[op_no] = false; // Clean up for next computation.
            h_ff += task_proxy.get_operators()[op_no].get_cost();
            rp.insert(rp.begin(), op_no);
        }
    }

    return make_tuple(h_ff, rp, helpful_actions, rescue_actions);
}

//BA: plugin patch
static class YahspFFCategoryPlugin : public plugins::TypedCategoryPlugin<YahspFFHeuristic> {
public:
    YahspFFCategoryPlugin() : TypedCategoryPlugin("YahspFFHeuristic") {
    }
}
_category_plugin;

class YahspFFHeuristicFeature
    : public plugins::TypedFeature<YahspFFHeuristic, YahspFFHeuristic> {
public:
    YahspFFHeuristicFeature() : TypedFeature("yahspff") {
        document_title("yahsp FF heuristic");

        yahsp_relaxation_heuristic::add_relaxation_heuristic_options_to_feature(*this, "yahspff");

        document_language_support("action costs", "supported");
        document_language_support("conditional effects", "supported");
        document_language_support("axioms", "supported");

        document_property("admissible", "no");
        document_property("consistent", "no");
        document_property("safe", "yes");
        document_property("preferred operators", "yes");
    }

    virtual shared_ptr<YahspFFHeuristic>
    create_component(const plugins::Options &opts) const override {
        return plugins::make_shared_from_arg_tuples<YahspFFHeuristic>(
            yahsp_relaxation_heuristic::get_relaxation_heuristic_arguments_from_options(opts)
            );
    }
};

static plugins::FeaturePlugin<YahspFFHeuristicFeature> _plugin;
}
