#include "yahsp_additive_heuristic.h"

#include "../plugins/plugin.h"

#include "../task_utils/task_properties.h"
#include "../utils/logging.h"

#include <cassert>
#include <vector>

using namespace std;

namespace yahsp_additive_heuristic {
const int YahspAdditiveHeuristic::MAX_COST_VALUE;

YahspAdditiveHeuristic::YahspAdditiveHeuristic(
    tasks::AxiomHandlingType axioms,
    const shared_ptr<AbstractTask> &transform, bool cache_estimates,
    const string &description, utils::Verbosity verbosity)
    : YahspRelaxationHeuristic(
          axioms, transform, cache_estimates, description,
          verbosity),
      did_write_overflow_warning(false) {
    if (log.is_at_least_normal()) {
        log << "Initializing yahsp additive heuristic..." << endl;
    }
}

void YahspAdditiveHeuristic::write_overflow_warning() {
    if (!did_write_overflow_warning) {
        // TODO: Should have a planner-wide warning mechanism to handle
        // things like this.
        if (log.is_warning()) {
            log << "WARNING: overflow on h^add! Costs clamped to "
                << MAX_COST_VALUE << endl;
        }
        cerr << "WARNING: overflow on h^add! Costs clamped to "
             << MAX_COST_VALUE << endl;
        did_write_overflow_warning = true;
    }
}

// heuristic computation
void YahspAdditiveHeuristic::setup_exploration_queue() {
    queue.clear();

    for (Proposition &prop : propositions) {
        prop.cost = -1;
        prop.marked = false;
    }

    // Deal with operators and axioms without preconditions.
    for (UnaryOperator &op : unary_operators) {
        op.unsatisfied_preconditions = op.num_preconditions;
        op.cost = op.base_cost; // will be increased by precondition costs

        if (op.unsatisfied_preconditions == 0)
            enqueue_if_necessary(op.effect, op.base_cost, get_op_id(op));
    }
}

void YahspAdditiveHeuristic::setup_exploration_queue_state(const State &state) {
    for (FactProxy fact : state) {
        PropID init_prop = get_prop_id(fact);
        enqueue_if_necessary(init_prop, 0, NO_OP);
    }
}

void YahspAdditiveHeuristic::relaxed_exploration(bool gp) {
    int unsolved_goals = goal_propositions.size();
    while (!queue.empty()) {
        //BA: int is the cost of just one path to reach said proposition. not the final cost of the proposition, but while during the search. propid is the proposition id
        pair<int, PropID> top_pair = queue.pop();
        int distance = top_pair.first;
        PropID prop_id = top_pair.second;
        Proposition *prop = get_proposition(prop_id);
        //BA: these r the currently best cost found for the given proposition:
        int prop_cost = prop->cost;
        assert(prop_cost >= 0);
        assert(prop_cost <= distance);
        //BA: here we check whether the distance (cost!) of the current path is better to reach the proposition than the currently known best cost
        if (prop_cost < distance)
            continue;
        //BA: we only stop if the proposition we found is a goal and there are no goal propositions left to achieve
        if (prop->is_goal && --unsolved_goals == 0)
            return;
        //BA: if the distance is better, we look for all the unary operators for whom this current proposition is a precondition
        //BA: we look at it because there is a chance that for some operators have become executable
        for (OpID op_id : precondition_of_pool.get_slice(
                 prop->precondition_of, prop->num_precondition_occurences)) {
            //BA: if not in ga (goal preferred actions), continue
            if (gp && !gp_unaryops[op_id]) {
                continue;
            }
            UnaryOperator *unary_op = get_operator(op_id);
            increase_cost(unary_op->cost, prop_cost);
            //BA: for those operators for whom the proposition is a precondition, we decrease the count of unsatisfied preconditions it possesses.
            --unary_op->unsatisfied_preconditions;
            //BA: and now we check whether all preconditions are true. if it is, then we just apply it and get a new proposition and put it into the queue
            assert(unary_op->unsatisfied_preconditions >= 0);
            if (unary_op->unsatisfied_preconditions == 0)
            //BA: btw in the enqueue if necessary the reached_by is set
                enqueue_if_necessary(unary_op->effect,
                                     unary_op->cost, op_id);
        }
    }
}

void YahspAdditiveHeuristic::mark_preferred_operators(
    const State &state, PropID goal_id) {
    Proposition *goal = get_proposition(goal_id);
    if (!goal->marked) { // Only consider each subgoal once.
        goal->marked = true;
        OpID op_id = goal->reached_by;
        if (op_id != NO_OP) { // We have not yet chained back to a start node.
            UnaryOperator *unary_op = get_operator(op_id);
            bool is_preferred = true;
            for (PropID precond : get_preconditions(op_id)) {
                mark_preferred_operators(state, precond);
                if (get_proposition(precond)->reached_by != NO_OP) {
                    is_preferred = false;
                }
            }
            int operator_no = unary_op->operator_no;
            if (is_preferred && operator_no != -1) {
                // This is not an axiom.
                OperatorProxy op = task_proxy.get_operators()[operator_no];
                assert(task_properties::is_applicable(op, state));
                set_preferred(op);
            }
        }
    }
}

int YahspAdditiveHeuristic::compute_add_and_ff(const State &state, bool gp) {
    setup_exploration_queue();
    setup_exploration_queue_state(state); //BA: puts the initially true propositions into the queue
    relaxed_exploration(gp); //BA: after this is executed, we have basically input all the cost values into the relaxed graph

    int total_cost = 0;
    for (PropID goal_id : goal_propositions) {
        const Proposition *goal = get_proposition(goal_id);
        int goal_cost = goal->cost;
        if (goal_cost == -1)
            return DEAD_END;
        increase_cost(total_cost, goal_cost);
    }
    return total_cost; //BA: everything added, cuz hadd
}

int YahspAdditiveHeuristic::compute_heuristic(const State &ancestor_state) {
    State state = convert_ancestor_state(ancestor_state);
    int h = compute_add_and_ff(state);
    if (h != DEAD_END) {
        for (PropID goal_id : goal_propositions)
            mark_preferred_operators(state, goal_id);
    }
    return h;
}

void YahspAdditiveHeuristic::compute_heuristic_for_cegar(const State &state) {
    compute_heuristic(state);
}

class YahspAdditiveHeuristicFeature
    : public plugins::TypedFeature<Evaluator, YahspAdditiveHeuristic> {
public:
    YahspAdditiveHeuristicFeature() : TypedFeature("yahspadd") {
        document_title("Yahsp Additive heuristic");

        yahsp_relaxation_heuristic::add_relaxation_heuristic_options_to_feature(*this, "add");

        document_language_support("action costs", "supported");
        document_language_support("conditional effects", "supported");
        document_language_support("axioms", "supported");

        document_property("admissible", "no");
        document_property("consistent", "no");
        document_property("safe", "yes");
        document_property("preferred operators", "yes");
    }

    virtual shared_ptr<YahspAdditiveHeuristic>
    create_component(const plugins::Options &opts) const override {
        return plugins::make_shared_from_arg_tuples<YahspAdditiveHeuristic>(
            yahsp_relaxation_heuristic::get_relaxation_heuristic_arguments_from_options(opts)
            );
    }
};

static plugins::FeaturePlugin<YahspAdditiveHeuristicFeature> _plugin;
}
