#include "zero_one_pdbs.h"

#include "compression.h"
#include "pattern_database.h"
#include "pattern_database_factory.h"

#include "../task_proxy.h"

#include "../utils/logging.h"

#include <iostream>
#include <limits>
#include <memory>
#include <vector>

using namespace std;

namespace pdbs {
ZeroOnePDBs::ZeroOnePDBs(
    const TaskProxy &task_proxy, const PatternCollection &patterns, bool compress, const Options &opts) {
    vector<int> remaining_operator_costs;
    OperatorsProxy operators = task_proxy.get_operators();
    remaining_operator_costs.reserve(operators.size());
    for (OperatorProxy op : operators)
        remaining_operator_costs.push_back(op.get_cost());

    pattern_databases.reserve(patterns.size());

    // The ZOPDB call from the genetic generation algorithm does not provide an options object -> empty, check
    int state_budget = 0;
    if (opts.contains("max_size_compr_col")) {
        state_budget = opts.get<int>("max_size_compr_col");
    }
    int remaining_budget = state_budget;

    for (const Pattern &pattern : patterns) {
        shared_ptr<PatternDatabase> pdb = create_default_pdb(
            task_proxy, pattern, false, remaining_operator_costs);

         if (compress && remaining_budget > 0) {
            int used_budget = compute_best_pdb(
                    pattern,
                    pdb,
                    task_proxy,
                    opts,
                    remaining_budget,
                    remaining_operator_costs
            );
            remaining_budget -= used_budget;
        }

        /* Set cost of relevant operators to 0 for further iterations
           (action cost partitioning). */
        for (OperatorProxy op : operators) {
            if (pdb->is_operator_relevant(op))
                remaining_operator_costs[op.get_id()] = 0;
        }
        pattern_databases.push_back(pdb);
    }
    print_avg_finite_mean_for_PDBCollection(pattern_databases);
}


int ZeroOnePDBs::get_value(const State &state) const {
    /*
      Because we use cost partitioning, we can simply add up all
      heuristic values of all patterns in the pattern collection.
    */
    int h_val = 0;
    for (const shared_ptr<PatternDatabase> &pdb : pattern_databases) {
        int pdb_value = pdb->get_value(state);
        if (pdb_value == numeric_limits<int>::max())
            return numeric_limits<int>::max();
        h_val += pdb_value;
    }
    return h_val;
}

double ZeroOnePDBs::compute_approx_mean_finite_h() const {
    double approx_mean_finite_h = 0;
    for (const shared_ptr<PatternDatabase> &pdb : pattern_databases) {
        approx_mean_finite_h += pdb->compute_mean_finite_h();
    }
    return approx_mean_finite_h;
}

void ZeroOnePDBs::dump() const {
    for (const shared_ptr<PatternDatabase> &pdb : pattern_databases) {
        utils::g_log << pdb->get_pattern() << endl;
    }
}
}
