#include "saturated_pdbs_heuristic.h"

#include "compression.h"
#include "pattern_generator.h"

#include "../option_parser.h"
#include "../plugin.h"

using namespace std;

namespace pdbs {
    SaturatedPDBs get_saturated_pdbs_from_options(
    const shared_ptr<AbstractTask> &task, const Options &opts) {
    shared_ptr<PatternCollectionGenerator> pattern_generator =
        opts.get<shared_ptr<PatternCollectionGenerator>>("patterns");
    PatternCollectionInformation pattern_collection_info =
        pattern_generator->generate(task);
    shared_ptr<PatternCollection> patterns =
        pattern_collection_info.get_patterns();
    TaskProxy task_proxy(*task);
    return SaturatedPDBs(task_proxy, *patterns, opts);
}

    SaturatedPDBsHeuristic::SaturatedPDBsHeuristic(
    const options::Options &opts)
    : Heuristic(opts),
      saturated_pdbs(get_saturated_pdbs_from_options(task, opts)) {
}

int SaturatedPDBsHeuristic::compute_heuristic(const GlobalState &global_state) {
    State state = convert_global_state(global_state);
    return compute_heuristic(state);
}

int SaturatedPDBsHeuristic::compute_heuristic(const State &state) const {
    int h = saturated_pdbs.get_value(state);
    if (h == numeric_limits<int>::max())
        return DEAD_END;
    return h;
}

static shared_ptr<Heuristic> _parse(OptionParser &parser) {
    parser.document_synopsis(
        "Saturated PDB",
        "TEXT");
    parser.document_language_support("action costs", "supported");
    parser.document_language_support("conditional effects", "not supported");
    parser.document_language_support("axioms", "not supported");
    parser.document_property("admissible", "yes");
    parser.document_property("consistent", "yes");
    parser.document_property("safe", "yes");
    parser.document_property("preferred operators", "no");

    parser.add_option<shared_ptr<PatternCollectionGenerator>>(
        "patterns",
        "pattern generation method",
        "systematic(1)");

    add_compression_parser_parameters(parser);

    Heuristic::add_options_to_parser(parser);

    Options opts = parser.parse();
    if (parser.dry_run())
        return nullptr;

    return make_shared<SaturatedPDBsHeuristic>(opts);
}

static Plugin<Evaluator> _plugin("spdbs", _parse, "heuristics_pdb");
}
