#ifndef PDBS_PATTERN_DATABASE_DOUBLE_H
#define PDBS_PATTERN_DATABASE_DOUBLE_H

#include "types.h"

#include "../task_proxy.h"

#include <utility>
#include <vector>

namespace pdbs {
// Implements a single pattern database
class PatternDatabaseDouble {
    Pattern pattern;

    // size of the PDB
    std::size_t num_states;

    // Operator transition system, used for SCP
    const std::vector<std::vector<std::pair<int, int>>> operator_transitions;

    /*
      final h-values for abstract-states.
      dead-ends are represented by numeric_limits<int>::max()
    */
    std::vector<double> distances;

    // multipliers for each variable for perfect hash function
    std::vector<std::size_t> hash_multipliers;

    /*
      The given concrete state is used to calculate the index of the
      according abstract state. This is only used for table lookup
      (distances) during search.
    */
    std::size_t hash_index(const State &state) const;
public:
    /*
      Important: It is assumed that the pattern (passed via Options) is
      sorted, contains no duplicates and is small enough so that the
      number of abstract states is below numeric_limits<int>::max()
      Parameters:
       dump:           If set to true, prints the construction time.
       operator_costs: Can specify individual operator costs for each
       operator. This is useful for action cost partitioning. If left
       empty, default operator costs are used.
    */
    PatternDatabaseDouble(
        const Pattern &pattern,
        std::size_t num_states,
        std::vector<double> &&distances,
        std::vector<std::size_t> &&hash_multipliers,
        const std::vector<std::vector<std::pair<int, int>>> &operator_transitions);
    ~PatternDatabaseDouble() = default;

    double get_value(const State &state) const;

    // Returns the pattern (i.e. all variables used) of the PDB
    const Pattern &get_pattern() const {
        return pattern;
    }

    // Returns the size (number of abstract states) of the PDB
    int get_size() const {
        return num_states;
    }

    const std::vector<double> &get_distances() const {
        return distances;
    };

    const std::vector<std::size_t> &get_hashmultipliers() const {
        return hash_multipliers;
    }

    const std::vector<std::vector<std::pair<int, int>>> &get_operator_transitions() const {
        return operator_transitions;
    }

    /*
      Returns the average h-value over all states, where dead-ends are
      ignored (they neither increase the sum of all h-values nor the
      number of entries for the mean value calculation). If all states
      are dead-ends, infinity is returned.
      Note: This is only calculated when called; avoid repeated calls to
      this method!
    */
    double compute_mean_finite_h() const;

    // Returns true iff op has an effect on a variable in the pattern.
    bool is_operator_relevant(const OperatorProxy &op) const;
};
}

#endif
