#ifndef SEARCH_ENGINES_LANDMARK_META_SEARCH_H
#define SEARCH_ENGINES_LANDMARK_META_SEARCH_H

#include "../search_engine.h"
#include "../meta_node.h"
#include "../tasks/modified_initial_task.h"
#include "../open_lists/open_list.h"
#include "../utils/hash.h"
#include "../search_statistics.h"

namespace options {
class Options;
}

namespace landmarks {
class LandmarkGraph;
class LandmarkStatusManager;
class LandmarkFactory;
}

namespace landmark_meta_search {
struct hash_meta_node {
    size_t operator()(const shared_ptr<MetaNode::MetaNode> &node) const
    {
        const MetaNode::MetaNode &n = *node;
        size_t hash = utils::hash_sequence<std::vector<bool>>(n.achieved_lms, n.achieved_lms.size());
        utils::hash_combine(hash, n.state_id);
        utils::hash_combine(hash, n.goal_lm);
        return hash;
    }
};

struct eq_meta_node {
    bool operator()(const shared_ptr<MetaNode::MetaNode> &a, const shared_ptr<MetaNode::MetaNode> &b) const
    {
		if (a->state_id != b->state_id) {
			return false;
		}
		if (a->goal_lm != b->goal_lm) {
			return false;
		}
		if (a->achieved_lms != b->achieved_lms) {
			return false;
		}
		return true;
    }
};

struct metaStats {
	int sub_searches = 0;
	int solvable_sub_searches = 0;
	int nodes_already_closed = 0;

};


class MetaSearch : public SearchEngine {
    std::shared_ptr<landmarks::LandmarkGraph> lgraph;
    std::unique_ptr<landmarks::LandmarkStatusManager> lm_status_manager;
	std::shared_ptr<landmarks::LandmarkFactory> lm_graph_factory;
    std::unique_ptr<MetaNodeOpenList> open_list;
	std::unordered_set<shared_ptr<MetaNode::MetaNode>, hash_meta_node, eq_meta_node> closed;
	bool lmcount = true; // dirty: normal heur vs lmcount
	SearchStatistics search_stats;
    Heuristic* heuristic;
	Heuristic* subheuristic;
	std::string heuristic_string;
	bool succCut;
	metaStats meta_stats;
protected:
    virtual void initialize() override;
    virtual SearchStatus step() override;
	void create_initial_meta_nodes();
	void openlist_insert_accordingly(EvaluationContext &eval_context, int hval, shared_ptr<MetaNode::MetaNode> node);
	shared_ptr<extra_tasks::ModifiedInitialTask> transform_task_for_metanode(shared_ptr<AbstractTask> parent, MetaNode::MetaNode m);
	/* The generate_metanodes methods add metanodes to the open_list */
	void generate_metanodes_next_lm(const MetaNode::MetaNode &m, const vector<const GlobalOperator *>& m_solution, const StateID new_state_id);
	void generate_metanodes_delete_lm(const MetaNode::MetaNode &m);
	void generate_metanodes_cut_parents(MetaNode::MetaNode m, const vector<const GlobalOperator *>& m_solution, const StateID new_state_id);
	void generate_metanodes_restart_cut_parents(const MetaNode::MetaNode &m, const StateID new_state_id);
	void achieve_lm_node_ancestors(const landmarks::LandmarkNode& node, std::vector<bool> &achieved);
		std::vector<int> get_applicable_operators(const shared_ptr<AbstractTask> parent, MetaNode::MetaNode m);
	shared_ptr<std::vector<landmarks::LandmarkNode *>> get_root_landmarks(std::shared_ptr<landmarks::LandmarkGraph> lg, const std::vector<bool> &achieved);
	bool achieves(const OperatorProxy &o, const landmarks::LandmarkNode *lmp) const;
    bool sub_task_solvable(const MetaNode::MetaNode& m);
	int count_non_achieved_landmarks(vector<bool> achieved);
	void add_subsearch_statistics_to_search_statistics(const SearchStatistics& to_add);
	void dump_meta_statistics();
	std::pair<bool,vector<const GlobalOperator *>> sub_task_get_solution(const MetaNode::MetaNode& m);
    void generate_metanodes(const MetaNode::MetaNode &m, const vector<const GlobalOperator *>& m_solution, const StateID new_state_id);
public:
	explicit MetaSearch(const options::Options &opts);
	virtual ~MetaSearch() = default;
};
}

#endif
