#include "landmark_meta_search.h"

#include "../landmarks/exploration.h"
#include "../landmarks/landmark_factory.h"
#include "../landmarks/landmark_graph.h"
#include "../landmarks/landmark_status_manager.h"
#include "sub_search.h"
#include "search_common.h"

#include "../tasks/modified_operators_task.h"
#include "../tasks/modified_goals_task.h"
#include "../tasks/modified_initial_task.h"

#include "../open_lists/standard_scalar_open_list.h"

#include "../option_parser.h"
#include "../plugin.h"

using namespace std;

namespace landmark_meta_search {
MetaSearch::MetaSearch(const Options &opts)
	:SearchEngine(opts) {
    verify_no_axioms_no_conditional_effects();
    landmarks::LandmarkFactory *lm_graph_factory = opts.get<landmarks::LandmarkFactory *>("lm_factory");
    heuristic = opts.get<Heuristic *>("heur");
	subheuristic = opts.get<Heuristic *>("subheur");
	//heuristic_string = opts.get<std::string>("heur");
	succCut = opts.get<bool>("succ_cut", true);
    Options exploration_opts;
    exploration_opts.set<shared_ptr<AbstractTask>>("transform", g_root_task());
    exploration_opts.set<bool>("cache_estimates", false);
    landmarks::Exploration exploration(exploration_opts);
    lgraph = lm_graph_factory->compute_lm_graph(g_root_task(), exploration);
    lm_status_manager = utils::make_unique_ptr<landmarks::LandmarkStatusManager>(*lgraph);
    Options open_list_options;
	open_list_options.set<ScalarEvaluator*>("eval", heuristic);
    open_list_options.set<bool>("pref_only", false);
    open_list = StandardScalarOpenListFactory(open_list_options).create_meta_node_open_list();
}

void MetaSearch::initialize() {
	cout << "HELLO, META-WORLD" << endl;
    create_initial_meta_nodes();
	search_stats = SearchStatistics();
}

SearchStatus MetaSearch::step() {
	cout << "Calling MetaSearch::step()" << endl;
	if (!open_list->empty()) {
		shared_ptr<MetaNode::MetaNode> m = open_list->remove_min(); // get and remove metanode with smallest heur_value
		//m->dump_metanode(state_registry); //prints the opened metanode TODO: fix give_state_registry
		/*for (auto i : m->plan) {
			cout << i->get_name() << endl;
		}
		cout << "End Metanode Plan" << endl;*/

		if (closed.find(m) == closed.end()) { // if m is not in closed
			closed.insert(m);
			pair<bool,vector<const GlobalOperator *>> sol = sub_task_get_solution(*m);
			meta_stats.sub_searches++;
            if (sol.first) { // PI_m has a solution
				meta_stats.solvable_sub_searches++;
                const vector<const GlobalOperator *>& m_solution = sol.second;
            
				GlobalState next_state = state_registry.lookup_state(m->state_id);
                for (auto i : m_solution) { // apply solution_plan to s to get s'
                    next_state = state_registry.get_successor_state(next_state, *i);
                }
                if (test_goal(next_state)) { // if s' is a solution to the outer task
					Plan &solution_plan = m->plan;

					solution_plan.insert(solution_plan.end(), m_solution.begin(), m_solution.end());
					set_plan(solution_plan);
					dump_meta_statistics(); // custom meta-search stats
					cout << "(Meta)search-statistics:" << endl;
					search_stats.print_detailed_statistics();
					return SOLVED;
                }
                generate_metanodes(*m, m_solution, next_state.get_id());
            }
		}
		else {
			meta_stats.nodes_already_closed++;
		}
		return IN_PROGRESS;
	}
	return FAILED;
}

void MetaSearch::generate_metanodes(const MetaNode::MetaNode &m, const vector<const GlobalOperator *>& m_solution, const StateID new_state_id) {
	if (!succCut) { // TODO: fix succ_cut opt
        generate_metanodes_next_lm(m, m_solution, new_state_id);
		generate_metanodes_delete_lm(m); // TODO: check state parameter (paper!)
    }
	if (succCut) {
		const GlobalState& new_initial = state_registry.get_initial_state();
		generate_metanodes_next_lm(m, m_solution, new_state_id);
		generate_metanodes_cut_parents(m, m_solution, new_state_id);
		generate_metanodes_restart_cut_parents(m, new_initial.get_id());
	}
}

void MetaSearch::create_initial_meta_nodes() {
	const GlobalState& initial_state = state_registry.get_initial_state();
	lm_status_manager->set_landmarks_for_initial_state(initial_state);
	vector<bool>& achieved_lms = lm_status_manager->get_reached_landmarks(initial_state); // TODO: delete these comments
	shared_ptr<std::vector<landmarks::LandmarkNode *>> root_landmarks = get_root_landmarks(lgraph, achieved_lms);
	std::vector<const GlobalOperator *> plan;
	StateID initial_state_id = initial_state.get_id();
	EvaluationContext context(initial_state);
	for (auto lm_node : *root_landmarks) {
		shared_ptr<MetaNode::MetaNode> node = make_shared<MetaNode::MetaNode>(initial_state_id, achieved_lms, lm_node, plan);
		openlist_insert_accordingly(context, count_non_achieved_landmarks(achieved_lms), node);

	}
}


void MetaSearch::openlist_insert_accordingly(EvaluationContext &eval_context, int hval, shared_ptr<MetaNode::MetaNode> node) {
	if (lmcount) { // dirty way of choosing normal heur vs lmcount
		open_list->insert_hval_directly(hval, node);
		return;
	}
	if (!lmcount) {
		open_list->insert(eval_context, node);
		return;
	}
}

// [Not tested]
shared_ptr<extra_tasks::ModifiedInitialTask> MetaSearch::transform_task_for_metanode(const shared_ptr<AbstractTask> parent, MetaNode::MetaNode m) {
	std::vector<FactPair> new_goal = m.goal_lm->facts;
	std::vector<int> appl_ops = get_applicable_operators(parent, m);
	std::vector<int> new_initial = state_registry.lookup_state(m.state_id).get_values();
	shared_ptr<extra_tasks::ModifiedOperatorsTask> ops_transformed = make_shared<extra_tasks::ModifiedOperatorsTask>(parent, std::move(appl_ops));
	shared_ptr<extra_tasks::ModifiedGoalsTask> goals_ops_transformed = make_shared<extra_tasks::ModifiedGoalsTask>(ops_transformed, move(new_goal));
	shared_ptr<extra_tasks::ModifiedInitialTask> init_goals_ops_transformed = make_shared<extra_tasks::ModifiedInitialTask>(goals_ops_transformed, move(new_initial));
		return init_goals_ops_transformed;
}

// Definition 10
void MetaSearch::generate_metanodes_next_lm(const MetaNode::MetaNode &m, const vector<const GlobalOperator *>& m_solution, const StateID new_state_id) {
	/* If PI_m has a solution, then:
	For all root landmarks l', create a metanode with following changes:
	achieve m.goal_lm, make l' the goal_lm,
	change initial, heur, plan based on plan of Solution(PI_m) */
	EvaluationContext context(state_registry.lookup_state(new_state_id));
	vector<bool> new_achieved_lms = m.achieved_lms;
	new_achieved_lms[m.goal_lm->get_id()] = true;
	vector<const GlobalOperator *> extended_plan = m.plan;
	extended_plan.insert(extended_plan.end(), m_solution.begin(), m_solution.end()); // append solution_plan to m.plan
	shared_ptr<std::vector<landmarks::LandmarkNode *>> root_landmarks = get_root_landmarks(lgraph, new_achieved_lms);
	for (landmarks::LandmarkNode* new_goal_lm : *root_landmarks) {
		shared_ptr<MetaNode::MetaNode> new_metanode = make_shared<MetaNode::MetaNode>(new_state_id, new_achieved_lms, new_goal_lm, extended_plan);
		openlist_insert_accordingly(context, count_non_achieved_landmarks(new_achieved_lms), new_metanode);
	}
	return;
}

// Definition 13
// h' = h in this case
void MetaSearch::generate_metanodes_delete_lm(const MetaNode::MetaNode &m) {
	/* For all root landmarks l', create a metanode with following changes:
	   achieve l.goal_lm, make l' the goal */
	vector<bool> new_achieved_lms = m.achieved_lms;
	new_achieved_lms[m.goal_lm->get_id()] = true;
	shared_ptr<std::vector<landmarks::LandmarkNode *>> root_landmarks = get_root_landmarks(lgraph, new_achieved_lms);
	EvaluationContext context(state_registry.lookup_state(m.state_id));
	for (landmarks::LandmarkNode* new_goal_lm : *root_landmarks) {
		shared_ptr<MetaNode::MetaNode> new_metanode = make_shared<MetaNode::MetaNode>(m.state_id, new_achieved_lms, new_goal_lm, m.plan);
		openlist_insert_accordingly(context, count_non_achieved_landmarks(new_achieved_lms), new_metanode);
	}
	return;
}

// Definition 11
void MetaSearch::generate_metanodes_cut_parents(MetaNode::MetaNode m, const vector<const GlobalOperator *>& m_solution, const StateID new_state_id) {
	/* For all children l' of m.goal_lm, create a metanode with following changes:
	achieve l's ancestors, make l' the goal_lm,
	change initial, heur, plan based on plan of Solution(PI_m) */
	for (const auto& ch : m.goal_lm->children) {
		EvaluationContext context(state_registry.lookup_state(new_state_id));
		landmarks::LandmarkNode& node = *(ch.first);
	    vector<bool> new_achieved_lms = m.achieved_lms;
		achieve_lm_node_ancestors(node, new_achieved_lms);
		std::vector<const GlobalOperator *> extended_plan = m.plan;
		extended_plan.insert(extended_plan.end(), m_solution.begin(), m_solution.end());
		shared_ptr<MetaNode::MetaNode> new_metanode = make_shared<MetaNode::MetaNode>(new_state_id, new_achieved_lms, &node, extended_plan);
		openlist_insert_accordingly(context, count_non_achieved_landmarks(new_achieved_lms), new_metanode);
	}
	return;
}

// TODO: test this - including achieve_lm_node_ancestors()
// Definition 12
void MetaSearch::generate_metanodes_restart_cut_parents(const MetaNode::MetaNode &m, const StateID initial_state_id) {
	/* For all children l' of m.goal_lm, create a metanode with following changes:
	Go to initial_state (including heuristic), achieve l's ancestors, make l' the goal_lm */
	EvaluationContext context(state_registry.lookup_state(initial_state_id));
	for (const auto& ch : m.goal_lm->children) {
		landmarks::LandmarkNode& node = *(ch.first);
	    vector<bool> new_achieved_lms = m.achieved_lms;
		achieve_lm_node_ancestors(node, new_achieved_lms);
		std::vector<const GlobalOperator *> empty_plan;
		shared_ptr<MetaNode::MetaNode> new_metanode = make_shared<MetaNode::MetaNode>(initial_state_id, new_achieved_lms, &node, empty_plan);
		openlist_insert_accordingly(context, count_non_achieved_landmarks(new_achieved_lms), new_metanode);
	}
	return;
}

/* Recursively adds all ancestor_nodes of 'node' to the vector 'achieved' */
void MetaSearch::achieve_lm_node_ancestors(const landmarks::LandmarkNode& node, std::vector<bool> &achieved) {
	for (const auto& parent : node.parents) { //for every parent
		landmarks::LandmarkNode& parent_node = *(parent.first);
		achieved[parent_node.get_id()] = true; //achieve it
	    achieve_lm_node_ancestors(parent_node, achieved); //start recursion
	}
	return;
}

std::vector<int> MetaSearch::get_applicable_operators(const shared_ptr<AbstractTask> parent, MetaNode::MetaNode m) {
	TaskProxy task = TaskProxy(*parent);
	OperatorsProxy operators = task.get_operators();
	const landmarks::LandmarkNode* l = m.goal_lm;
	std::vector<int> allowed;

    shared_ptr<std::vector<landmarks::LandmarkNode *>> root_landmarks = get_root_landmarks(lgraph, m.achieved_lms);
	for (OperatorProxy op : operators) {

		bool achieves_goal_l = achieves(op,l);
		if (achieves_goal_l) { //if the goal landmark is achieved => allowed
			allowed.push_back(op.get_id());
			continue;
		}
		bool achieves_a_non_goal_l = false;

		for (landmarks::LandmarkNode* node : *root_landmarks) {
			if (node != l) {
				if (achieves(op, node)) { //if a non-goal landmark is achieved => not allowed
					achieves_a_non_goal_l = true;
                    break;
				}
			}
		}

		if (!achieves_a_non_goal_l) {
			allowed.push_back(op.get_id());
		}
	}

	return allowed;
}

bool MetaSearch::achieves(const OperatorProxy &o, const landmarks::LandmarkNode* lmp) const {
	/* Test whether the landmark is achieved by the operator.
	A disjunctive landmarks is achieved if one of its disjuncts is achieved. */
	assert(lmp);
	for (EffectProxy effect: o.get_effects()) {
		for (const FactPair &lm_fact : lmp->facts) {
			FactProxy effect_fact = effect.get_fact();
			if (effect_fact.get_pair() == lm_fact) {
					return true;
			}
		}
	}
	return false;
}


	// [Not tested]
shared_ptr<std::vector<landmarks::LandmarkNode *>> MetaSearch::get_root_landmarks(std::shared_ptr<landmarks::LandmarkGraph> lg,
    const vector<bool> &achieved) {
	shared_ptr<std::vector<landmarks::LandmarkNode *>> root_landmarks = make_shared<std::vector<landmarks::LandmarkNode *>>();
	for (landmarks::LandmarkNode* node : lg->get_nodes()){ // go over landmark nodes
		bool is_root_lm = !achieved[node->get_id()];
		for (const auto &p : node->parents){ // go over parents
			landmarks::LandmarkNode& parent = *(p.first);
			if (!achieved[parent.get_id()]) { // root landmark <=> all parents reached
				is_root_lm = false;
			}
		}
		if (is_root_lm){
			root_landmarks->push_back(node);
		}
	}
	return root_landmarks;
}

/* counts the number of non-achieved landmarks in an achieved-vector */
int MetaSearch::count_non_achieved_landmarks(vector<bool> achieved) {
	int result = 0;
	for (auto i : achieved) {
		if (!i) {
			result++;
		}
	}
	return result;
}

void MetaSearch::add_subsearch_statistics_to_search_statistics(const SearchStatistics& to_add) {
	int to_add_evaluated = to_add.get_evaluated_states();
	int to_add_generated = to_add.get_generated();
	int to_add_reopened = to_add.get_reopened();
	int to_add_expanded = to_add.get_expanded();
	search_stats.inc_evaluated_states(to_add_evaluated);
	search_stats.inc_generated(to_add_generated);
	search_stats.inc_reopened(to_add_reopened);
	search_stats.inc_expanded(to_add_expanded);
}

void MetaSearch::dump_meta_statistics() {
	cout << "Meta-stats: ";
	cout << "Sub-searches done: " << meta_stats.sub_searches << "." << endl;
	cout << "Sub-searches solved: " << meta_stats.solvable_sub_searches << "." << endl;
	cout << "Closed nodes met: " << meta_stats.nodes_already_closed << "." << endl;
}

pair<bool,vector<const GlobalOperator *>> MetaSearch::sub_task_get_solution(const MetaNode::MetaNode& m) {
    OptionParser parser("dummy()", false);
    parser.add_option<bool>("reopen_closed",
                            "reopen closed nodes", "false");
    SearchEngine::add_options_to_parser(parser);
    Options search_opts = parser.parse();
    search_opts.set("open", search_common::create_standard_scalar_open_list_factory(heuristic, false));
	std::vector<Heuristic *> preferred;
	std::vector<ScalarEvaluator*> evals;
	preferred.push_back(subheuristic);
	evals.push_back(subheuristic);
	search_opts.set<std::vector<Heuristic*>>("preferred", preferred);
	search_opts.set<std::vector<ScalarEvaluator*>>("evals", evals);
	search_opts.set("boost", 0);
	search_opts.set("open", search_common::create_greedy_open_list_factory(search_opts));
	sub_search::SubSearch subsearch(search_opts, transform_task_for_metanode(g_root_task(),m));
    subsearch.search();
	const SearchStatistics& sub_stats = subsearch.get_statistics();
	add_subsearch_statistics_to_search_statistics(sub_stats);
	if (subsearch.found_solution()) {
		return make_pair(true, subsearch.get_plan());
	}
	else {
		std::vector<const GlobalOperator *> empty_plan;
		return make_pair(false, empty_plan);
	}
}

// do we need this anymore?
bool MetaSearch::sub_task_solvable(const MetaNode::MetaNode& m) {
	OptionParser parser("dummy()", false);
	parser.add_option<bool>("reopen_closed",
							"reopen closed nodes", "false");
	parser.add_list_option<Heuristic *>(
			"preferred",
			"use preferred operators of these heuristics", "[]");
	SearchEngine::add_options_to_parser(parser);
	Options search_opts = parser.parse();
	search_opts.set("open", search_common::create_standard_scalar_open_list_factory(heuristic, false));
	sub_search::SubSearch subsearch(search_opts, transform_task_for_metanode(g_root_task(),m));
	subsearch.search();
	return subsearch.found_solution();
}

static SearchEngine *_parse(OptionParser &parser) {
        SearchEngine::add_options_to_parser(parser);
    parser.add_option<landmarks::LandmarkFactory *>(
        "lm_factory",
        "the set of landmarks to use for the meta search. "
        "The set of landmarks can be specified here, "
        "or predefined (see LandmarkFactory).");
    parser.add_option<Heuristic *>("heur", "heuristic");
	parser.add_option<Heuristic *>("subheur", "heuristic for subsearches");
	//parser.add_option<bool>("succ_cut", "succCut");
	Options opts = parser.parse();

	MetaSearch *engine = nullptr;
	if (!parser.dry_run()) {
		opts.set<bool>("mpd", false);
		engine = new MetaSearch(opts);
	}
	return engine;
}

static Plugin<SearchEngine> _plugin("landmark_meta_search", _parse);
}
