// -*- mode: C++; c-file-style: "stroustrup"; c-basic-offset: 4; -*-
////////////////////////////////////////////////////////////////////
//
// $Id: search_engine.h 930 2017-01-12 10:00:47Z Kevin Grimm $
//
////////////////////////////////////////////////////////////////////

#ifndef SEARCH_SEARCH_ENGINE_H
#define SEARCH_SEARCH_ENGINE_H

#include <inttypes.h>
#include <iosfwd>
#include <vector>
#include <queue>

#include "system/constraint.h"
#include "tools/options.h"

#include "ff_heuristic/ffback.h"
#include "closed_list.h"
#include "open_list.h"
#include "transition.h"

class Task;
class State;
class OpenList;
class ClosedList;
class Heuristic;
class Normalizer;
class Options;
class Trace;
class Useless;
class SuccessorGenerator;

class SearchEngine {
public:
    enum status_t {UNSAT = 0, SAT};
protected:
    const Task* task;
    const Options* opts;
public:
    const State* goal;

    OpenList* open;
    ClosedList* closed;
    Heuristic* heur;
    Normalizer* norm;

    status_t stat;
    uint32_t exploredStates;
    uint32_t generatedStates;
protected:
    int32_t indep_size;
    std::vector<const Constraint*> removed_target_constraints;

    virtual std::ostream& printProgress(std::ostream&) const;
    virtual std::ostream& printTrace(std::ostream&) const;
    virtual void generateTrace(Trace& trace, const State* goal) const;
    virtual void insert(State* succ);
    virtual void prepareExploration(const SuccessorGenerator& succgen, State* initial);
public:
    SearchEngine(const Task* task, const Options* opts);

    virtual void init(OpenList*, ClosedList*, Heuristic*, Normalizer*);
    virtual void explore(State* initial);
    virtual std::ostream& report(std::ostream&) const;
};

////////////////////////////////////////////////////////////////////

class UTSearchEngine : public SearchEngine {
protected:
    Useless* useless;
public:
    UTSearchEngine(const Task* task, const Options* opts) :
        SearchEngine(task, opts) {}
    virtual void init(OpenList*, ClosedList*, Heuristic*, Normalizer*);
    virtual void insert(State* succ);
};

////////////////////////////////////////////////////////////////////

class UTIPSearchEngine : public UTSearchEngine {
private:
    int32_t interference_level;
public:
    UTIPSearchEngine(const Task* task, const Options* opts);
    virtual void insert(State* succ);
};

////////////////////////////////////////////////////////////////////

class IPSearchEngine : public SearchEngine {
private:
    int32_t interference_level;
public:
    IPSearchEngine(const Task* task, const Options* opts);
    virtual void insert(State* succ);
};

////////////////////////////////////////////////////////////////////

class GraphvizEngine : public SearchEngine {
private:
    uint32_t diff_states;
    int32_t id;
    void label(State* s, FILE* file);
public:
    GraphvizEngine(const Task* task, const Options* opts);
    virtual void explore(State* initial);
};

////////////////////////////////////////////////////////////////////

class UASearchEngine : public SearchEngine {
private:
    std::vector<bool> allowedEdges;
    std::vector<Transition*> permittedTransitions; //needed to track if a new transition has been added (it is possible that a new Edge has been added but no new transition is possible since a transition is induced by up to 2 edges)
    std::vector<State*> permittedStates; //contains States which are created by a permitted transition
    PriorityQueue *notComputed; //contains States of the closed list that have not been used in a refinement step
    uint32_t plateauGuard;
    uint32_t hCurrent;
    ff::UA_FFBackHeuristic* ua_heur; //Heuristic for the refinements

    bool insertSuccs(std::vector<State*> *succs);
    bool insertAllowedStates();
    bool isNewTransAllowed();
    bool isTransAllowed(const Transition* t);
    void refineRelaxed(uint32_t hMin, uint32_t hMax);
    void refineApplicable(uint32_t hMin, uint32_t hMax);
    void castHeuristic();

public:
    UASearchEngine(const Task* task, const Options* opts);
    virtual void explore(State* initial);   
};


////////////////////////////////////////////////////////////////////


#endif /* SEARCH_SEARCH_ENGINE_H */
